/* acodec.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <stdio.h>
#include <string.h>

#include "acodec.h"
#ifdef HAVE_LIBMP3LAME
#include "ac_mp3lame.h"
#endif /* HAVE_LIBMP3LAME */
#include "ac_pcm.h"
#ifdef HAVE_OGGVORBIS
#include "ac_oggvorbis.h"
#endif /* HAVE_OGGVORBIS */

typedef struct {
  uint32_t codec;
  ACODEC_FUNCS *funcs;
} ACODEC_ENTRY;

static ACODEC_ENTRY acodec_entry[] = {
#ifdef HAVE_LIBMP3LAME
  { ACODEC_MP3, &mp3lame_funcs },
#endif /* HAVE_LIBMP3LAME */
  { ACODEC_PCM, &pcm_funcs },
  { ACODEC_PCM_LE, &pcm_funcs },
  { ACODEC_PCM_BE, &pcm_funcs },
#ifdef HAVE_OGGVORBIS
  { ACODEC_OGGVORBIS, &oggvorbis_funcs },
#endif /* HAVE_OGGVORBIS */
  { 0, NULL },
};

int acodec_entry_num = sizeof(acodec_entry)/sizeof(ACODEC_ENTRY) - 1;

static ACODEC acodec_enc;
static ACODEC acodec_dec;

int (*acodec_encode) (void *pic_data[], int *num_samples, uint8_t *stream_buf,
                       int stream_buf_size);

int (*acodec_decode) (void *pic_data[], uint8_t *stream_buf, int stream_size);

const char*
acodec_get_codec_name(uint32_t codec)
{
  int i;
  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec)
      return acodec_entry[i].funcs->get_codec_name();
  }
  return "UNKNOWN";
}

uint32_t
acodec_get_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;
  uint32_t cap = ACODEC_CAP_NONE;
  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec) {
      cap = acodec_entry[i].funcs->get_cap(codec);
      break;
    }
  }
  if (cap_flag != ACODEC_CAP_NONE)
    return (cap & cap_flag);
  return cap;
}

uint32_t
acodec_fourcc_to_codec(const char *fourcc)
{
  int i;
  uint32_t codec;
  for (i = 0; i < acodec_entry_num; i++) {
    codec = acodec_entry[i].funcs->fourcc_to_codec(fourcc);
    if (codec != ACODEC_UNKNOWN) 
      return codec;
  }
  return ACODEC_UNKNOWN;
}

uint32_t
acodec_code_to_codec(int code)
{
  int i;
  uint32_t codec;
  for (i = 0; i < acodec_entry_num; i++) {
    codec = acodec_entry[i].funcs->code_to_codec(code);
    if (codec != ACODEC_UNKNOWN) 
      return codec;
  }
  return ACODEC_UNKNOWN;
}

const char*
acodec_codec_to_fourcc(uint32_t codec)
{
  int i;
  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec)
      return acodec_entry[i].funcs->codec_to_fourcc(codec);
  }
  return "UNKNOWN";
}

int
acodec_codec_to_code(uint32_t codec)
{
  int i;
  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec)
      return acodec_entry[i].funcs->codec_to_code(codec);
  }
  return 0;
}


/* encoder functions */

int
acodec_encode_flush(int *num_samples, uint8_t *stream_buf, int stream_buf_size)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_flush) {
    return -1;
  }
  return acodec_enc.funcs->encode_flush(num_samples,stream_buf,stream_buf_size);
}

int
acodec_encode_set_codec(uint32_t codec)
{
  int i;

  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec) {
      acodec_enc.funcs = acodec_entry[i].funcs;
      acodec_enc.codec = codec;
      break;
    }
  }
  if (i >= acodec_entry_num) {
    fprintf(stderr, "acodec_encode_set_codec: cannot find audio codec.\n");
    return ACODEC_FAIL;
  }

  return ACODEC_OK;
}

int
acodec_encode_rate_cap(int rate)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_rate_cap) {
    return -1;
  }
  return acodec_enc.funcs->encode_rate_cap(rate);
}

int
acodec_encode_channels_cap(int channels)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_channels_cap) {
    return -1;
  }
  return acodec_enc.funcs->encode_channels_cap(channels);
}

int
acodec_encode_bits_cap(int bits)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_bits_cap) {
    return -1;
  }
  return acodec_enc.funcs->encode_bits_cap(bits);
}

int
acodec_encode_get_frame_size(void)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_get_frame_size) {
    return -1;
  }
  return acodec_enc.funcs->encode_get_frame_size();
}

int
acodec_encode_get_out_rate(void)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_get_out_rate) {
    return -1;
  }
  return acodec_enc.funcs->encode_get_out_rate();
}

int
acodec_encode_get_bitrate(void)
{
  if (!acodec_enc.funcs || !acodec_enc.funcs->encode_get_bitrate) {
    return -1;
  }
  return acodec_enc.funcs->encode_get_bitrate();
}

int
acodec_encode_reset(void)
{
  acodec_enc.funcs->encode_quit();
  return acodec_enc.funcs->encode_init(&acodec_enc);
}

int
acodec_encode_quit(void)
{
  int ret = ACODEC_OK;
  if (acodec_enc.funcs && acodec_enc.funcs->encode_quit)
    ret = acodec_enc.funcs->encode_quit();
  acodec_enc.codec = ACODEC_UNKNOWN;
  acodec_enc.rate = 0;
  acodec_enc.channels = 0;
  acodec_enc.bits = 0;
  acodec_enc.funcs = NULL;
  acodec_encode = NULL;
  return ret;
}

int
acodec_encode_init(uint32_t codec, int rate, int channels, int bits)
{
  int need_config = 0;
  static int first = 1;

  if (codec == ACODEC_UNKNOWN) {
    fprintf(stderr, "acodec_init: unknown codec.\n");
    return ACODEC_FAIL;
  }

  if (first) {
    acodec_enc.codec = ACODEC_UNKNOWN;
    acodec_enc.rate = 0;
    acodec_enc.channels = 0;
    acodec_enc.bits = 0;
    acodec_enc.funcs = NULL;

    first = 0;
    need_config = 1;
  }

  if (codec != acodec_enc.codec)
    need_config = 1;
  if (rate != acodec_enc.rate)
    need_config = 1;
  if (channels != acodec_enc.channels)
    need_config = 1;
  if (bits != acodec_enc.bits)
    need_config = 1;

  if (!need_config)
    return ACODEC_OK;

  if (acodec_enc.codec != ACODEC_UNKNOWN && acodec_enc.funcs != NULL)
    acodec_encode_quit();

  acodec_enc.codec = codec;
  acodec_enc.rate = rate;
  acodec_enc.channels = channels;
  acodec_enc.bits = bits;

  if (acodec_encode_set_codec(codec) != ACODEC_OK) {
    fprintf(stderr, "acodec_init: codec invalid.\n");
    return ACODEC_FAIL;
  }
  if (acodec_encode_rate_cap(rate) != rate) {
    fprintf(stderr, "acodec_init: rate invalid.\n");
    return ACODEC_FAIL;
  }
  if (acodec_encode_channels_cap(channels) != channels) {
    fprintf(stderr, "acodec_init: channels invalid.\n");
    return ACODEC_FAIL;
  }
  if (acodec_encode_bits_cap(bits) != bits) {
    fprintf(stderr, "acodec_init: bits invalid.\n");
    return ACODEC_FAIL;
  }

  acodec_enc.funcs->encode_init(&acodec_enc);
  acodec_encode = acodec_enc.funcs->encode;
  acodec_enc.rate = acodec_encode_get_out_rate();

  return ACODEC_OK;
}

void
acodec_encode_print_param(void)
{
  fprintf(stdout, "AUDIO CODEC: %s\n", acodec_get_codec_name(acodec_enc.codec));
  if (acodec_enc.funcs && acodec_enc.funcs->encode_print_param)
    acodec_enc.funcs->encode_print_param();
}


/* decoder functions */

int
acodec_decode_set_codec(uint32_t codec)
{
  int i;

  for (i = 0; i < acodec_entry_num; i++) {
    if (codec == acodec_entry[i].codec) {
      acodec_dec.funcs = acodec_entry[i].funcs;
      acodec_dec.codec = codec;
      break;
    }
  }
  if (i >= acodec_entry_num) {
    fprintf(stderr, "acodec_decode_set_codec: cannot find audio codec.\n");
    return ACODEC_FAIL;
  }

  return ACODEC_OK;
}

int
acodec_decode_get_rate(void)
{
  if (!acodec_dec.funcs || !acodec_dec.funcs->decode_get_rate) {
    return -1;
  }
  return acodec_dec.funcs->decode_get_rate();
}

int
acodec_decode_get_channels(void)
{
  if (!acodec_dec.funcs || !acodec_dec.funcs->decode_get_channels) {
    return -1;
  }
  return acodec_dec.funcs->decode_get_channels();
}

int
acodec_decode_get_bits(void)
{
  if (!acodec_dec.funcs || !acodec_dec.funcs->decode_get_bits) {
    return -1;
  }
  return acodec_dec.funcs->decode_get_bits();
}

int
acodec_decode_get_bitrate(void)
{
  if (!acodec_dec.funcs || !acodec_dec.funcs->decode_get_bitrate) {
    return -1;
  }
  return acodec_dec.funcs->decode_get_bitrate();
}

int
acodec_decode_analyze_stream(uint8_t *stream, int stream_size, int *rate, int *channels, int *bits)
{
  int ret;

  *rate = *channels = *bits = 0;
  if (!acodec_dec.funcs || !acodec_dec.funcs->decode_analyze_stream) {
    return -1;
  }
  ret =acodec_dec.funcs->decode_analyze_stream(&acodec_dec,stream,stream_size);
  if (ret != ACODEC_OK)
    return ret; 
  *rate = acodec_dec.rate;
  *channels = acodec_dec.channels;
  *bits = acodec_dec.bits;
  return ACODEC_OK;
}

int
acodec_decode_reset(void)
{
  acodec_dec.funcs->decode_quit();
  return acodec_dec.funcs->decode_init(&acodec_dec);
}

int
acodec_decode_quit(void)
{
  int ret = ACODEC_OK;
  if (acodec_dec.funcs && acodec_dec.funcs->decode_quit)
    ret = acodec_dec.funcs->decode_quit();
  acodec_dec.codec = ACODEC_UNKNOWN;
  acodec_dec.rate = 0;
  acodec_dec.channels = 0;
  acodec_dec.bits = 0;
  acodec_dec.funcs = NULL;
  acodec_decode = NULL;
  return ret;
}

int
acodec_decode_init(uint32_t codec, int rate, int channels, int bits)
{
  int need_config = 0;
  static int first = 1;

  if (codec == ACODEC_UNKNOWN) {
    fprintf(stderr, "acodec_init: unknown codec.\n");
    return ACODEC_FAIL;
  }

  if (first) {
    acodec_dec.codec = ACODEC_UNKNOWN;
    acodec_dec.rate = 0;
    acodec_dec.channels = 0;
    acodec_dec.bits = 0;
    acodec_dec.funcs = NULL;

    first = 0;
    need_config = 1;
  }

  if (codec != acodec_dec.codec)
    need_config = 1;
  if (rate != acodec_dec.rate)
    need_config = 1;
  if (channels != acodec_dec.channels)
    need_config = 1;
  if (bits != acodec_dec.bits)
    need_config = 1;

  if (!need_config)
    return ACODEC_OK;

  if (acodec_dec.codec != ACODEC_UNKNOWN && acodec_dec.funcs != NULL)
    acodec_decode_quit();

  if (acodec_decode_set_codec(codec) != ACODEC_OK) {
    fprintf(stderr, "acodec_init: codec invalid.\n");
    return ACODEC_FAIL;
  }

  acodec_dec.codec = codec;
  acodec_dec.rate = rate;
  acodec_dec.channels = channels;
  acodec_dec.bits = bits;


  acodec_decode = acodec_dec.funcs->decode;
  acodec_dec.funcs->decode_init(&acodec_dec);

  return ACODEC_OK;
}

void
acodec_decode_print_param(void)
{
  fprintf(stdout, "AUDIO CODEC: %s\n", acodec_get_codec_name(acodec_dec.codec));
  if (acodec_dec.funcs && acodec_dec.funcs->decode_print_param)
    acodec_dec.funcs->decode_print_param();
}

