/* ac_oggvorbis.c */

/*
 *
 * Referenced:
 * libvorbis-1.0:  encoder_example.c
 *
 */

#ifdef HAVE_OGGVORBIS


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <time.h>

#include <vorbis/vorbisenc.h>

#include "parseopt.h"
#include "acodec.h"
#include "ac_oggvorbis.h"

static OGGVORBIS_CONFIG oggvorbis_config = {
/* option parameter */
/* max_bitrate */ 64,
/* bitrate */     64,
/* min_bitrate */ 64,
/* quality */     5,

/* internal only */
/* rate */         0,
/* channels */     0,
/* bits */         0,
/* frame_size */   0,
/* use_vbr */      0,

};

static OGGVORBIS_CONFIG oggvorbis_dec_config;


static SEL_COMPO oggvorbis_encmode[] = {
  { "vbr", OGGVORBIS_ENCODE_MODE_VBR },
  { "abr", OGGVORBIS_ENCODE_MODE_ABR },
  { "managed", OGGVORBIS_ENCODE_MODE_MANAGED },
};

OptionDef oggvorbis_param[] = {
  {"oggvorbis-maxbitrate", "OGGVORBIS_MAXBITRATE", HAS_ARG, {(void*)&oggvorbis_config.max_bitrate}, {64}, {0}, 320, "oggvorbis max bitrate", "b"},
  {"oggvorbis-bitrate", "OGGVORBIS_BITRATE", HAS_ARG, {(void*)&oggvorbis_config.bitrate}, {64}, {0}, 320, "oggvorbis bitrate", "b"},
  {"oggvorbis-minbitrate", "OGGVORBIS_MINBITRATE", HAS_ARG, {(void*)&oggvorbis_config.min_bitrate}, {64}, {0}, 320, "oggvorbis min bitrate", "b"},
  {"oggvorbis-quality", "OGGVORBIS_QUALITY", HAS_ARG, {(void*)&oggvorbis_config.quality}, {5}, {0}, 10, "oggvorbis quality", "q"},
  {"oggvorbis-encmode", "OGGVORBIS_ENCMODE", HAS_ARG|OPT_SEL, {(void*)&oggvorbis_config.vbr}, {OGGVORBIS_ENCODE_MODE_VBR}, {(int)oggvorbis_encmode}, sizeof(oggvorbis_encmode)/sizeof(SEL_COMPO), "oggvorbis encode mode", "m"}
};

int oggvorbis_param_num = sizeof(oggvorbis_param) / sizeof(OptionDef);

const char*
oggvorbis_get_codec_name(void)
{
  return "OGGVORBIS";
}

uint32_t
oggvorbis_get_cap(codec)
{
  if (codec == ACODEC_OGGVORBIS)
    return ACODEC_CAP_ANALYSIS;
  return ACODEC_CAP_NONE;
}

uint32_t
oggvorbis_fourcc_to_codec(const char *fourcc)
{
  if (strncmp(fourcc, "OggS", 4) == 0) {
    return ACODEC_OGGVORBIS;
  }
  return ACODEC_UNKNOWN;
}

uint32_t
oggvorbis_code_to_codec(int code)
{
  if (code == 0xFFFE)
    return ACODEC_OGGVORBIS;
  return ACODEC_UNKNOWN;
}

const char*
oggvorbis_codec_to_fourcc(uint32_t codec)
{
  if (codec == ACODEC_OGGVORBIS) {
    return "OggS";
  }
  return NULL;
}

int
oggvorbis_codec_to_code(uint32_t codec)
{
  if (codec == ACODEC_OGGVORBIS)
    return 0xFFFE;
  return 0;
}


/* encode functions */

int
oggvorbis_encode_header(uint8_t *stream_buf,int stream_buf_size)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_config;
  ogg_packet header;
  ogg_packet header_comm;
  ogg_packet header_code;
  int length;


  if (conf->stream_header)
    return 0;

  vorbis_analysis_headerout(&conf->vd, &conf->vc, &header, &header_comm, &header_code);
  ogg_stream_packetin(&conf->os, &header);
  ogg_stream_packetin(&conf->os, &header_comm);
  ogg_stream_packetin(&conf->os, &header_code);

  length = 0;
  while (1) {
    int result;
    result = ogg_stream_flush(&conf->os, &conf->og);
    if (result == 0) break;
    memcpy(&stream_buf[length], conf->og.header, conf->og.header_len);
    length += conf->og.header_len;
    memcpy(&stream_buf[length], conf->og.body, conf->og.body_len);
    length += conf->og.body_len;
  }

  conf->stream_header = 1;

  return length;
}

int
oggvorbis_encode (void *pcm_buf[], int *num_samples, uint8_t *stream_buf,int stream_buf_size)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_config;
  int length;
  float **buffer;
  int i;
  int result;
  int num_spl;
  int eos = 0;
  short *sp[2];

  if (*num_samples != conf->frame_size) {
    fprintf(stderr, "oggvorbis_encode: num_samples %d.\n", *num_samples);
  }

  length = 0;

#if 0
  if (conf->header_buf_pos > 0) {
    memcpy(stream_buf, conf->header_buf, conf->header_buf_pos);
    length += conf->header_buf_pos;
    conf->header_buf_pos = 0;
  }
#endif
#if 0
  if (!conf->stream_header) {
    length = oggvorbis_encode_header(stream_buf, stream_buf_size);
  }
#endif

  num_spl = *num_samples;

  buffer = vorbis_analysis_buffer(&conf->vd, num_spl);

  sp[0] = (short*)pcm_buf[0];
  sp[1] = (short*)pcm_buf[1];
  for (i = 0; i < num_spl; i++) {
    buffer[0][i] = sp[0][i] / 32768.f;
    buffer[1][i] = sp[1][i] / 32768.f;
  }

  vorbis_analysis_wrote(&conf->vd, i);

  while (vorbis_analysis_blockout(&conf->vd, &conf->vb) == 1) {
    vorbis_analysis(&conf->vb, NULL);
    vorbis_bitrate_addblock(&conf->vb);

    while (vorbis_bitrate_flushpacket(&conf->vd, &conf->op)) {
      ogg_stream_packetin(&conf->os, &conf->op);

      while (!eos) {
	result = ogg_stream_pageout(&conf->os, &conf->og);
	if (result == 0) break;
        if (!conf->stream_header) {
          length += oggvorbis_encode_header(stream_buf, stream_buf_size);
        }
	memcpy(&stream_buf[length], conf->og.header, conf->og.header_len);
	length += conf->og.header_len;
	memcpy(&stream_buf[length], conf->og.body, conf->og.body_len);
	length += conf->og.body_len;
	if (ogg_page_eos(&conf->og)) eos = 1;
      }
    }
  }

  if (length > stream_buf_size) {
    fprintf(stderr, "oggvorbis_encode: stream_buf overflow, %d, %d.\n", stream_buf_size, length);
    exit(1);
  }

  return length;
}

int
oggvorbis_encode_flush (int *num_samples, uint8_t *stream_buf, int stream_buf_size)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_config;
  int length;
  int result;
  int eos;

  if (conf->flushed)
    return 0;

  length = 0;
  eos = 0;

  vorbis_analysis_wrote(&conf->vd, 0);

  while (vorbis_analysis_blockout(&conf->vd, &conf->vb) == 1) {
    vorbis_analysis(&conf->vb, NULL);
    vorbis_bitrate_addblock(&conf->vb);

    while (vorbis_bitrate_flushpacket(&conf->vd, &conf->op)) {
      ogg_stream_packetin(&conf->os, &conf->op);

      while (!eos) {
	result = ogg_stream_pageout(&conf->os, &conf->og);
	if (result == 0) break;
	memcpy(&stream_buf[length], conf->og.header, conf->og.header_len);
	length += conf->og.header_len;
	memcpy(&stream_buf[length], conf->og.body, conf->og.body_len);
	length += conf->og.body_len;

	if (ogg_page_eos(&conf->og)) eos = 1;
      }
    }
  }

  if (length > stream_buf_size) {
    fprintf(stderr, "oggvorbis_encode: stream_buf overflow, %d, %d.\n", stream_buf_size, length);
    exit(1);
  }

  conf->flushed = 1;

  return length;
}

int
oggvorbis_encode_rate_cap(int rate)
{
  if (rate < 100)
    rate *= 1000;

  if (rate < 8000)
    rate = 8000;
  else if (rate < 11025)
    rate = 8000;
  else if (rate < 12000)
    rate = 11025;
  else if (rate < 16000)
    rate = 12000;
  else if (rate < 22050)
    rate = 16000;
  else if (rate < 24000)
    rate = 22050;
  else if (rate < 32000)
    rate = 24000;
  else if (rate < 44100)
    rate = 32000;
  else if (rate < 48000)
    rate = 44100;
  else if (rate >= 48000)
    rate = 48000;
  return rate;
}

int
oggvorbis_encode_channels_cap(int channels)
{
  int ret;
  switch (channels) {
    case 1:
      ret = 1;
      break;
    default:
    case 2:
      ret = 2;
      break;
  }
  return ret;
}

int
oggvorbis_encode_bits_cap(int bits)
{
#if 0
  int ret;
  switch (bits) {
    default:
    case 16:
      ret = 16;
      break;
  }
  return ret;
#endif
  return 16;
}

int
oggvorbis_encode_get_frame_size(void)
{
  return oggvorbis_config.frame_size;
}

int
oggvorbis_encode_get_bitrate(void)
{
  return oggvorbis_config.bitrate;
}

int
oggvorbis_encode_quit (void)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_config;

  if (conf->header_buf)
    free(conf->header_buf);
  conf->header_buf = NULL;
  conf->header_buf_size = 0;
  conf->header_buf_pos = 0;

  ogg_stream_clear(&conf->os);
  vorbis_block_clear(&conf->vb);
  vorbis_dsp_clear(&conf->vd);
  vorbis_comment_clear(&conf->vc);
  vorbis_info_clear(&conf->vi);

  conf->flushed = 0;
  conf->vbr = 0;
  conf->stream_header = 0;

  return ACODEC_OK;
}

int
oggvorbis_encode_init (ACODEC *codec)
{
  static int first = 1;
  OGGVORBIS_CONFIG *conf = &oggvorbis_config;
  int ret;
  int need_config = 0;

#if 0
  ogg_packet header;
  ogg_packet header_comm;
  ogg_packet header_code;
#endif

  if (first) {
    conf->rate = 0;
    conf->channels = 0;
    conf->bits = 0;
    conf->frame_size = 0;
    conf->header_buf = NULL;
    conf->header_buf_size = 0;
    conf->header_buf_pos = 0;
    conf->flushed = 0;
    conf->vbr = 0;
    conf->stream_header = 0;
    first = 0;
    need_config = 1;
  }

  if (conf->rate != codec->rate)
    need_config = 1;
  else if (conf->channels != codec->channels)
    need_config = 1;
  else if (conf->bits != codec->bits)
    need_config = 1;

  if (!need_config) {
    return ACODEC_OK;
  }

  conf->rate = codec->rate;
  conf->channels = codec->channels;
  conf->bits = codec->bits;
  conf->frame_size = OGGVORBIS_FRAME_SIZE;

  vorbis_info_init(&conf->vi);

  printf("channels %d, rate %d, quality %f\n", conf->channels, conf->rate, (((float)conf->quality)/10.f));
  switch (conf->vbr) {
    case OGGVORBIS_ENCODE_MODE_VBR:
      ret = vorbis_encode_init_vbr(&conf->vi, conf->channels, conf->rate,
	  ((float)conf->quality) / 10.f);
      break;
    case OGGVORBIS_ENCODE_MODE_ABR:
      ret = vorbis_encode_init(&conf->vi, conf->channels, conf->rate,
	  conf->max_bitrate, conf->bitrate, conf->min_bitrate);
      break;
    case OGGVORBIS_ENCODE_MODE_MANAGED:
      ret = (
	  vorbis_encode_setup_managed(&conf->vi, conf->channels, conf->rate,
	    conf->max_bitrate, conf->bitrate, conf->min_bitrate) ||
	  vorbis_encode_ctl(&conf->vi, OV_ECTL_RATEMANAGE_AVG, NULL) ||
	  vorbis_encode_setup_init(&conf->vi) );
      break;
    default:
      fprintf (stderr, "oggvorbis_init(): not support %d channel enccode. channel must be 1 or 2.\n", conf->channels);
      return ACODEC_FAIL;
      break;
  }

  if (ret) {
    fprintf(stderr, "oggvorbis_encode_init: encoder initialize failed.\n");
    return ACODEC_FAIL;
  }

  vorbis_comment_init(&conf->vc);
  vorbis_comment_add_tag(&conf->vc, "ENCODER", "acodec encoder");

  vorbis_analysis_init(&conf->vd, &conf->vi);
  vorbis_block_init(&conf->vd, &conf->vb);

  srand(time(NULL));
  ogg_stream_init(&conf->os, rand());

#if 0
  vorbis_analysis_headerout(&conf->vd, &conf->vc, &header, &header_comm, &header_code);
  ogg_stream_packetin(&conf->os, &header);
  ogg_stream_packetin(&conf->os, &header_comm);
  ogg_stream_packetin(&conf->os, &header_code);

  conf->header_buf_pos = 0;
  while (1) {
    int result;
    uint8_t *hp;
    result = ogg_stream_flush(&conf->os, &conf->og);
    if (result == 0) break;
    while ((conf->header_buf_pos + conf->og.header_len + conf->og.body_len) >
	  conf->header_buf_size) {
      hp = (uint8_t*) realloc(conf->header_buf, conf->header_buf_size + 1024);
      if (!hp) {
	fprintf(stderr, "oggvorbis_encode_init: memory_allocate error.\n");
	oggvorbis_encode_quit();
	return ACODEC_FAIL;
      }
      conf->header_buf = hp;
      conf->header_buf_size += 1024;
    }
    memcpy(conf->header_buf+conf->header_buf_pos, conf->og.header, conf->og.header_len);
    conf->header_buf_pos += conf->og.header_len;
    memcpy(conf->header_buf+conf->header_buf_pos, conf->og.body, conf->og.body_len);
    conf->header_buf_pos += conf->og.body_len;
  }
#endif

  conf->flushed = 0;
  conf->stream_header = 0;

  return ACODEC_OK;
}

void
oggvorbis_encode_print_param (void)
{
  SEL_COMPO *sel;
  int i;

  fprintf (stderr, "ENCPARAM OGGVORBIS\n");
#if 0
  fprintf (stderr, "bitrate:                 %d\n", oggvorbis_config.bitrate);
  fprintf (stderr, "quality:                 %d\n", oggvorbis_config.quality);
  sel = oggvorbis_mode;
  for (i=0; i < sizeof(oggvorbis_mode)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == oggvorbis_config.mode) {
      fprintf (stderr, "mode:                    %s\n", sel->name);
      break;
    }
  }
  if (i == sizeof(oggvorbis_mode)/sizeof(SEL_COMPO))
    fprintf (stderr, "mode:                    %s\n", "UNKNOWN");
  fprintf (stderr, "frame size:              %d\n", oggvorbis_config.frame_size);
#endif

  fflush (stderr);
}


/* decode functions */

#if 0
int
oggvorbis_decode (void *pcm_buf[], uint8_t *stream, int stream_size)
{
  int num_spl;
//  OGGVORBIS_CONFIG *conf = &oggvorbis_dec_config;

  num_spl = lame_decode(stream, stream_size,
                   (short*)pcm_buf[0], (short*)pcm_buf[1]);
  return num_spl;
}

int
oggvorbis_decode_get_rate(void)
{
  return oggvorbis_dec_config.rate;
}

int
oggvorbis_decode_get_channels(void)
{
  return oggvorbis_dec_config.channels;
}

int
oggvorbis_decode_get_bits(void)
{
  return 16;
}

int
oggvorbis_decode_get_bitrate(void)
{
  return oggvorbis_dec_config.bitrate;
}

int
oggvorbis_decode_quit (void)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_dec_config;

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
  }
  conf->rate = 0;
  conf->channels = 0;
  conf->bits = 0;
//  conf->bitrate = 0;
  conf->analyzed = 0;
  conf->vbr = 0;

  return ACODEC_OK;
}

int
oggvorbis_decode_analyze_stream(ACODEC *acodec, uint8_t *stream, int stream_size)
{
  OGGVORBIS_CONFIG *conf = &oggvorbis_dec_config;
  int start = 0;
  int vbr = 0;
  uint8_t *strm;
  int size;
  int buf_pos;
  int frame_length;

  buf_pos = conf->stream_buf_pos;
  size = buf_pos + stream_size;

  strm = conf->stream_buf;
  if (size > conf->stream_buf_size) {
    strm = (uint8_t*)realloc(conf->stream_buf, size);
    if (!strm) {
      fprintf(stderr,"oggvorbis_decode_analyze_stream: memory allocate error.\n");
      exit(1);
    }
    conf->stream_buf_size = size;
  }
  memcpy(strm + buf_pos, stream, stream_size);
  buf_pos += stream_size;
  conf->stream_buf = strm;
  conf->stream_buf_pos = buf_pos;

  vbr = oggvorbis_frame_header_vbr(&conf->oggvorbisframe, strm, buf_pos, &start);
  if (vbr < 0) {
    if (start > 0)
      memmove(strm, strm + start, buf_pos - start);
    conf->stream_buf_pos = buf_pos - start;
    return ACODEC_NEED_MORE_DATA;
  } else if (vbr > 0) {
    conf->vbr = 1;
    if (buf_pos < (start+vbr)) {
      if (start > 0)
        memmove(strm, strm + start, buf_pos - start);
      conf->stream_buf_pos = buf_pos - start;
      return ACODEC_NEED_MORE_DATA;
    }
    memmove(strm, strm + start + vbr, buf_pos - start - vbr);
    buf_pos -= (start + vbr);
    conf->stream_buf_pos = buf_pos;
  }

  frame_length = oggvorbis_frame_header(&conf->oggvorbisframe, strm, buf_pos, &start);
  if (frame_length <= 0) {
    if (start > 0)
      memmove(strm, strm + start, buf_pos - start);
    buf_pos -= start;
    conf->stream_buf_pos = buf_pos;
    return ACODEC_NEED_MORE_DATA;
  }

  acodec->rate = conf->rate = conf->oggvorbisframe.frequency;
  switch (conf->oggvorbisframe.mode) {
    case MP3_MODE_STEREO:
    case MP3_MODE_JOINT:
    case MP3_MODE_DUAL:
      acodec->channels = conf->channels = 2;
      break;
    case MP3_MODE_MONO:
      conf->channels = 1;
  }
  acodec->bits = conf->bits = 16;
  conf->bitrate = conf->oggvorbisframe.bitrate;
  conf->analyzed = 1;

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_pos = 0;
    conf->stream_buf_size = 0;
  }

  return ACODEC_OK;
}

int
oggvorbis_decode_init (ACODEC *codec)
{
  static int first = 1;
  OGGVORBIS_CONFIG *conf = &oggvorbis_dec_config;

  if (first) {
    conf->rate = 0;
    conf->channels = 0;
    conf->bits = 0;
    conf->bitrate = 0;
    conf->analyzed = 0;
    conf->vbr = 0;
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
    first = 0;
  }

  if (!conf->analyzed) {
    conf->rate = codec->rate;
    conf->channels = codec->channels;
    conf->bits = codec->bits;
  }

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
  }

  lame_decode_init();

  return ACODEC_OK;
}

void
oggvorbis_decode_print_param (void)
{
  fprintf (stderr, "DECPARAM OGGVORBIS\n");
  fprintf (stderr, "rate:                    %d\n", oggvorbis_dec_config.rate);
  fprintf (stderr, "channels:                %d\n",oggvorbis_dec_config.channels);
  fprintf (stderr, "bits:                    %d\n", oggvorbis_dec_config.bits);
  fprintf (stderr, "bitrate:                 %d\n", oggvorbis_dec_config.bitrate);
  fflush (stderr);
}
#endif


/* function structure */

ACODEC_FUNCS oggvorbis_funcs = {
  oggvorbis_get_codec_name,
  oggvorbis_get_cap,
  oggvorbis_fourcc_to_codec,
  oggvorbis_code_to_codec,
  oggvorbis_codec_to_fourcc,
  oggvorbis_codec_to_code,

  oggvorbis_encode_init,
  oggvorbis_encode_quit,
  oggvorbis_encode,
  oggvorbis_encode_flush,
  oggvorbis_encode_rate_cap,
  oggvorbis_encode_channels_cap,
  oggvorbis_encode_bits_cap,
  oggvorbis_encode_get_frame_size,
  oggvorbis_encode_get_bitrate,
  oggvorbis_encode_print_param,

#if 0
  oggvorbis_decode_init,
  oggvorbis_decode_quit,
  oggvorbis_decode,
  oggvorbis_decode_get_rate,
  oggvorbis_decode_get_channels,
  oggvorbis_decode_get_bits,
  oggvorbis_decode_get_bitrate,
  oggvorbis_decode_analyze_stream,
#endif
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
};

#endif /* HAVE_OGGVORBIS */

