/* util.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <inttypes.h>
#include <pwd.h>
#include <errno.h>

#include "util.h"

void
mem_alloc_fail(const char *message, int status)
{
  fprintf(stderr, "%s: %s\n", message, "memory allocate error.");
  if (status)
    exit(status);
}

void*
mem_malloc(int size, const char *mes)
{
  void *mp;
  mp = malloc(size);
  if (!mp) {
    if (!mes)
      mes = "mem_malloc";
    mem_alloc_fail(mes, IMM_EXIT);
  }
  return mp;
}

void*
mem_realloc(void *op, int size, const char *mes)
{
  void *mp;
  mp = realloc(op, size);
  if (!mp) {
    if (!mes)
      mes = "mem_realloc";
    mem_alloc_fail(mes, IMM_EXIT);
  }
  return mp;
}

void*
mem_align(int byte_size, int align)
{
  void *ptr;

  if (posix_memalign(&ptr, align, byte_size) == ENOMEM) {
    fprintf(stderr, "mem_align: not enough memory, %d bytes.\n", byte_size);
    return NULL;
  }

  return ptr;
}

u_time
get_cur_u_time (void)
{
  u_time t;
  struct timeval tv;

  gettimeofday (&tv, 0);

  t = (((u_time)tv.tv_sec) * 1000000) + ((u_time)tv.tv_usec);
  return t;
}

int
get_home_dir(const char *user_name, char *buf, int buf_size)
{
  int len;
  char *home;
  struct passwd *pw;

  buf[0] = '\0';
  if (user_name) {
    pw = getpwnam(user_name);
    if (!pw) {
      return 0;
    }
    len = strlen(pw->pw_dir);
    if (len >= buf_size) {
      fprintf(stderr, "get_home_dir: buf_size small.\n");
      return 0;
    }
    strcpy(buf, pw->pw_dir);
    return len;
  } else {
    home = getenv("HOME");
    if (home && getuid() != 0) {
      len = strlen(home);
      if (len >= buf_size) {
        fprintf(stderr, "get_home_dir: buf_size small.\n");
        return 0;
      }
      strcpy(buf, home);
      return len;
    } else {
      uid_t uid = getuid();
      pw = getpwuid(uid);
      if (!pw) {
        return 0;
      }
      len = strlen(pw->pw_dir);
      if (len >= buf_size) {
        fprintf(stderr, "get_home_dir: buf_size small.\n");
        return 0;
      }
      strcpy(buf, pw->pw_dir);
      return len;
    }
  }
  return 0;
}

int
expand_path(const char *path, char *buf, int buf_size)
{
  char user_name[64];
  const char *p, *u;
  int i, path_len, len;

  buf[0] = '\0';
  len = 0;
  if (path[0] == '~') {
    p = path + 1;
    if (*p == '/') {
      u = NULL;
    } else {
      for (i = 0; *p != '/' && *p != '\0' && i < 64; p++, i++)
	user_name[i] = *p;
      if (i == 64)
	return 0;
      user_name[i] = '\0';
      u = user_name;
    }
    path_len = strlen(p);
    if ((len = get_home_dir(u, buf, buf_size - path_len)) == 0 ||
        (len + path_len) >= buf_size)
      return 0;
    strcpy(buf+len, p);
    len += path_len;
  } else {
    len = strlen(path);
    if (len >= buf_size)
      return 0;
    strcpy(buf, path);
  }
  return len;
}

FILE*
xfopen(const char *path, const char *mode)
{
  FILE *fp;
  int buf_size = 4096;
  char buf[4096];

  if (expand_path(path, buf, buf_size) == 0)
    return NULL;
  fp = fopen(buf, mode);
  return fp;
}

