/* median_filt.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include "csp.h"
#include "median_filt.h"

typedef struct {
  unsigned char r;
  unsigned char g;
  unsigned char b;
} PIXEL_RGB;

static int 
med_sort(int data[], int num, int shift)
{
  int i;
  int val;
  int big_count;
  int small_count;
  int near_big;
  int near_small;
  int big_num = num / 2 - shift;
  int small_num = num / 2 + shift;

  near_big = UCHAR_MAX;
  near_small = 0;

  val = data[0];

  big_count = small_count = 0;
  for (i = 0; i < num; i++) {
    if (data[i] > val) {
      big_count++;
      if (data[i] < near_big)
        near_big = data[i];
      if (big_count > big_num) {
        big_count = small_count = i = 0;
        near_small = val;
        val = near_big;
        near_big = UCHAR_MAX;
        continue;
      }
    } else if (data[i] < val) {
      small_count++;
      if (data[i] > near_small)
        near_small = data[i];
      if (small_count > small_num) {
	big_count = small_count = i = 0;
        near_big = val;
        val = near_small;
        near_small = 0;
	continue;
      }
    }
  }

  return val;
}

static void
median_filter_rgb(unsigned char *dest, unsigned char *image_data, int width, int height, int stride)
{
  int red[9];
  int blue[9];
  int green[9];
  int r;
  int g;
  int b;
  int x, y;
  int i;
  int pos;

  PIXEL_RGB *sp;
  PIXEL_RGB *dp;
  PIXEL_RGB line_buf[3][width + 2];
  PIXEL_RGB *yp[3];
  PIXEL_RGB *tmp;

  dp = (PIXEL_RGB*)dest;
  sp = (PIXEL_RGB*)image_data;

  yp[0] = line_buf[0];
  yp[1] = line_buf[1];
  yp[2] = line_buf[2];

  memcpy(&yp[0][1], sp, width * sizeof(PIXEL_RGB));
  yp[0][0] = sp[0];
  yp[0][width] = sp[width-1];
  memcpy(&yp[1][1], sp, width * sizeof(PIXEL_RGB));
  yp[1][0] = sp[0];
  yp[1][width] = sp[width-1];
  sp += stride;
  memcpy(&yp[2][1], sp, width * sizeof(PIXEL_RGB));
  yp[2][0] = sp[0];
  yp[2][width] = sp[width-1];
  sp += stride;

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      pos = 0;
      for (i = 0; i < 3; i++) {
	red[pos] = yp[i][x-1].r;
	green[pos] = yp[i][x-1].g;
	blue[pos] = yp[i][x-1].b;
	pos++;
	red[pos] = yp[i][x].r;
	green[pos] = yp[i][x].g;
	blue[pos] = yp[i][x].b;
	pos++;
	red[pos] = yp[i][x+1].r;
	green[pos] = yp[i][x+1].g;
	blue[pos] = yp[i][x+1].b;
	pos++;
      }
      r = med_sort(red, 9, 0);
      g = med_sort(green, 9, 0);
      b = med_sort(blue, 9, 0);
      dp[x].r = r;
      dp[x].g = g;
      dp[x].b = b;
    }
    dp += stride;

    tmp = yp[0];
    yp[0] = yp[1];
    yp[1] = yp[2];
    if (y < height-2) {
      yp[2] = tmp;
      memcpy(&yp[2][1], sp, width * sizeof(PIXEL_RGB));
      yp[2][0] = sp[0];
      yp[2][width] = sp[width-1];
      sp += stride;
    }
  }
}

static void
median_filter_yuv(unsigned char *dest, unsigned char *image_data, int width, int height, int stride)
{
  int point[9];
  int x, y;
  unsigned char *sp;
  int *xp[3];
  int *tmpx;

  unsigned char *dp;
  unsigned char *yp[3];

  dp = dest;
  sp = image_data;

  xp[0] = &point[0];
  xp[1] = &point[3];
  xp[2] = &point[6];

  yp[0] = sp;
  yp[1] = sp;
  sp += stride;
  yp[2] = sp;
  sp += stride;

  width--;
  for (y = 0; y < height; y++) {
    xp[0][0] = yp[0][0];
    xp[0][1] = yp[1][0];
    xp[0][2] = yp[2][0];
    xp[1][0] = yp[0][0];
    xp[1][1] = yp[1][0];
    xp[1][2] = yp[2][0];
    for (x = 0; x < width; x++) {
      xp[2][0] = yp[0][x+1];
      xp[2][1] = yp[1][x+1];
      xp[2][2] = yp[2][x+1];
#if 0
      for (i = 0; i < 3; i++) {
	point[pos++] = yp[i][x-1];
	point[pos++] = yp[i][x];
	point[pos++] = yp[i][x+1];
      }
#endif
      dp[x] = med_sort(point, 9, 0);

      tmpx = xp[0];
      xp[0] = xp[1];
      xp[1] = xp[2];
      xp[2] = tmpx;
    }
    dp[x] = med_sort(point, 9, 1);
    dp += stride;

    yp[0] = yp[1];
    yp[1] = yp[2];
    if (y < height-2) {
      yp[2] = sp;
      sp += stride;
    }

  }
}

int
median_filter(unsigned char *dest, unsigned char *image_data, int width, int height, int stride, int csp)
{
  if (csp == CSP_RGB24) {
    median_filter_rgb(dest, image_data, width, height, stride);
    return 0;
  } else if (csp == CSP_YV12 || csp == CSP_YUV420P || csp == CSP_I420) {
    int v_offset = height * stride;
    int u_offset = v_offset + (v_offset / 4);
    median_filter_yuv(dest, image_data, width, height, stride);
    memcpy(dest + v_offset, image_data + v_offset, u_offset - v_offset);
    memcpy(dest + u_offset, image_data + u_offset, u_offset - v_offset);
//    width /= 2;
//    height /= 2;
//    stride /= 2;
//    median_filter_yuv(dest + v_offset, image_data+v_offset, width, height, stride);
//    median_filter_yuv(dest + v_offset, image_data+u_offset, width, height, stride);
    return 0;
  } else
    return -1;
}
