/* vmhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "vmhd.h"

void
vmhd_init(vmhd_t *vmhd)
{
  atom_init(&vmhd->atom);
  vmhd->atom.type = QTIME_TYPE_VMHD;
  vmhd->version = 0;
  qtime_flags_set(vmhd->flags, 1);
  vmhd->graphics_mode = 64;
  vmhd->opcolor[0] = 32768;
  vmhd->opcolor[1] = 32768;
  vmhd->opcolor[2] = 32768;
}

void
vmhd_clean(vmhd_t *vmhd)
{
  qtime_error_type_check(QTIME_TYPE_VMHD, vmhd->atom.type)
  vmhd_init(vmhd);
}

vmhd_t*
vmhd_new(void)
{
  vmhd_t *vmhd;
  vmhd = (vmhd_t*)qtime_malloc(sizeof(vmhd_t));
  if (!vmhd) return NULL;
  vmhd_init(vmhd);
  return vmhd;
}

void
vmhd_delete(vmhd_t *vmhd)
{
  qtime_error_type_check(QTIME_TYPE_VMHD, vmhd->atom.type)
  qtime_free(vmhd);
}

vmhd_t*
vmhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, vmhd_t *vmhd_ptr)
{
  vmhd_t *vmhd;

#ifndef NDEBUG
  if (atom_head->media_type != QTIME_MEDIA_TYPE_UNKNOWN &&
      atom_head->media_type != QTIME_MEDIA_TYPE_VIDEO) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    qtime_error_invalid_media(QTIME_TYPE_VMHD, QTIME_MEDIA_TYPE_VIDEO, atom_head->media_type);
    return NULL;
  }
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_VMHD, atom_head->type, NULL)
  if (vmhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_VMHD, vmhd_ptr->atom.type, NULL)
    vmhd = vmhd_ptr;
    vmhd_clean(vmhd);
  } else {
    if ((vmhd = vmhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  vmhd->atom.size = atom_head->size;
  vmhd->atom.type = QTIME_TYPE_VMHD;
  vmhd->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &vmhd->version, 1);
  qtime_io_read(qtio, vmhd->flags, 3);
  qtime_io_read16(qtio, &vmhd->graphics_mode);
  qtime_io_read16(qtio, &vmhd->opcolor[0]);
  qtime_io_read16(qtio, &vmhd->opcolor[1]);
  qtime_io_read16(qtio, &vmhd->opcolor[2]);

  atom_head->media_type = QTIME_MEDIA_TYPE_VIDEO;
  return vmhd;
}

vmhd_t*
vmhd_create(vmhd_t *vmhd_ptr)
{
  vmhd_t *vmhd;

  if (vmhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_VMHD, vmhd_ptr->atom.type, NULL)
    vmhd = vmhd_ptr;
    vmhd_clean(vmhd);
  } else {
    if ((vmhd = vmhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return vmhd;
}

int64_t
vmhd_calc_size(vmhd_t *vmhd)
{
  int64_t size = 0;
  qtime_error_type_check_v(QTIME_TYPE_VMHD, vmhd->atom.type, 0)
  size = 8 + VMHD_PROP_SIZE;
  vmhd->atom.size = size;
  return size;
}

int
vmhd_write_atom(qtime_io_t *qtio, vmhd_t *vmhd)
{
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_VMHD, vmhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = vmhd->atom.size;
  atom_head.type = vmhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_VMHD);
    return QTIME_ERROR_WRITE;
  }

  qtime_io_write(qtio, &vmhd->version, 1);
  qtime_io_write(qtio, vmhd->flags, 3);
  qtime_io_write16(qtio, &vmhd->graphics_mode);
  qtime_io_write16(qtio, &vmhd->opcolor[0]);
  qtime_io_write16(qtio, &vmhd->opcolor[1]);
  qtime_io_write16(qtio, &vmhd->opcolor[2]);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_VMHD);
    return QTIME_ERROR_WRITE;
  }

  return QTIME_OK;
}

void
vmhd_dump(const char *parent_types, vmhd_t *vmhd)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  qtime_type_to_str(vmhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(vmhd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)vmhd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, vmhd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: graphics mode %u\n", types, vmhd->graphics_mode);
  for (i = 0; i < 3; i++) {
    fprintf(stdout, "%s: opcolor[%d]  %u\n", types, i, vmhd->opcolor[i]);
  }
}

