/* trak.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "trak.h"
#include "mdia.h"
#include "tkhd.h"
//#include "clip.h"
//#include "matt.h"
#include "edts.h"
#include "tref.h"
#include "load.h"
//#include "imap.h"
#include "udta.h"

void
trak_init(trak_t *trak)
{
  atom_init(&trak->atom);
  trak->atom.type = QTIME_TYPE_TRAK;
  tkhd_init(&trak->tkhd);
  mdia_init(&trak->mdia);
  edts_init(&trak->edts);
  trak->tref = NULL;
  trak->load = NULL;
  trak->udta = NULL;
//  trak->matt = NULL;
//  trak->imap = NULL;
}

void
trak_clean(trak_t *trak)
{
  int i;
  atom_t *atom = &trak->atom;

  qtime_error_type_check(QTIME_TYPE_TRAK, trak->atom.type);

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&trak->tkhd) {
      tkhd_clean((tkhd_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&trak->edts) {
      edts_clean((edts_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&trak->mdia) {
      mdia_clean((mdia_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_TKHD:
        tkhd_delete((tkhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_EDTS:
        edts_delete((edts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_delete((udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_LOAD:
        load_delete((load_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MDIA:
	mdia_delete((mdia_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TREF:
        tref_delete((tref_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_delete((clip_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MATT:
        matt_delete((matt_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_IMAP:
        imap_delete((imap_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_TRAK, atom->childs[i]->type);
	break;
    }
  }

  atom_clean(&trak->atom);
  trak->atom.type = QTIME_TYPE_TRAK;
  trak->tref = NULL;
  trak->load = NULL;
  trak->udta = NULL;
//  trak->matt = NULL;
//  trak->clip = NULL;
//  trak->imap = NULL;
}

trak_t*
trak_new(void)
{
  trak_t *trak;
  trak = (trak_t*)qtime_malloc(sizeof(trak_t));
  if (!trak) return NULL;
  trak_init(trak);
  return trak;
}

void
trak_delete(trak_t *trak)
{
  qtime_error_type_check(QTIME_TYPE_TRAK, trak->atom.type)
  trak_clean(trak);
  qtime_free(trak);
}

trak_t*
trak_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, trak_t *trak_ptr)
{
  atom_t *atom;
  atom_head_t subatom;
  int tkhd_index, mdia_index, edts_index, tref_index, load_index, udta_index;
  trak_t *trak;
  tkhd_t *tkhd;
  mdia_t *mdia;
  edts_t *edts;
  tref_t *tref;
  load_t *load;
  udta_t *udta;
#if 0
  int clip_index, matt_index, imap_index;
  clip_t *clip;
  matt_t *matt;
  imap_t *imap;
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_TRAK, atom_head->type, NULL);

  if (trak_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_TRAK, trak_ptr->atom.type, NULL);
    trak = trak_ptr;
    trak_clean(trak);
  } else {
    if ((trak = trak_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  trak->atom.size = atom_head->size;
  trak->atom.type = QTIME_TYPE_TRAK;
  trak->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)trak;

  tkhd = &trak->tkhd;
  mdia = &trak->mdia;
  edts = &trak->edts;
  tref = NULL;
  load = NULL;
  udta = NULL;
  tkhd_index = mdia_index = edts_index = tref_index = load_index = udta_index=0;

#if 0
  clip = &trak->clip;
  matt = &trak->matt;
  imap = &trak->imap;
  clip_index = matt_index = imap_index = 0;
#endif
  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_TKHD:
	if ((tkhd = tkhd_read_atom(qtio, &subatom, tkhd)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)tkhd;
	atom->index = ++tkhd_index;
	tkhd = NULL;
	break;
      case QTIME_TYPE_EDTS:
	if ((edts = edts_read_atom(qtio, &subatom, edts)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)edts;
	atom->index = ++edts_index;
	edts = NULL;
	break;
      case QTIME_TYPE_MDIA:
	if ((mdia = mdia_read_atom(qtio, &subatom, mdia)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)mdia;
	atom->index = ++mdia_index;
	mdia = NULL;
	break;
      case QTIME_TYPE_UDTA:
	if ((udta = udta_read_atom(qtio, &subatom, udta)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)udta;
	atom->index = ++udta_index;
	udta = NULL;
	break;
      case QTIME_TYPE_LOAD:
	if ((load = load_read_atom(qtio, &subatom, load)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)load;
	atom->index = ++load_index;
	load = NULL;
	break;
      case QTIME_TYPE_TREF:
	if ((tref = tref_read_atom(qtio, &subatom, tref)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)tref;
	atom->index = ++tref_index;
	tref = NULL;
	break;
#if 0
      case QTIME_TYPE_CLIP:
	if ((clip = clip_read_atom(qtio, &subatom, clip)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)clip;
	atom->index = ++clip_index;
	clip = NULL;
	break;
      case QTIME_TYPE_MATT:
	if ((matt = matt_read_atom(qtio, &subatom, matt)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)matt;
	atom->index = ++matt_index;
	matt = NULL;
	break;
      case QTIME_TYPE_IMAP:
	if ((imap = imap_read_atom(qtio, &subatom, imap)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)imap;
	atom->index = ++imap_index;
	imap = NULL;
	break;
#endif
      default:
        qtime_error_unknown_atom(QTIME_TYPE_TRAK, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      atom->parent = (atom_t*)trak;
      if ((atom_head->error_code = atom_add_child((atom_t*)trak, atom)) < 0) {
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  trak->tref = (tref_t*)atom_find((atom_t*)trak, QTIME_TYPE_TREF, ATOM_FIND_CHILDS_ONLY);
  trak->load = (load_t*)atom_find((atom_t*)trak, QTIME_TYPE_LOAD, ATOM_FIND_CHILDS_ONLY);
  trak->udta = (udta_t*)atom_find((atom_t*)trak, QTIME_TYPE_UDTA, ATOM_FIND_CHILDS_ONLY);

  return trak;

fail:
  if (trak_ptr)
    trak_clean(trak);
  else
    trak_delete(trak);
  qtime_error_atom_read(QTIME_TYPE_TRAK);
  return NULL;
}

trak_t*
trak_create(trak_t *trak_ptr)
{
  trak_t *trak;
  int error_code;

  if (trak_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_TRAK, trak_ptr->atom.type, NULL);
    trak = trak_ptr;
    trak_clean(trak);
  } else {
    if ((trak = trak_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  tkhd_create(&trak->tkhd);
  edts_create(&trak->edts);
  mdia_create(&trak->mdia);

  if ((error_code = atom_add_child((atom_t*)trak, (atom_t*)&trak->tkhd)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)trak, (atom_t*)&trak->edts)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)trak, (atom_t*)&trak->mdia)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return trak;

fail:
  if (trak_ptr)
    trak_clean(trak);
  else
    trak_delete(trak);
  qtime_error_debug_info(error_code);
  return NULL;
}

int64_t
trak_calc_size(trak_t *trak)
{
  int i;
  int64_t size = 0;
  atom_t *atom = &trak->atom;

  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, 0);

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_TKHD:
        size += tkhd_calc_size((tkhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_EDTS:
        size += edts_calc_size((edts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MDIA:
        size += mdia_calc_size((mdia_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_LOAD:
        size += load_calc_size((load_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        size += udta_calc_size((udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TREF:
        size += tref_calc_size((tref_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        size += clip_calc_size((clip_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MATT:
        size += matt_calc_size((matt_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_IMAP:
        size += imap_calc_size((imap_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_TRAK, atom->childs[i]->type);
	break;
    }
  }

  if (size & SIZE64MASK)
    size += 8 + 8;
  else
    size += 8;
  trak->atom.size = size;
  return size;
}

int
trak_write_atom_pre(trak_t *trak)
{
  udta_t *udta;

  udta = (udta_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_UDTA);
  if (udta) {
    atom_remove_child((atom_t*)trak, (atom_t*)udta);
    atom_add_child((atom_t*)trak, (atom_t*)udta);
  }
  return QTIME_OK;
}

int
trak_write_atom(qtime_io_t *qtio, trak_t *trak)
{
  int i;
  atom_head_t atom_head;
  atom_t *atom = &trak->atom;

  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);

  trak_write_atom_pre(trak);

  atom_head_init(&atom_head);
  atom_head.size = trak->atom.size;
  atom_head.type = trak->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TRAK);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_TKHD:
        tkhd_write_atom(qtio, (tkhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_EDTS:
        edts_write_atom(qtio, (edts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MDIA:
        mdia_write_atom(qtio, (mdia_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_LOAD:
        load_write_atom(qtio, (load_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_write_atom(qtio, (udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TREF:
        tref_write_atom(qtio, (tref_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_write_atom(qtio, (clip_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MATT:
        matt_write_atom(qtio, (matt_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_IMAP:
        imap_write_atom(qtio, (imap_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_TRAK, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TRAK);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
trak_dump(const char *parent_types, trak_t *trak)
{
  int len = strlen(parent_types);
  uint8_t types[len+6+10];
  uint8_t type[5];
  int i, id;
  atom_t *atom = &trak->atom;

  id = trak_get_id(trak);
  qtime_type_to_str(trak->atom.type, type);
  sprintf(types, "%s.%.4s[%d]", parent_types, type, id);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)trak->atom.size);

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_TKHD:
        tkhd_dump(types, (tkhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_EDTS:
        edts_dump(types, (edts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MDIA:
        mdia_dump(types, (mdia_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_LOAD:
        load_dump(types, (load_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_dump(types, (udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TREF:
        tref_dump(types, (tref_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_dump(types, (clip_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MATT:
        matt_dump(types, (matt_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_IMAP:
        imap_dump(types, (imap_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_TRAK, atom->childs[i]->type);
	break;
    }
  }
}

int
trak_get_id(trak_t *trak)
{
  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  return trak->tkhd.track_id;
}

int
trak_set_id(trak_t *trak, int id)
{
  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  if (!trak || id <= 0)
    return QTIME_ERROR;
  trak->tkhd.track_id = id;
  return trak->tkhd.track_id;
}

uint32_t
trak_get_media_time_scale(trak_t *trak)
{
  mdia_t *mdia;
  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  mdia = (mdia_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_MDIA);
  return mdia_get_media_time_scale(mdia);
}

uint32_t
trak_get_media_duration(trak_t *trak)
{
  mdia_t *mdia;
  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  mdia = (mdia_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_MDIA);
  return mdia_get_media_duration(mdia);
}

int
trak_set_movie_duration(trak_t *trak, uint32_t movie_duration)
{
  tkhd_t *tkhd;
  elst_t *elst;

  qtime_error_type_check_v(QTIME_TYPE_TRAK, trak->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  tkhd = (tkhd_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_TKHD);
  tkhd_set_duration(tkhd, movie_duration);
  elst = (elst_t*)atom_find_recursive((atom_t*)trak, QTIME_TYPE_ELST);
  elst_add_table(elst, movie_duration, 0, 1.);
  return QTIME_OK;
}

