/* qtime_track.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "qtime_track.h"
#include "qtime_media.h"
#include "qtime_sample.h"
#include "qtime_sample_desc.h"
#include "atom.h"
#include "trak.h"
#include "tkhd.h"
#include "udta.h"
#include "edts.h"
#include "tref.h"
#include "load.h"
//#include "clip.h"
//#include "imap.h"
//#include "matt.h"


void
qtime_track_init(qtime_track_t *qtrk)
{
  qtrk->trak = NULL;
  qtrk->tkhd = NULL;
  qtrk->edts = NULL;

  qtime_media_init(&qtrk->qtmd);

  qtrk->qtio = NULL;
  qtrk->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  qtrk->qtsp = NULL;

  qtrk->sample_count = 0;
  qtrk->sample_duration = 0;
  qtrk->sample_description_id = 1;
  qtrk->compressor[0] = qtrk->compressor[1] = qtrk->compressor[2] = 
  qtrk->compressor[3] = qtrk->compressor[4] = '\0';
}

void
qtime_track_clean(qtime_track_t *qtrk)
{
  qtime_media_clean(&qtrk->qtmd);
  qtime_track_init(qtrk);
}

qtime_track_t*
qtime_track_new(void)
{
  qtime_track_t *qtrk;
  qtrk = (qtime_track_t*)qtime_malloc(sizeof(qtime_track_t));
  if (!qtrk) return NULL;
  qtime_track_init(qtrk);
  return qtrk;
}

void
qtime_track_delete(qtime_track_t *qtrk)
{
  qtime_track_clean(qtrk);
  qtime_free(qtrk);
}

qtime_track_t*
qtime_track_create(qtime_track_t *qtrk_ptr, trak_t *trak)
{
  qtime_track_t *qtrk;
  mdia_t *mdia;

  if (trak == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'trak' atom is NULL, cannot create qtime_track.\n");
    return NULL;
  }

  if (qtrk_ptr) {
    qtrk = qtrk_ptr;
    qtime_track_clean(qtrk);
  } else {
    if ((qtrk = qtime_track_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  qtrk->trak = trak;
  qtrk->tkhd = (tkhd_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_TKHD);
  qtrk->edts = (edts_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_EDTS);

  mdia = (mdia_t*)atom_find_child((atom_t*)trak, QTIME_TYPE_MDIA);
  qtime_media_create(&qtrk->qtmd, mdia);
  qtrk->qtsp = qtime_media_get_qtime_sample(&qtrk->qtmd);
  qtrk->media_type = qtime_media_get_media_type(&qtrk->qtmd);

  return qtrk;
}

int
qtime_track_set_moov_io(qtime_track_t *qtrk, qtime_io_t *mvio)
{
  return qtime_media_set_moov_io(&qtrk->qtmd, mvio);
}

int
qtime_track_get_id(qtime_track_t *qtrk)
{
  return trak_get_id(qtrk->trak);
}

int
qtime_track_set_id(qtime_track_t *qtrk, int id)
{
  return trak_set_id(qtrk->trak, id);
}

int
qtime_track_set_movie_duration(qtime_track_t *qtrk, uint32_t duration)
{
  qtrk->tkhd->duration = duration;
  return elst_add_table(&qtrk->edts->elst, duration, 0, 1.);
}

uint32_t
qtime_track_get_media_time_scale(qtime_track_t *qtrk)
{
  return qtime_media_get_time_scale(&qtrk->qtmd);
}

int
qtime_track_set_media_time_scale(qtime_track_t *qtrk, uint32_t time_scale)
{
  return qtime_media_set_time_scale(&qtrk->qtmd, time_scale);
}

uint32_t
qtime_track_get_media_duration(qtime_track_t *qtrk)
{
  return qtime_media_get_max_duration(&qtrk->qtmd);
}

uint32_t
qtime_track_get_media_type(qtime_track_t *qtrk)
{
  qtime_error_media_check_v(QTIME_TYPE_QTRK, qtrk->media_type, qtime_media_get_media_type(&qtrk->qtmd), QTIME_MEDIA_TYPE_UNKNOWN)

  if (qtrk->media_type != QTIME_MEDIA_TYPE_UNKNOWN)
    return qtrk->media_type;
  qtrk->media_type = qtime_media_get_media_type(&qtrk->qtmd);
  return qtrk->media_type;
}

int
qtime_track_set_media_type(qtime_track_t *qtrk, uint32_t media_type)
{
  qtime_error_media_check_v(QTIME_TYPE_QTRK, qtrk->media_type, media_type, QTIME_ERROR_INVALID_MEDIA)

  qtrk->media_type = media_type;
  switch (media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      qtrk->tkhd->volume = qtime_float_to_fixed32(0);
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      qtrk->tkhd->track_width = qtime_float_to_fixed32(0);
      qtrk->tkhd->track_height = qtime_float_to_fixed32(0);
      qtrk->tkhd->volume = qtime_float_to_fixed16(1);
      break;
    default:
      fprintf(stderr, "QTIME_ERROR: qtime_track_set_media_type: unsupported media_type.\n");
      return QTIME_ERROR;
      break;
  }
  if (qtime_media_set_media_type(&qtrk->qtmd, media_type) != QTIME_OK)
    return QTIME_ERROR;

  return QTIME_OK;
}

char*
qtime_track_get_compressor(qtime_track_t *qtrk)
{
  qtime_media_get_compressor(&qtrk->qtmd, qtrk->compressor);
  return qtrk->compressor;
}

// video

uint32_t
qtime_track_get_video_position(qtime_track_t *qtrk)
{
  return qtime_media_get_count(&qtrk->qtmd);
}

uint32_t
qtime_track_set_video_position(qtime_track_t *qtrk, uint32_t position)
{
  return qtime_media_set_count(&qtrk->qtmd, position);
}

uint32_t
qtime_track_get_video_frames(qtime_track_t *qtrk)
{
  return qtime_media_get_max_count(&qtrk->qtmd);
}

int
qtime_track_get_video_width(qtime_track_t *qtrk)
{
  int width;
  width = qtime_media_get_video_width(&qtrk->qtmd);
  return width;
}

int
qtime_track_get_video_height(qtime_track_t *qtrk)
{
  int height;
  height = qtime_media_get_video_height(&qtrk->qtmd);
  return height;
}

double
qtime_track_get_video_fps(qtime_track_t *qtrk)
{
  return qtime_media_get_video_fps(&qtrk->qtmd);
}

int
qtime_track_set_video_fps(qtime_track_t *qtrk, double fps)
{
  return qtime_media_set_video_fps(&qtrk->qtmd, fps);
}

int
qtime_track_video_is_keyframe(qtime_track_t *qtrk, uint32_t count)
{
  return qtime_media_video_is_keyframe(&qtrk->qtmd, count);
}

uint32_t
qtime_track_video_next_keyframe(qtime_track_t *qtrk, uint32_t count)
{
  return qtime_media_video_next_keyframe(&qtrk->qtmd, count);
}

uint32_t
qtime_track_video_prev_keyframe(qtime_track_t *qtrk, uint32_t count)
{
  return qtime_media_video_prev_keyframe(&qtrk->qtmd, count);
}

int
qtime_track_read_video(qtime_track_t *qtrk, uint8_t *buf, int *keyframe)
{
  return qtime_media_read_video(&qtrk->qtmd, buf, keyframe);
}

int
qtime_track_write_video(qtime_track_t *qtrk, uint8_t *buf, int size, int keyframe)
{
  return qtime_media_write_video(&qtrk->qtmd, buf, size, keyframe);
}

int
qtime_track_set_video(qtime_track_t *qtrk, int width, int height, double fps, uint8_t *compressor)
{
  if (qtime_track_set_media_type(qtrk, QTIME_MEDIA_TYPE_VIDEO) != QTIME_OK) {
    return QTIME_ERROR;
  }

  if (qtime_media_set_video(&qtrk->qtmd, width, height, compressor) < 0) {
    fprintf(stderr, "QTIME_ERROR: qtime_media_set_video failed.\n");
    return QTIME_ERROR;
  }

  if (qtime_media_set_video_fps(&qtrk->qtmd, fps) != QTIME_OK) {
    fprintf(stderr, "QTIME_ERROR: qtime_media_set_fps failed.\n");
    return QTIME_ERROR;
  }

  memcpy(qtrk->compressor, compressor, 4);
  qtrk->compressor[5] = '\0';
  qtrk->tkhd->track_width = qtime_float_to_fixed32((float)width);
  qtrk->tkhd->track_height = qtime_float_to_fixed32((float)height);

  return QTIME_OK;
}

// audio

int
qtime_track_get_audio_channels(qtime_track_t *qtrk)
{
  int channels;
  channels = qtime_media_get_sound_channels(&qtrk->qtmd);
  return channels;
}

int
qtime_track_get_audio_bits(qtime_track_t *qtrk)
{
  int bits;
  bits = qtime_media_get_sound_bits(&qtrk->qtmd);
  return bits;
}

int
qtime_track_get_audio_rate(qtime_track_t *qtrk)
{
  int rate;
  rate = qtime_media_get_sound_rate(&qtrk->qtmd);
  return rate;
}

int
qtime_track_set_audio(qtime_track_t *qtrk, int channels, int rate, int bits, uint8_t *compressor)
{
  if (qtrk->media_type != QTIME_MEDIA_TYPE_SOUND &&
      qtime_track_set_media_type(qtrk, QTIME_MEDIA_TYPE_SOUND) != QTIME_OK) {
    fprintf(stderr, "QTIME_ERROR: qtime_track_set_audio: set media type failed.\n");
    return QTIME_ERROR;
  }

  if (qtime_media_set_sound(&qtrk->qtmd, channels, rate, bits, compressor) < 0){
    fprintf(stderr, "QTIME_ERROR: qtime_media_set_sound failed.\n");
    return QTIME_ERROR;
  }

  qtrk->tkhd->volume = qtime_float_to_fixed16(1);

  return QTIME_OK;
}

int
qtime_track_set_audio_ext(qtime_track_t *qtrk, int8_t version, int16_t compression_id, uint32_t samples_per_packet, uint32_t bytes_per_packet, uint32_t bytes_per_frame, uint32_t bytes_per_sample) {
  return qtime_media_set_sound_ext(&qtrk->qtmd, version, compression_id, samples_per_packet, bytes_per_packet, bytes_per_frame, bytes_per_sample);
}

int
qtime_track_get_audio_ext(qtime_track_t *qtrk, int8_t *version, int16_t *compression_id, uint32_t *samples_per_packet, uint32_t *bytes_per_packet, uint32_t *bytes_per_frame, uint32_t *bytes_per_sample) {
  return qtime_media_get_sound_ext(&qtrk->qtmd, version, compression_id, samples_per_packet, bytes_per_packet, bytes_per_frame, bytes_per_sample);
}

uint32_t
qtime_track_get_audio_samples(qtime_track_t *qtrk)
{
  if (qtrk->media_type != QTIME_MEDIA_TYPE_SOUND)
    return 0;
  return qtime_media_get_max_duration(&qtrk->qtmd);
}

uint32_t
qtime_track_get_audio_position(qtime_track_t *qtrk)
{
  return qtime_media_get_duration(&qtrk->qtmd);
}

uint32_t
qtime_track_set_audio_position(qtime_track_t *qtrk, uint32_t position)
{
  return qtime_media_set_duration(&qtrk->qtmd, position);
}

int
qtime_track_read_audio(qtime_track_t *qtrk, uint8_t *buf,int *num_samples)
{
  return qtime_media_read_audio(&qtrk->qtmd, buf, num_samples);
}

int
qtime_track_write_audio(qtime_track_t *qtrk, uint8_t *data, int data_size, int number_of_samples)
{
  return qtime_media_write_audio(&qtrk->qtmd, data, data_size, number_of_samples);
}

