/* qtime_io.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_be.h"

void
qtime_io_init(qtime_io_t *qtio)
{
  qtio->name = NULL;
  qtio->fd = -1;
  qtio->mode = 0;
  qtio->end_of_file = 0;
  qtio->end_offset = 0;
}

void
qtime_io_clean(qtime_io_t *qtio)
{
  if (!qtio)          return;
  if (qtio->name)     qtime_free((void*)qtio->name);
  if (qtio->fd != -1) close(qtio->fd);
  qtime_io_init(qtio);
}

qtime_io_t*
qtime_io_new(void)
{
  qtime_io_t *qtio;

  qtio = (qtime_io_t*) qtime_malloc(sizeof(qtime_io_t));
  if (!qtio) {
    return NULL;
  }
  qtime_io_init(qtio);
  return qtio;
}

void
qtime_io_delete(qtime_io_t *qtio)
{
  qtime_io_clean(qtio);
  qtime_free(qtio);
}

int
qtime_io_open(qtime_io_t* qtio, const char *file_name, int mode)
{
  int name_length;

  if (!file_name || file_name[0] == '\0') {
    qtime_error_debug_info(QTIME_ERROR_EMPTY_FILENAME);
    qtime_error_empty_filename();
    return QTIME_ERROR_OPEN_FAIL;
  }

  name_length = strlen(file_name);
  if ((qtio->name = qtime_malloc(name_length+1)) == NULL)
    return QTIME_ERROR_MEMORY;
  ((char*)qtio->name)[name_length] = '\0';
  strncpy((char*)qtio->name, (char*)file_name, name_length);

  if (mode == QTIME_IO_MODE_READ) {
    if ((qtio->fd = open(qtio->name, O_RDONLY)) < 0) {
      qtime_error_debug_info(QTIME_ERROR_OPEN_FAIL);
      qtime_error_open_fail(qtio->name);
      qtime_io_clean(qtio);
      return QTIME_ERROR_OPEN_FAIL;
    }
  } else if (mode == QTIME_IO_MODE_WRITE) {
    if ((qtio->fd = open(qtio->name, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
      qtime_error_debug_info(QTIME_ERROR_OPEN_FAIL);
      qtime_error_open_fail(qtio->name);
      qtime_io_clean(qtio);
      return QTIME_ERROR_OPEN_FAIL;
    }
  }
#if 0
  else if (mode == (QTIME_IO_MODE_WRITE|QTIME_IO_MODE_READ)) {
    if ((qtio->fd = open(qtio->name, O_RDWR|O_CREAT, 0644)) < -1) {
      qtime_error_debug_info(QTIME_ERROR_OPEN_FAIL);
      qtime_error_open_fail(qtio->name);
      qtime_io_clean(qtio);
      return QTIME_ERROR_OPEN_FAIL;
    }
  }
#else
  else {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: unknown open mode.\n");
    return QTIME_ERROR;
  }
#endif
  qtio->mode = mode;
  qtio->end_offset = qtime_io_end_offset(qtio);

  return QTIME_OK;
}

int
qtime_io_close(qtime_io_t *qtio)
{
  qtime_io_clean(qtio);
  return QTIME_OK;
}

int
qtime_io_read(qtime_io_t *qtio, void *buf, int size)
{
  ssize_t result = 0, r;

  while (size > 0) {
    if ((r = read(qtio->fd, buf, size)) == 0) {
      qtio->end_of_file = 1;
      break;
    }
    if (r < 0)
      return result > 0 ? result : r;
    size -= r;
    result += r;
    buf += r;
  }
  return result;
}

int
qtime_io_write(qtime_io_t *qtio, void *buf, int size)
{
  return write(qtio->fd, buf, size);
} 

int
qtime_io_read16(qtime_io_t *qtio, uint16_t *p16)
{
#ifndef WORDS_BIGENDIAN
  uint16_t ui16;
  if (qtime_io_read(qtio, &ui16, sizeof(uint16_t)) != sizeof(uint16_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
  QTIME_BE16(ui16, *p16);
#else
  if (qtime_io_read(qtio, p16, sizeof(uint16_t)) != sizeof(uint16_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
#endif
  return sizeof(uint16_t);
}

int
qtime_io_read32(qtime_io_t *qtio, uint32_t *p32)
{
#ifndef WORDS_BIGENDIAN
  uint32_t ui32;
  if (qtime_io_read(qtio, &ui32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
  QTIME_BE32(ui32, *p32);
#else /* __BIG_ENDIAN */
  if (qtime_io_read(qtio, p32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint32_t);
}

int
qtime_io_read64(qtime_io_t *qtio, uint64_t *p64)
{
#ifndef WORDS_BIGENDIAN
  uint64_t ui64;
  if (qtime_io_read(qtio, &ui64, sizeof(uint64_t)) != sizeof(uint64_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
  QTIME_BE64(ui64, *p64);
#else /* __BIG_ENDIAN */
  if (qtime_io_read(qtio, p64, sizeof(uint64_t)) != sizeof(uint64_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint64_t);
}

int
qtime_io_read_type(qtime_io_t *qtio, uint32_t *p32)
{
#ifndef WORDS_BIGENDIAN
  uint32_t ui32;
  if (qtime_io_read(qtio, &ui32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
  QTIME_BE32(ui32, *p32);
#else /* __BIG_ENDIAN */
  if (qtime_io_read(qtio, p32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_READ);
    qtime_error_read();
    return QTIME_ERROR_READ;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint32_t);
}

int
qtime_io_write16(qtime_io_t *qtio, uint16_t *p16)
{
#ifndef WORDS_BIGENDIAN
  uint16_t ui16;
  QTIME_BE16(*p16, ui16);
  if (qtime_io_write(qtio, &ui16, sizeof(uint16_t)) != sizeof(uint16_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#else /* __BIG_ENDIAN */
  if (qtime_io_write(qtio, p16, sizeof(uint16_t)) != sizeof(uint16_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint16_t);
}

int
qtime_io_write32(qtime_io_t *qtio, uint32_t *p32)
{
#ifndef WORDS_BIGENDIAN
  uint32_t ui32;
  QTIME_BE32(*p32, ui32);
  if (qtime_io_write(qtio, &ui32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#else /* __BIG_ENDIAN */
  if (qtime_io_write(qtio, p32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint32_t);
}

int
qtime_io_write64(qtime_io_t *qtio, uint64_t *p64)
{
#ifndef WORDS_BIGENDIAN
  uint64_t ui64;
  QTIME_BE64(*p64, ui64);
  if (qtime_io_write(qtio, &ui64, sizeof(uint64_t)) != sizeof(uint64_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#else /* __BIG_ENDIAN */
  if (qtime_io_write(qtio, p64, sizeof(uint64_t)) != sizeof(uint64_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint64_t);
}

int
qtime_io_write_type(qtime_io_t *qtio, uint32_t *p32)
{
#ifndef WORDS_BIGENDIAN
  uint32_t ui32;
  QTIME_BE32(*p32, ui32);
  if (qtime_io_write(qtio, &ui32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#else /* __BIG_ENDIAN */
  if (qtime_io_write(qtio, p32, sizeof(uint32_t)) != sizeof(uint32_t)) {
    qtime_error_debug_info(QTIME_ERROR_WRITE);
    qtime_error_wirte();
    return QTIME_ERROR_WRITE;
  }
#endif /* WORDS_BIGENDIAN */
  return sizeof(uint32_t);
}

int64_t
qtime_io_get_offset(qtime_io_t *qtio)
{
  int64_t offset;

  offset = lseek(qtio->fd, 0, SEEK_CUR);
  if (offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_LSEEK);
    qtime_error_lseek();
  }
  return offset;
}

int64_t
qtime_io_set_offset(qtime_io_t *qtio, int64_t offset)
{
  offset = lseek(qtio->fd, offset, SEEK_SET);
  if (offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_LSEEK);
    qtime_error_lseek();
  }
  return offset;
}

int64_t
qtime_io_end_offset(qtime_io_t *qtio)
{
  int64_t offset;
  int64_t end_offset;
  offset = lseek(qtio->fd, 0, SEEK_CUR);
  if (offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_LSEEK);
    qtime_error_lseek();
    return offset;
  }
  end_offset = lseek(qtio->fd, 0, SEEK_END);
  if (end_offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_LSEEK);
    qtime_error_lseek();
    return end_offset;
  }
  offset = lseek(qtio->fd, offset, SEEK_SET);
  if (offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_LSEEK);
    qtime_error_lseek();
    return offset;
  }
  return end_offset;
}

int
qtime_io_read_data(qtime_io_t *qtio, qtime_io_data_t *qtio_data)
{
  int size;
  qtio_data->start_offset = qtime_io_set_offset(qtio, qtio_data->start_offset);
  size = qtime_io_read(qtio, qtio_data->data, qtio_data->size);
  qtio_data->end_offset = qtime_io_get_offset(qtio);
  qtio_data->size = size;
  return size;
}

int
qtime_io_write_data(qtime_io_t *qtio, qtime_io_data_t *qtio_data)
{
  int size;
  qtio_data->start_offset = qtime_io_get_offset(qtio);
  size = qtime_io_write(qtio, qtio_data->data, qtio_data->size);
  qtio_data->end_offset = qtime_io_get_offset(qtio);
  qtio_data->size = size;
  return qtio_data->size;
}

