/* qtime_error.c */

#include <stdio.h>
#include <inttypes.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include "qtime_util.h"
#include "qtime_error.h"

static struct QTIME_ERROR_CODE_STRING {
  int error_code;
  const char *error_code_string;
} qtime_error_code_string[] = {
  { QTIME_OK, "QTIME_OK" },
  { QTIME_ERROR, "QTIME_ERROR" },
  { QTIME_ERROR_MEMORY, "QTIME_ERROR_MEMORY" },
  { QTIME_ERROR_TOO_MANY_ATOM, "QTIME_ERROR_TOO_MANY_ATOM" },
  { QTIME_ERROR_OPEN_FAIL, "QTIME_ERROR_OPEN_FAIL" },
  { QTIME_ERROR_EMPTY_FILENAME, "QTIME_ERROR_EMPTY_FILENAME" },
  { QTIME_ERROR_LSEEK, "QTIME_ERROR_LSEEK" },
  { QTIME_ERROR_READ, "QTIME_ERROR_READ" },
  { QTIME_ERROR_WRITE, "QTIME_ERROR_WRITE" },
  { QTIME_ERROR_ATOM_READ, "QTIME_ERROR_ATOM_READ" },
  { QTIME_ERROR_ATOM_WRITE, "QTIME_ERROR_ATOM_WRITE" },
  { QTIME_ERROR_UNKNOWN_ATOM, "QTIME_ERROR_UNKNOWN_ATOM" },
  { QTIME_ERROR_ILLEGAL_ATOM, "QTIME_ERROR_ILLEGAL_ATOM" },
  { QTIME_ERROR_ATOM_SIZE, "QTIME_ERROR_ATOM_SIZE" },
  { QTIME_ERROR_IO, "QTIME_ERROR_IO" },
  { QTIME_ERROR_INVALID_MEDIA, "QTIME_ERROR_INVALID_MEDIA" },
};

static int qtime_error_code_string_num = sizeof(qtime_error_code_string) / sizeof(struct QTIME_ERROR_CODE_STRING);

const char*
qtime_error_code_to_string(int error_code)
{
  int i;
  for (i = 0; i < qtime_error_code_string_num; i++)
    if (error_code == qtime_error_code_string[i].error_code)
      return qtime_error_code_string[i].error_code_string;
  return NULL;
}

#if 0
void
qtime_error_print(int error_code, ...)
{
  va_list ap;
  const char *mess = "QTIME_ERROR: ";
  char fcc[5];
  char fcc2[5];

  fcc[4]  = '\0';
  fcc2[4] = '\0';

  va_start(ap, error_code);

  switch (error_code) {
    uint32_t type, subtype;
    int32_t i32, size;
    char *cp;
    case QTIME_ERROR_MEMORY:
      i32 = va_arg(ap, int32_t);
      fprintf(stderr, "%smemory allocate error, demanded size %d.\n",mess,i32);
      break;
    case QTIME_ERROR_TOO_MANY_ATOM:
      type    = va_arg(ap, uint32_t);
      subtype = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      qtime_type_to_str(subtype, fcc2);
      fprintf(stderr, "%sTwo or more '%4s' atoms were found in '%4s' atom.\n", mess, fcc2, fcc);
      break;
    case QTIME_ERROR_OPEN_FAIL:
      cp = va_arg(ap, char *);
      fprintf(stderr, "%s%s open error.\n", mess, cp);
      perror("");
      break;
    case QTIME_ERROR_EMPTY_FILENAME:
      fprintf(stderr, "%sfile name is not specified.\n", mess);
      break;
    case QTIME_ERROR_LSEEK:
      perror(mess);
      break;
    case QTIME_ERROR_READ:
      perror(mess);
      break;
    case QTIME_ERROR_WRITE:
      perror(mess);
      break;
    case QTIME_ERROR_ATOM_READ:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom read failed.\n", mess, fcc);
      break;
    case QTIME_ERROR_ATOM_WRITE:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom write failed.\n", mess, fcc);
      break;
    case QTIME_ERROR_UNKNOWN_ATOM:
      type = va_arg(ap, uint32_t);
      subtype = va_arg(ap, uint32_t);
      size = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc2);
      qtime_type_to_str(subtype, fcc);
      fprintf(stderr, "%sunknown atom '%4s' (0x%02x%02x%02x%02x) is found in '%4s' atom, size %d.\n", mess, fcc, fcc[0], fcc[1], fcc[2], fcc[3], fcc2, size);
      break;
    case QTIME_ERROR_ILLEGAL_ATOM:
      type = va_arg(ap, uint32_t);
      subtype = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc2);
      qtime_type_to_str(subtype, fcc);
      fprintf(stderr, "%sillegal atom '%4s' (0x%02x%02x%02x%02x) is found in '%4s' atom.\n", mess, fcc, fcc[0], fcc[1], fcc[2], fcc[3], fcc2);
      break;
    case QTIME_ERROR_ATOM_SIZE:
      type = va_arg(ap, uint32_t);
      i32 = va_arg(ap, int32_t);
      size = va_arg(ap, int32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom size doesn't suit, %d, %d.\n", mess, fcc, i32, size);
      break;
    case QTIME_ERROR_IO:
      cp = va_arg(ap, char *);
      fprintf(stderr, "%s'%s' input or output error.\n", mess, cp);
      break;
    case QTIME_ERROR_INVALID_MEDIA:
      fprintf(stderr, "%s media type doesn't suit.\n", mess);
      break;
  }

  va_end(ap);
}

void
qtime_print_error(int error_code, const char *mess, ...)
{
  va_list ap;
  int len = strlen(mess);
  char buf[len+10];
  char fcc[5];

  if (!mess)
    mess = "";
  else
    sprintf(buf, "%s: ", mess);
  fcc[4] = '\0';

  va_start(ap, mess);

  switch (error_code) {
    uint32_t type;
    int32_t i32, size;
    char *cp;
    case QTIME_ERROR_MEMORY:
      i32 = va_arg(ap, int32_t);
      fprintf(stderr, "%smemory allocate error, required size %d.\n", buf, i32);
      break;
    case QTIME_ERROR_TOO_MANY_ATOM:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%sTwo or more '%4s' atoms were found, it skips.\n",
       	              buf, fcc);
      break;
    case QTIME_ERROR_OPEN_FAIL:
      cp = va_arg(ap, char *);
      fprintf(stderr, "%s%s open error.\n", buf, cp);
      perror("");
      break;
    case QTIME_ERROR_EMPTY_FILENAME:
      fprintf(stderr, "%sfile name is not specified.\n", buf);
      break;
    case QTIME_ERROR_LSEEK:
      perror(buf);
      break;
    case QTIME_ERROR_READ:
      perror(buf);
      break;
    case QTIME_ERROR_WRITE:
      perror(buf);
      break;
    case QTIME_ERROR_ATOM_READ:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom read failed.\n", buf, fcc);
      break;
    case QTIME_ERROR_ATOM_WRITE:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom write failed.\n", buf, fcc);
      break;
    case QTIME_ERROR_UNKNOWN_ATOM:
      type = va_arg(ap, uint32_t);
      size = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%sunknown atom '%4s' (0x%02x%02x%02x%02x) is found, it skips, size %d.\n", buf, fcc, fcc[0], fcc[1], fcc[2], fcc[3], size);
      break;
    case QTIME_ERROR_ILLEGAL_ATOM:
      type = va_arg(ap, uint32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%sillegal atom '%4s' (0x%02x%02x%02x%02x) is found.\n", buf, fcc, fcc[0], fcc[1], fcc[2], fcc[3]);
      break;
    case QTIME_ERROR_ATOM_SIZE:
      type = va_arg(ap, uint32_t);
      i32 = va_arg(ap, int32_t);
      size = va_arg(ap, int32_t);
      qtime_type_to_str(type, fcc);
      fprintf(stderr, "%s'%4s' atom size doesn't suit, %d, %d.\n", buf, fcc, i32, size);
      break;
    case QTIME_ERROR_IO:
      cp = va_arg(ap, char *);
      fprintf(stderr, "%s'%s' input or output error.\n", buf, cp);
      break;
    case QTIME_ERROR_INVALID_MEDIA:
      fprintf(stderr, "%s media type doesn't suit.\n", buf);
      break;
  }

  va_end(ap);
}
#endif

int
qtime_error_type_check_func(uint32_t type, uint32_t subtype, const char *file, const int line, const char *func)
{
  if (type != subtype) {
    fprintf(stderr, "QTIME_TYPE_CHECK:%s:%d:%s:%s\n", file, line, func,
	qtime_error_code_to_string(QTIME_ERROR_ILLEGAL_ATOM));
    qtime_error_illegal_atom(type, subtype);
    return 1;
  }
  return 0;
}

void
qtime_error_illegal_atom(uint32_t cur_type, uint32_t type)
{
  char cur_fcc[5];
  char fcc[5];
  qtime_type_to_str(cur_type, cur_fcc);
  qtime_type_to_str(type, fcc);
  fprintf(stderr, "QTIME_ERROR_ILLEGAL_ATOM: illegal atom '%4s' (0x%02x%02x%02x%02x) is found in '%4s' atom.\n", fcc, fcc[0], fcc[1], fcc[2], fcc[3], cur_fcc);
}

void
qtime_error_atom_read(uint32_t cur_type)
{
  char cur_fcc[5];
  qtime_type_to_str(cur_type, cur_fcc);
  fprintf(stderr, "QTIME_ERROR_ATOM_READ: '%4s' atom read failed.\n", cur_fcc);
}

void
qtime_error_atom_write(uint32_t cur_type)
{
  char cur_fcc[5];
  qtime_type_to_str(cur_type, cur_fcc);
  fprintf(stderr, "QTIME_ERROR_ATOM_WRITE: '%4s' atom write failed.\n",cur_fcc);
}

void
qtime_error_unknown_atom(uint32_t cur_type, uint32_t type, int size)
{
  char cur_fcc[5];
  char fcc[5];
  qtime_type_to_str(cur_type, cur_fcc);
  qtime_type_to_str(type, fcc);
  fprintf(stderr, "QTIME_ERROR_UNKNOWN_ATOM: unknown atom '%4s' (0x%02x%02x%02x%02x) is found in '%4s' atom, it skips, size %d.\n", fcc, fcc[0], fcc[1], fcc[2], fcc[3], cur_fcc, size);
}

void
qtime_error_too_many_atom(uint32_t cur_type, uint32_t type)
{
  char cur_fcc[5];
  char fcc[5];
  qtime_type_to_str(cur_type, cur_fcc);
  qtime_type_to_str(type, fcc);
  fprintf(stderr, "QTIME_ERROR_TOO_MANY_ATOM: Two or more '%4s' atoms were found in '%4s' atom.\n", fcc, cur_fcc);
}

void
qtime_error_invalid_media(uint32_t cur_type, uint32_t cur_media_type, uint32_t media_type)
{
  char fcc[5];
  char cmfcc[5];
  char mfcc[5];
  qtime_type_to_str(cur_type, fcc);
  qtime_type_to_str(cur_media_type, cmfcc);
  qtime_type_to_str(media_type, mfcc);
  fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: media type doesn't suit, '%4s', '%4s', in '%4s' atom.\n", cmfcc, mfcc, fcc);
}

void
qtime_error_atom_size(uint32_t cur_type, int32_t cur_size, int32_t size)
{
  char fcc[5];
  qtime_type_to_str(cur_type, fcc);
  fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: '%4s' atom size doesn't suit, %d, %d.\n", fcc, cur_size, size);
}

void
qtime_error_empty_filename(void)
{
  fprintf(stderr, "QTIME_ERROR: file name is not specified.\n");
}

void
qtime_error_io(const char *cp)
{
  fprintf(stderr, "QTIME_ERROR_IO: '%s' input or output error.\n", cp);
}

void
qtime_error_memory(int size)
{
  fprintf(stderr, "QTIME_ERROR_MEMORY: memory allocate error, requested size %d.\n", size);
}

void
qtime_error_lseek(void)
{
  perror("QTIME_ERROR_LSEEK");
}

void
qtime_error_open_fail(const char *name)
{
  fprintf(stderr, "QTIME_ERROR_OPEN_FAIL: %s open error.\n", name);
  perror("QTIME_ERROR_OPEN_FAIL");
}

void
qtime_error_read(void)
{
  perror("QTIME_ERROR_READ");
}

void
qtime_error_wirte(void)
{
  perror("QTIME_ERROR_WRITE");
}

