/* kmat.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "kmat.h"

void
kmat_init(kmat_t *kmat)
{
  kmat->atom.size = 0;
  kmat->atom.type = QTIME_TYPE_KMAT;
  kmat->version = 0;
  qtime_flags_set(kmat->flags, 0);
  kmat->matte_data_size = 0;
  kmat->matte_data= NULL;
}

void
kmat_clean(kmat_t *kmat)
{
  if (kmat->matte_data)
    qtime_free(kmat->matte_data);
  kmat_init(kmat);
}

kmat_t*
kmat_new(void)
{
  kmat_t *kmat;
  kmat = (kmat_t*)qtime_malloc(sizeof(kmat_t));
  if (!kmat)
    return NULL;
  kmat_init(kmat);
  return kmat;
}

void
kmat_delete(kmat_t *kmat)
{
  if (!kmat) return;
  if (kmat->matte_data)
    qtime_free(kmat->matte_data);
  qtime_free(kmat);
}

int
kmat_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, kmat_t *kmat)
{
  const char *func = "kmat_read_atom";

#ifdef DEBUG
  if (atom_head->type != QTIME_TYPE_KMAT) {
    qtime_print_error(QTIME_ERROR_ILLEGAL_ATOM, func, atom_head->type);
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif /* DEBUG */

  kmat->atom.size = atom_head->size;
  kmat->atom.type = QTIME_TYPE_KMAT;

  qtime_io_read(qtio, &kmat->version, 1);
  qtime_io_read(qtio, kmat->flags, 3);

  fprintf(stderr, "kmat_read_atom: not implemented.\n");

  return QTIME_OK;
}

int64_t
kmat_calc_size(kmat_t *kmat)
{
  int64_t size = 0;

  size += sizeof(KMAT_PROP);
  size += 8;
  kmat->atom.size = size;
  return size;
}

int
kmat_write_atom(qtime_io_t *qtio, kmat_t *kmat)
{
  atom_head_t atom_head;

  kmat->atom.size = kmat_calc_size(kmat);
  atom_head.size = kmat->atom.size;
  atom_head.type = kmat->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &kmat->version, 1);
  qtime_io_write(qtio, kmat->flags, 3);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
kmat_dump(const char *parent_types, kmat_t *kmat)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint32_t flags;

  sprintf(types, "%s.%.4s", parent_types, (char*)&kmat->atom.type);
  qtime_flags_get(kmat->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, kmat->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, kmat->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
}

