/* gmhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "gmhd.h"

void
gmhd_init(gmhd_t *gmhd)
{
  atom_init(&gmhd->atom);
  gmhd->atom.type = QTIME_TYPE_GMHD;
}

void
gmhd_clean(gmhd_t *gmhd)
{
  qtime_error_type_check(QTIME_TYPE_GMHD, gmhd->atom.type)
  gmhd_init(gmhd);
}

gmhd_t*
gmhd_new(void)
{
  gmhd_t *gmhd;
  gmhd = (gmhd_t*)qtime_malloc(sizeof(gmhd_t));
  if (!gmhd) return NULL;
  gmhd_init(gmhd);
  return gmhd;
}

void
gmhd_delete(gmhd_t *gmhd)
{
  qtime_error_type_check(QTIME_TYPE_GMHD, gmhd->atom.type)
  qtime_free(gmhd);
}

gmhd_t*
gmhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, gmhd_t *gmhd_ptr)
{
  gmhd_t *gmhd;

#ifndef NDEBUG
  qtime_io_t *io = NULL;
  io = qtio;

  if (atom_head->media_type == QTIME_MEDIA_TYPE_VIDEO ||
      atom_head->media_type == QTIME_MEDIA_TYPE_SOUND) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    qtime_error_invalid_media(QTIME_TYPE_GMHD, QTIME_MEDIA_TYPE_BASE, atom_head->media_type);
    return NULL;
  }
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_GMHD, atom_head->type, NULL)
  if (gmhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_GMHD, gmhd_ptr->atom.type, NULL)
    gmhd = gmhd_ptr;
    gmhd_clean(gmhd);
  } else {
    if ((gmhd = gmhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  gmhd->atom.size = atom_head->size;
  gmhd->atom.type = QTIME_TYPE_GMHD;
  gmhd->atom.parent = atom_head->parent;

  atom_head->media_type = QTIME_MEDIA_TYPE_BASE;

  return gmhd;
}

gmhd_t*
gmhd_create(gmhd_t *gmhd_ptr)
{
  gmhd_t *gmhd;

  if (gmhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_GMHD, gmhd_ptr->atom.type, NULL)
    gmhd = gmhd_ptr;
    gmhd_clean(gmhd);
  } else {
    if ((gmhd = gmhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return gmhd;
}

int64_t
gmhd_calc_size(gmhd_t *gmhd)
{
  int64_t size = 0;
  qtime_error_type_check_v(QTIME_TYPE_GMHD, gmhd->atom.type, 0)
  size = 8;
  gmhd->atom.size = size;
  return size;
}

int
gmhd_write_atom(qtime_io_t *qtio, gmhd_t *gmhd)
{
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_GMHD, gmhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = gmhd->atom.size;
  atom_head.type = gmhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_GMHD);
    return QTIME_ERROR_ATOM_WRITE;
  }
  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_GMHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
gmhd_dump(const char *parent_types, gmhd_t *gmhd)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(gmhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)gmhd->atom.size);
}

