/* vcodec.c */

#include <stdio.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "csp.h"
#include "vcodec.h"
#ifdef HAVE_LIBXVIDCORE
#include "vc_xvid.h"
#endif /* HAVE_LIBXVIDCORE */
#include "vc_raw.h"
#ifdef HAVE_RTJPEG
#include "vc_rtjpeg.h"
#endif /* HAVE_RTJPEG */

#ifdef HAVE_LIBAVCODEC
#include "vc_libavcodec.h"
#endif /* HAVE_LIBAVCODEC */

typedef struct {
  unsigned long codec;
  VCODEC_FUNCS *funcs;
  const char *name;
} VCODEC_ENTRY;

static VCODEC_ENTRY vcodec_entry[] = {
#ifdef HAVE_LIBXVIDCORE
  { VCODEC_XVID, &vc_xvid_funcs, "XVID" },
#endif /* HAVE_LIBXVIDCORE */
  { VCODEC_YV12, &vc_yv12_funcs, "YV12" },
  { VCODEC_RGB24, &vc_rgb24_funcs, "RGB24" },
#ifdef HAVE_RTJPEG
  { VCODEC_RTJPEG, &vc_rtjpeg_funcs, "RTJPEG" },
#endif /* HAVE_RTJPEG */
#ifdef HAVE_LIBAVCODEC
  { VCODEC_MPEG1, &vc_libavcodec_funcs, "MPEG1" },
  { VCODEC_H263, &vc_libavcodec_funcs, "H263" },
  { VCODEC_H263P, &vc_libavcodec_funcs, "H263P" },
  { VCODEC_MJPEG, &vc_libavcodec_funcs, "MJPEG" },
  { VCODEC_MPEG4, &vc_libavcodec_funcs, "MPEG4" },
  { VCODEC_MSMPEG4V1, &vc_libavcodec_funcs, "MSMPEG4V1" },
  { VCODEC_MSMPEG4V2, &vc_libavcodec_funcs, "MSMPEG4V2" },
  { VCODEC_MSMPEG4V3, &vc_libavcodec_funcs, "MSMPEG4V3" },
  { VCODEC_WMV1, &vc_libavcodec_funcs, "WMV1" },
  { VCODEC_WMV2, &vc_libavcodec_funcs, "WMV2" },
  { VCODEC_HUFFYUV, &vc_libavcodec_funcs, "HUFFYUV" },
#endif /* HAVE_LIBAVCODEC */
  { 0, NULL, NULL },
};

int vcodec_entry_num = sizeof(vcodec_entry)/sizeof(VCODEC_ENTRY)-1;

static VCODEC vcodec_enc;
static VCODEC vcodec_dec;

int (*vcodec_encode) (unsigned char *pic_data, unsigned char *stream_buf,
                       int stream_buf_size, unsigned int *flag);

int (*vcodec_decode) (unsigned char *pic_data, unsigned char *stream,
                       int stream_length, unsigned int *flag);

const char*
vcodec_get_codec_name(uint32_t codec)
{
  int i;
  int ret;

  for (i = ret = 0; i < vcodec_entry_num; i++) {
    if (codec == vcodec_entry[i].codec) {
      return vcodec_entry[i].name;
    }
  }

  return "UNKNOWN";
}

uint32_t
vcodec_codec_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;
  uint32_t cap;

  for (i = 0; i < vcodec_entry_num; i++) {
    if ((cap = vcodec_entry[i].funcs->codec_cap(codec, cap_flag)) != VCODEC_CAP_NONE)
      return cap;
  }
  return VCODEC_CAP_NONE;
}

const char*
vcodec_codec_to_fourcc_str(uint32_t codec)
{
  int i;
  const char *str;

  for (i = 0; i < vcodec_entry_num; i++) {
    if ((str = vcodec_entry[i].funcs->codec_to_fourcc_str(codec)) != NULL)
      return str;
  }
  return NULL;
}

uint32_t
vcodec_fourcc_str_to_codec(const char *fourcc_str)
{
  int i;
  uint32_t codec;

  for (i = 0; i < vcodec_entry_num; i++) {
    if ((codec = vcodec_entry[i].funcs->fourcc_str_to_codec(fourcc_str)) !=
       	VCODEC_UNKNOWN)
      return codec;
  }
  return VCODEC_UNKNOWN;
}

/* encode functions */

int
vcodec_encode_set_codec(uint32_t codec)
{
  int i;
  int ret;

  for (i = ret = 0; i < vcodec_entry_num; i++) {
    if (codec == vcodec_entry[i].codec) {
      vcodec_enc.funcs = vcodec_entry[i].funcs;
      vcodec_enc.codec_idx = i;
      vcodec_enc.codec = codec;
      break;
    }
  }
  if (i >= vcodec_entry_num) {
    char fcc[5];
    *((unsigned long*)fcc) = codec;
    fcc[4] = 0;
    fprintf(stderr, "vcodec_encode_set_codec: cannot initialize video codec %s.\n", fcc);
    return VCODEC_FAIL;
  }

  return VCODEC_OK;
}

int
vcodec_encode_csp_cap(int in_csp)
{
  if (!vcodec_enc.funcs->encode_csp_cap) {
    return CSP_UNKNOWN;
  }
  return vcodec_enc.funcs->encode_csp_cap(in_csp);
}

int
vcodec_encode_get_out_csp(void)
{
  return vcodec_enc.out_csp;
}

int
vcodec_encode_reset(void)
{
  vcodec_enc.funcs->encode_quit();
  return vcodec_enc.funcs->encode_init(&vcodec_enc);
}

int
vcodec_encode_init(unsigned long codec, int width, int height, double fps, int in_csp)
{
  int csp_cap;
  int need_config = 0;
  static int first = 1;

  if (first) {
    vcodec_enc.codec = VCODEC_UNKNOWN;
    vcodec_enc.width = 0;
    vcodec_enc.height = 0;
    vcodec_enc.fps = 0;
    vcodec_enc.in_csp = CSP_UNKNOWN;
    vcodec_enc.out_csp = CSP_UNKNOWN;
    vcodec_enc.codec_idx = -1;
    vcodec_enc.funcs = NULL;
    first = 0;
    need_config = 1;
  }

  if (!vcodec_encode)
    need_config = 1;
  else if (!vcodec_enc.funcs)
    need_config = 1;
  else if (vcodec_enc.codec != codec)
    need_config = 1;
  else if (vcodec_enc.width != width)
    need_config = 1;
  else if (vcodec_enc.height != height)
    need_config = 1;
  else if (vcodec_enc.fps != fps)
    need_config = 1;
  else if (vcodec_enc.in_csp != in_csp)
    need_config = 1;

  if (!need_config) {
    return VCODEC_OK;
  }

  vcodec_enc.codec = codec;
  vcodec_enc.width = width;
  vcodec_enc.height = height;
  vcodec_enc.fps = fps;
  vcodec_enc.in_csp = in_csp;
  vcodec_enc.out_csp = CSP_UNKNOWN;
  if (vcodec_encode_set_codec(codec) != VCODEC_OK) {
    fprintf(stderr, "vcodec_encode_init: codec invalid.\n");
    return VCODEC_FAIL;
  }
  csp_cap = vcodec_enc.funcs->encode_csp_cap(in_csp);
  if ((in_csp != CSP_UNKNOWN && csp_cap != in_csp) || csp_cap == CSP_UNKNOWN) {
    fprintf(stderr, "vcodec_encode_init: csp invalid.\n");
    return VCODEC_FAIL;
  }
  vcodec_enc.funcs->encode_init(&vcodec_enc);
  vcodec_encode = vcodec_enc.funcs->encode;

  vcodec_enc.out_csp = CSP_XVID;

  return VCODEC_OK;
}

int
vcodec_encode_quit(void)
{
  int ret = VCODEC_OK;
  if (vcodec_enc.funcs && vcodec_enc.funcs->encode_quit)
    ret = vcodec_enc.funcs->encode_quit ();
  vcodec_enc.funcs = NULL;
  vcodec_encode = NULL;
  return ret;
}

void
vcodec_encode_print_param(void)
{
  fprintf(stdout, "VIDEO CODEC:             %s\n", vcodec_get_codec_name(vcodec_enc.codec));
  if (vcodec_enc.funcs && vcodec_enc.funcs->encode_print_param)
    vcodec_enc.funcs->encode_print_param();
}

/* decode functions */

int
vcodec_decode_set_codec(uint32_t codec)
{
  int i;
  int ret;

  for (i = ret = 0; i < vcodec_entry_num; i++) {
    if (codec == vcodec_entry[i].codec) {
      vcodec_dec.funcs = vcodec_entry[i].funcs;
      vcodec_dec.codec_idx = i;
      vcodec_dec.codec = codec;
      break;
    }
  }
  if (i >= vcodec_entry_num) {
    char fcc[5];
    *((unsigned long*)fcc) = codec;
    fcc[4] = 0;
    fprintf(stderr, "vcodec_decode_set_codec: cannot initialize video codec %s.\n", fcc);
    return VCODEC_FAIL;
  }

  return VCODEC_OK;
}

int
vcodec_decode_csp_cap(int out_csp)
{
  if (!vcodec_dec.funcs->decode_csp_cap) {
    return CSP_UNKNOWN;
  }
  return vcodec_dec.funcs->decode_csp_cap(out_csp);
}

int
vcodec_decode_get_out_csp(void)
{
  return vcodec_dec.out_csp;
}

int
vcodec_decode_reset(void)
{
  vcodec_dec.funcs->decode_quit();
  return vcodec_dec.funcs->decode_init(&vcodec_dec);
}

int
vcodec_decode_init(unsigned long codec, int width, int height, int out_csp)
{
  int csp_cap;
  int need_config = 0;
  static int first = 1;

  if (first) {
    vcodec_dec.codec = VCODEC_UNKNOWN;
    vcodec_dec.width = 0;
    vcodec_dec.height = 0;
    vcodec_dec.fps = 0;
    vcodec_dec.in_csp = CSP_UNKNOWN;
    vcodec_dec.out_csp = CSP_UNKNOWN;
    vcodec_dec.codec_idx = -1;
    vcodec_dec.funcs = NULL;
    first = 0;
    need_config = 1;
  }

  if (!vcodec_decode)
    need_config = 1;
  else if (!vcodec_dec.funcs)
    need_config = 1;
  else if (vcodec_dec.codec != codec)
    need_config = 1;
  else if (vcodec_dec.width != width)
    need_config = 1;
  else if (vcodec_dec.height != height)
    need_config = 1;
  else if (vcodec_dec.out_csp != out_csp)
    need_config = 1;

  if (!need_config) {
    return VCODEC_OK;
  }

  vcodec_dec.codec = codec;
  vcodec_dec.width = width;
  vcodec_dec.height = height;
  vcodec_dec.out_csp = out_csp;
  vcodec_dec.in_csp = CSP_XVID;
  if (vcodec_decode_set_codec(codec) != VCODEC_OK) {
    fprintf(stderr, "vcodec_decode_init: codec invalid.\n");
    return VCODEC_FAIL;
  }
  csp_cap = vcodec_dec.funcs->decode_csp_cap(out_csp);
  if ((out_csp != CSP_UNKNOWN && csp_cap != out_csp) || csp_cap == CSP_UNKNOWN){
    fprintf(stderr, "vcodec_decode_init: csp invalid.\n");
    return VCODEC_FAIL;
  }
  vcodec_dec.funcs->decode_init(&vcodec_dec);
  vcodec_decode = vcodec_dec.funcs->decode;

  return VCODEC_OK;
}

int
vcodec_decode_quit(void)
{
  int ret = VCODEC_OK;
  if (vcodec_dec.funcs && vcodec_dec.funcs->decode_quit)
    ret = vcodec_dec.funcs->decode_quit ();
  vcodec_dec.funcs = NULL;
  vcodec_decode = NULL;
  return ret;
}

void
vcodec_decode_print_param(void)
{
  fprintf(stdout, "VIDEO DECODE CODEC:      %s\n", vcodec_get_codec_name(vcodec_dec.codec));
  if (vcodec_dec.funcs && vcodec_dec.funcs->decode_print_param)
    vcodec_dec.funcs->decode_print_param();
}

