/* vc_rtjpeg.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_RTJPEG


#include <stdio.h>
#include <string.h>

#include "util.h"
#include "csp.h"
#include "vcodec.h"
#include "vc_rtjpeg.h"

#include "RTjpeg.h"

typedef struct {
  unsigned long codec;
  int width;
  int height;
  int in_csp;
  int out_csp;
  int quality;

  int data_size;
  int u_offset;
  int v_offset;
  RTjpeg_t *rtj;
} RTJPEG_PARAM;


static RTJPEG_PARAM vc_rtj_enc;
static RTJPEG_PARAM vc_rtj_dec;

static int enc_initialized = 0;
static int dec_initialized = 0;


static struct CODEC_FOURCC {
  uint32_t codec;
  const char *fourcc_str;
  uint32_t codec_cap;
} codec_fourcc[] = {
  { VCODEC_RTJPEG, "RTJ0", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
};

static int codec_fourcc_num = sizeof(codec_fourcc)/sizeof(struct CODEC_FOURCC);

uint32_t
vc_rtjpeg_codec_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (codec == codec_fourcc[i].codec)
      return (cap_flag & codec_fourcc[i].codec_cap);
  }
  return VCODEC_CAP_NONE;
}

const char*
vc_rtjpeg_codec_to_fourcc_str(uint32_t codec)
{
  int i;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (codec == codec_fourcc[i].codec)
      return codec_fourcc[i].fourcc_str;
  }
  return NULL;
}

uint32_t
vc_rtjpeg_fourcc_str_to_codec(const char *fourcc_str)
{
  int i;

  if (!fourcc_str)
    return VCODEC_UNKNOWN;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (!strcmp(fourcc_str, codec_fourcc[i].fourcc_str))
      return codec_fourcc[i].codec;
  }
  return VCODEC_UNKNOWN;
}

/* encode functions */

int
vc_rtjpeg_encode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_YUV420P;
      break;
    case CSP_YV12:
    case CSP_I420:
    case CSP_YUV420P:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_rtjpeg_encode (unsigned char *pic_data, unsigned char *stream_buf,
    int stream_buf_size, unsigned int *flag)
{
  uint8_t *planes[3];
  int data_size;

  if (!enc_initialized) {
    fprintf(stderr, "vc_rtjpeg_encode: uninitialized.\n");
    return VCODEC_FAIL;
  }

  planes[0] = pic_data;
  planes[1] = pic_data + vc_rtj_enc.u_offset;
  planes[2] = pic_data + vc_rtj_enc.v_offset;
  data_size = RTjpeg_compress(vc_rtj_enc.rtj, stream_buf, planes);

  *flag = VCODEC_IS_INTRA;
  return data_size;
}

int
vc_rtjpeg_encode_quit (void)
{
  vc_rtj_enc.codec = 0;

  RTjpeg_close(vc_rtj_enc.rtj);
  vc_rtj_enc.rtj = NULL;
  enc_initialized = 0;
  return VCODEC_OK;
}

void
vc_rtjpeg_encode_print_param(void)
{
  if (!enc_initialized) {
    fprintf (stdout, "RTJPEG ENCODE uninitialized.\n");
    return;
  }
  fprintf (stdout, "RTJPEG ENCODE PARAMETER\n");
  fprintf (stdout, "dimmention: %dx%d\n",vc_rtj_enc.width,vc_rtj_enc.height);
  fprintf (stdout, "quality %d\n", vc_rtj_enc.quality);
  fflush (stdout);
}

int
vc_rtjpeg_encode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, format;

  if (vcodec->codec != VCODEC_RTJPEG) {
    fprintf(stderr, "vc_rtjpeg_encode_init: invalid codec.\n");
    return FAIL;
  }

  if (first) {
    vc_rtj_enc.codec = 0;
    vc_rtj_enc.width = 0;
    vc_rtj_enc.height = 0;
    vc_rtj_enc.in_csp = CSP_UNKNOWN;
    vc_rtj_enc.out_csp = CSP_UNKNOWN;

    vc_rtj_enc.u_offset = 0;
    vc_rtj_enc.v_offset = 0;
    vc_rtj_enc.rtj = NULL;
    vc_rtj_enc.quality = 255;
    first = 0;
  }

  vc_rtj_enc.codec = vcodec->codec;

  csp = CSP_UNKNOWN;
  csp = vc_rtjpeg_encode_csp_cap(vcodec->in_csp);
  if (csp != vcodec->in_csp) {
    fprintf(stderr,"vc_rtjpeg_encode_init: invalid in csp %s.\n",csp_to_str(vcodec->in_csp));
    return VCODEC_FAIL;
  }
  vc_rtj_enc.in_csp = csp;
  format = RTJ_YUV420;

  vcodec->out_csp = CSP_RTJPEG;
  vc_rtj_enc.out_csp = vcodec->out_csp;

  vc_rtj_enc.width = vcodec->width;
  vc_rtj_enc.height = vcodec->height;

  if (vc_rtj_enc.in_csp == CSP_YV12) {
    vc_rtj_enc.v_offset = vcodec->width * vcodec->height;
    vc_rtj_enc.u_offset = (vcodec->width * vcodec->height) / 4 * 5;
  } else {
    vc_rtj_enc.u_offset = vcodec->width * vcodec->height;
    vc_rtj_enc.v_offset = (vcodec->width * vcodec->height) / 4 * 5;
  }

  vc_rtj_enc.rtj = RTjpeg_init();
  RTjpeg_set_format(vc_rtj_enc.rtj, &format);
  RTjpeg_set_size(vc_rtj_enc.rtj, &vc_rtj_enc.width, &vc_rtj_enc.height);
  RTjpeg_set_quality(vc_rtj_enc.rtj, &vc_rtj_enc.quality);

  enc_initialized = 1;

  return VCODEC_OK;
}

/* decode functions */

int
vc_rtjpeg_decode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_YUV420P;
      break;
    case CSP_I420:
    case CSP_YUV420P:
    case CSP_YV12:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_rtjpeg_decode (unsigned char *pic_data, unsigned char *stream,
    int stream_length, unsigned int *flag)
{
  uint8_t *planes[3];

  if (!dec_initialized) {
    fprintf(stderr, "vc_rtjpeg_decode: uninitialized.\n");
    return VCODEC_FAIL;
  }

  if (flag && *flag & VCODEC_NULL_DECODE)
    return 0;

  planes[0] = pic_data;
  planes[1] = pic_data + vc_rtj_dec.u_offset;
  planes[2] = pic_data + vc_rtj_dec.v_offset;
  RTjpeg_decompress(vc_rtj_dec.rtj, stream, planes);

  *flag = VCODEC_IS_INTRA;
  return vc_rtj_dec.data_size;
}

int
vc_rtjpeg_decode_quit (void)
{
  vc_rtj_dec.codec = 0;

  RTjpeg_close(vc_rtj_dec.rtj);
  vc_rtj_dec.rtj = NULL;
  dec_initialized = 0;
  return VCODEC_OK;
}

void
vc_rtjpeg_decode_print_param(void)
{
  if (!dec_initialized) {
    fprintf (stdout, "RTJPEG DECODE uninitialized.\n");
    return;
  }
  fprintf (stdout, "RTJPEG DECODE PARAMETER\n");
  fprintf (stdout, "dimmention: %dx%d\n",vc_rtj_dec.width,vc_rtj_dec.height);
  fflush (stdout);
}

int
vc_rtjpeg_decode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, format, depth;

  if (vcodec->codec != VCODEC_RTJPEG) {
    fprintf(stderr, "vc_rtjpeg_decode_init: invalid codec.\n");
    return FAIL;
  }

  if (first) {
    vc_rtj_dec.codec = 0;
    vc_rtj_dec.width = 0;
    vc_rtj_dec.height = 0;
    vc_rtj_dec.in_csp = CSP_UNKNOWN;
    vc_rtj_dec.out_csp = CSP_UNKNOWN;

    vc_rtj_dec.data_size = 0;
    vc_rtj_dec.u_offset  = 0;
    vc_rtj_dec.v_offset  = 0;
    vc_rtj_dec.rtj = NULL;
    vc_rtj_dec.quality = 255;
    first = 0;
  }

  vc_rtj_dec.codec = vcodec->codec;

  csp = CSP_UNKNOWN;
  csp = vc_rtjpeg_decode_csp_cap(vcodec->out_csp);
  if (csp != vcodec->out_csp) {
    fprintf(stderr,"vc_rtjpeg_decode_init: invalid in csp %s.\n",csp_to_str(vcodec->out_csp));
    return VCODEC_FAIL;
  }
  vc_rtj_dec.out_csp = csp;
  format = RTJ_YUV420;

  vcodec->in_csp = CSP_RTJPEG;
  vc_rtj_dec.in_csp = vcodec->in_csp;

  vc_rtj_dec.width = vcodec->width;
  vc_rtj_dec.height = vcodec->height;

  depth = csp_to_pixel_depth(vc_rtj_dec.out_csp);
  vc_rtj_dec.data_size = vcodec->width * vcodec->height * depth / 8;

  if (vc_rtj_dec.out_csp == CSP_YV12) {
    vc_rtj_dec.v_offset  = vcodec->width * vcodec->height;
    vc_rtj_dec.u_offset  = (vcodec->width * vcodec->height) / 4 * 5;
  } else {
    vc_rtj_dec.u_offset  = vcodec->width * vcodec->height;
    vc_rtj_dec.v_offset  = (vcodec->width * vcodec->height) / 4 * 5;
  }

  vc_rtj_dec.rtj = RTjpeg_init();
  RTjpeg_set_format(vc_rtj_dec.rtj, &format);
//  RTjpeg_set_size(vc_rtj_dec.rtj, &vc_rtj_dec.width, &vc_rtj_dec.height);
//  RTjpeg_set_quality(vc_rtj_dec.rtj, &vc_rtj_dec.quality);

  dec_initialized = 1;

  return VCODEC_OK;
}

/* vcodec functions struct */

VCODEC_FUNCS vc_rtjpeg_funcs = {
  vc_rtjpeg_encode_init,
  vc_rtjpeg_encode_quit,
  vc_rtjpeg_encode_csp_cap,
  vc_rtjpeg_encode,
  vc_rtjpeg_encode_print_param,

  vc_rtjpeg_decode_init,
  vc_rtjpeg_decode_quit,
  vc_rtjpeg_decode_csp_cap,
  vc_rtjpeg_decode,
  vc_rtjpeg_decode_print_param,

  vc_rtjpeg_codec_to_fourcc_str,
  vc_rtjpeg_fourcc_str_to_codec,
  vc_rtjpeg_codec_cap,
};

#endif /* HAVE_RTJPEG */

