/* channel-list.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "parseopt.h"
#include "preset.h"

static const char *file_name = NULL;
static int line_num;

#define skip_space(p) while(*p==' '||*p=='\t')p++;
#define skip_space_rev(p) while(*p==' '||*p=='\t')p--;

typedef struct {
  const char *com;
  const char *arg;
} COMMAND;

typedef struct {
  const char *name;
  int command_num;
  COMMAND *command;
} PRESET;

static int preset_num = 0;
static PRESET *preset = NULL;

int
preset_set_preset(const char *name)
{
  int i;
  int com_num = 0;
  COMMAND *com = NULL;

  for (i = 0; i < preset_num; i++) {
    if (strcasecmp(preset[i].name, name) == 0) {
      com_num = preset[i].command_num;
      com = preset[i].command;
      break;
    }
  }
  if (!com) {
    fprintf(stderr, "preset_set_preset: cannot find preset name, [%s].\n",name);
    return FAIL;
  }
  for (i = 0; i < com_num; i++) {
    if (parseopt_set_command(com[i].com, com[i].arg) != OK) {
      fprintf(stderr, "preset_set_preset: error, preset name: [%s], command: %s, argument: %s.\n", name, com[i].com, com[i].arg);
      return FAIL;
    }
  }

  return OK;
}

int
preset_set(const char *arg)
{
  char buf[4096];
  const char *ap;
  char *bp;

  ap = arg;
  bp = buf;
  while (*ap != '\0') {
    while (*ap == ' ' || *ap == '\t' || *ap == ',')
      ap++;
    while (*ap != ' ' && *ap != ',' && *ap != '\n' && *ap != '\0')
      *bp++ = *ap++;
    *bp = '\0';
    if (preset_set_preset(buf) != OK) {
      fprintf(stderr, "preset_set: preset name %s failed.\n", buf);
      return FAIL;
    }
    bp = buf;
  }
  return OK;
}

int
preset_quit(void)
{
  COMMAND *com;
  int i, j;

  if (!preset)
    return OK;

  for (i = 0; i < preset_num; i++) {
    free((void*)preset[i].name);
    com = preset[i].command;
    for (j = 0; j < preset[i].command_num; j++) {
      free((void*)com[j].com);
      free((void*)com[j].arg);
    }
    free(preset[i].command);
  }
  free(preset);
  preset = NULL;
  preset_num = 0;
  return OK;
}

int
preset_init(const char *filename)
{
  char line_buf[4096];
  char subj_buf[4096];
  char pred_buf[4096];
  FILE *fp;
  void *p;

  if (filename == NULL)
    filename = PRESET_FILE_NAME;

  if ((fp = xfopen(filename, "r")) == NULL) {
    fprintf(stderr, "preset_init: cannot open %s.\n", filename);
    //exit(1);
    return FAIL;
  }

  file_name = filename;
  line_num = 0;
  while (fgets(line_buf, 4096, fp) != NULL) {
    line_num++;
    if (parse_line(line_buf, subj_buf, pred_buf) == FAIL) {
      fprintf(stderr, "preset_init: %s error line %d\n", file_name, line_num);
      return FAIL;
    }

    if (subj_buf[0] == '#') {
      continue;
    } else if (subj_buf[0] == '[') {
      int pre_num;
      preset_num++;
      pre_num = preset_num;
      p = realloc(preset, sizeof(PRESET) * (pre_num));
      if (!p) {
	fprintf(stderr, "preset_init: memory allocate error.\n");
	exit(1);
      }
      preset = (PRESET*)p;
      pre_num--;
      preset[pre_num].command = NULL;
      preset[pre_num].command_num = 0;
      p = strdup(pred_buf);
      if (!p) {
	fprintf(stderr, "preset_init: memory allocate error.\n");
	exit(1);
      }
      preset[pre_num].name = (const char*)p;
    } else {
      int com_num;
      int pre_num;
      COMMAND *com;

      if (!preset) {
        fprintf(stderr, "preset_init: %s error line %d\n", file_name, line_num);
	exit(1);
      }

      pre_num = preset_num;
      pre_num--;
      preset[pre_num].command_num++;
      com_num = preset[pre_num].command_num;
      com = preset[pre_num].command;
      p = realloc(com, sizeof(COMMAND) * (com_num));
      if (!p) {
	fprintf(stderr, "preset_init: memory allocate error.\n");
	exit(1);
      }
      preset[pre_num].command = (COMMAND*) p;
      com_num--;
      com = &preset[pre_num].command[com_num];
      p = strdup(subj_buf);
      if (!p) {
	fprintf(stderr, "preset_init: memory allocate error.\n");
	exit(1);
      }
      com->com = (const char*) p;
      p = strdup(pred_buf);
      if (!p) {
	fprintf(stderr, "preset_init: memory allocate error.\n");
	exit(1);
      }
      com->arg = (const char*) p;
    }
  }

  return OK;
}

