/* video.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

//#include "xvid.h"
//#include "image/image.h"

#include "csp.h"
#include "video.h"
#include "vutil.h"
#include "vcodec.h"
#include "audio.h"
#include "deinterlace.h"
#include "term.h"
#include "zoom.h"
#include "ifile.h"
#include "ofile.h"
#include "util.h"
#include "median_filt.h"
#include "colour.h"
#include "denoise.h"
#include "flicker.h"

//#define DEBUG printf

#ifdef UINT_MAX
#define MAX_UINT UINT_MAX
#else
#define MAX_UINT 4294967295U
#endif

VFRAME **vframe;
#define VFRAME_NUM_NORMAL     1
#define VFRAME_NUM_INTERLACED 4
int vframe_num = VFRAME_NUM_NORMAL;
double fps_ratio = 1.0;
double fps_inv_ratio = 1.0;

static OUT_FILE *out_file = NULL;
static IN_FILE *in_file = NULL;

static int frame_buf_size;

static int interlaced_src_flag = 0;
static int deinterlace_flag = 0;
static int decode_flag = 0;
static int encode_flag = 0;
static int zoom_flag = 0;
//static unsigned char *work_buf[2];
#define WORK_BUF_NUM 4
static unsigned char **work_buf;
static int work_buf_num = WORK_BUF_NUM;
static int work_buf_size = 0;

static int out_frames_num;

static int last_actual_in_frame_pos = -1;
static int last_actual_in_field_pos = -1;

static unsigned char *interlaced_buf = NULL;
static unsigned char *video_frame_buf = NULL;

static double frame_shift = 0.0;

#define FLICKER_LEVEL 5
static int flicker_level = FLICKER_LEVEL;
static int median_filt_flag = 0;

static int dec_out_csp = CSP_YV12;
static int enc_in_csp = CSP_YV12;

static unsigned char *y_buf_prev = NULL;
static unsigned char *y_buf_cur = NULL;
static int y_buf_size = 0;

static int denoise_flag = 0;
static int colour_flag = 0;

const char*
video_get_in_filename(void)
{
  return in_file->name;
}

int
video_get_in_frames(void)
{
  if (in_file)
    return in_file->actual_frames;
//    return in_file->frames;
  return 0;
}

int
video_get_in_pos(void)
{
  return last_actual_in_frame_pos;
}

double
video_get_out_fps(void)
{
  if (!out_file)
    return 0;
  return out_file->fps;
}

int
video_get_total_out_frames(void)
{
  return out_file->frames;
}

static int
get_actual_in_position(int position)
{
  int i;
  int max_num = in_file->cut_blocks_num;
  CUT_BLOCK *cut_blocks = in_file->cut_blocks;

  for (i = 0; i < max_num; i++) {
    if (cut_blocks[i].start <= position) {
      position += cut_blocks[i].end - cut_blocks[i].start + 1;
    }
  }
  return position;
}

static int
get_logical_in_position(int position)
{
  int num = in_file->cut_blocks_num - 1;
  CUT_BLOCK *cut_blocks = in_file->cut_blocks;

  for (; num >= 0; num--) {
    if (cut_blocks[num].start <= position) {
      position -= cut_blocks[num].end - cut_blocks[num].start + 1;
    }
  }
  return position;
}

static int
cut_frame_check(int frame_pos)
{
  int i;
  int max_num = in_file->cut_blocks_num;
  CUT_BLOCK *cut_blocks = in_file->cut_blocks;

  for (i = 0; i < max_num; i++) {
    if (cut_blocks[i].start <= frame_pos && cut_blocks[i].end >= frame_pos) {
      return 1;
    }
  }
  return 0;
}

static int get_key_pos_internal_call = 0;
int
video_get_prev_key_pos(int position)
{
  int ifrm_pos;
  int actual_pos;
  int cut_block_check = 1;
  double frame_time;
  int dt;

  frame_time = (double)position * fps_inv_ratio + frame_shift;
  ifrm_pos = (int)(frame_time + 0.5);

  actual_pos = get_actual_in_position(ifrm_pos);
  if (actual_pos >= in_file->actual_frames) {
    actual_pos = in_file->actual_frames - 1;
  }

  while (cut_block_check) {
    int pos;
    pos = prev_key_frame(in_file, actual_pos - 1);
    cut_block_check = cut_frame_check(pos);
    if (cut_block_check && pos == actual_pos) {
      if (get_key_pos_internal_call) {
	fprintf(stderr, "video_get_prev_key_pos: cannot find previous key position %d\n", position);
	return position;
      } else {
        get_key_pos_internal_call = 1;
        return video_get_next_key_pos(position);
      }
    }
    actual_pos = pos;
  }

  actual_pos = get_logical_in_position(actual_pos);
  frame_time = (double)actual_pos * fps_ratio - frame_shift;
  ifrm_pos = (int)(frame_time + 0.5);
  dt = (double)(ifrm_pos * fps_inv_ratio + 0.5);
  if (dt < actual_pos) {
    ifrm_pos++;
  }
  if (ifrm_pos == position && position > 0) {
    int pos;
    get_key_pos_internal_call = 1;
//    pos = video_get_prev_key_pos((int)((double)position-fps_inv_ratio));
    pos = video_get_prev_key_pos(--position);
    return pos;
  }
  get_key_pos_internal_call = 0;
  return ifrm_pos;
}

int
video_get_next_key_pos(int position)
{
  int ifrm_pos;
  int actual_pos;
  int cut_block_check = 1;
  double frame_time;
  int dt;

  frame_time = (double)position * fps_inv_ratio + frame_shift;
  ifrm_pos = (int)(frame_time + 0.5);

  actual_pos = get_actual_in_position(ifrm_pos);
  if (actual_pos >= in_file->actual_frames) {
    actual_pos = in_file->actual_frames - 1;
  }

  while (cut_block_check) {
    int pos;
    pos = next_key_frame(in_file, actual_pos + 1);
    cut_block_check = cut_frame_check(pos);
    if (cut_block_check && pos == actual_pos) {
      if (get_key_pos_internal_call) {
	fprintf(stderr, "video_get_next_key_pos: cannot find next key position %d\n", position);
	return position;
      } else {
        get_key_pos_internal_call = 1;
        return video_get_prev_key_pos(position);
      }
    }
    actual_pos = pos;
  }

  actual_pos = get_logical_in_position(actual_pos);
  frame_time = (double)actual_pos * fps_ratio - frame_shift;
  ifrm_pos = (int)(frame_time + 0.5);
  dt = (double)(ifrm_pos * fps_inv_ratio + 0.5);
  if (dt < actual_pos) {
    ifrm_pos++;
  }
  if (ifrm_pos == position) {
    int pos;
    get_key_pos_internal_call = 1;
    pos = video_get_next_key_pos(++position);
    return pos;
  }
  get_key_pos_internal_call = 0;
  return ifrm_pos;
}

static VFRAME **
init_vframe(int width, int height, int pix_depth, int csp)
{
  VFRAME **vfrm;
  int data_size;
  int i;

  data_size = width * height * pix_depth / 8;

  vfrm = (VFRAME**)malloc(sizeof(VFRAME*) * vframe_num);
  if (vfrm == NULL)
    goto fail;
  for (i = 0; i < vframe_num; i++) {
    vfrm[i] = (VFRAME*) malloc(sizeof(VFRAME));
    if (vfrm[i] == NULL)
      goto fail;
    vfrm[i]->data = (char*) malloc(data_size);
    if (vfrm[i]->data == NULL)
      goto fail;
    vfrm[i]->width = width;
    vfrm[i]->height = height;
    vfrm[i]->pix_depth = pix_depth;
    vfrm[i]->csp = csp;
    vfrm[i]->keyframe = 0;
    vfrm[i]->data_size = data_size;
    vfrm[i]->intensity = MAX_UINT;
    vfrm[i]->prev_intensity = MAX_UINT;
  }
  vframe = vfrm;
  return vfrm;

fail:
  fprintf(stderr, "init_vframe: memory allocate error.\n");
  exit(1);
  return NULL;
}

static void
vframe_free(void)
{
  int vfrm_num;
  for (vfrm_num = 0; vfrm_num < vframe_num; vfrm_num++) {
    if (vframe[vfrm_num]->data)
      free(vframe[vfrm_num]->data);
    free(vframe[vfrm_num]);
  }
  free(vframe);
  vframe_num = 0;
  vframe = NULL;
}

static void
rotate_vframe(void)
{
  VFRAME *tmp;
  int num = vframe_num;
  num--;
  tmp = vframe[num];
  for (num--; num >= 0; num--)
    vframe[num+1] = vframe[num];
  vframe[0] = tmp;
}

static void
free_work_buf(void)
{
  int num = work_buf_num;
  for (num = 0; num < work_buf_num; num++) {
    if (work_buf[num]) {
      free(work_buf[num]);
      work_buf[num] = NULL;
    }
  }
  free(work_buf);
  work_buf = NULL;
}

static int
set_work_buf(int size)
{
  int i;

  if (work_buf && size <=work_buf_size)
    return 0;

  if (!work_buf) {
    work_buf = (unsigned char**)calloc(work_buf_num, sizeof(unsigned char*));
    if (!work_buf) {
      mem_alloc_fail("set_work_buf", 1);
      exit(1);
    }
  }
  for (i = 0; i < work_buf_num; i++) {
    work_buf[i] = (unsigned char*) realloc(work_buf[i], size);
    if (!work_buf[i]) {
      mem_alloc_fail("set_work_buf", 1);
      exit(1);
    }
  }
  work_buf_size = size;
  return work_buf_num;
}

static unsigned char *
get_work_buf(void)
{
  unsigned char *tmp;

  tmp = work_buf[1];
  work_buf[1] = work_buf[0];
  work_buf[0] = tmp;
  return work_buf[0];
}

static int
video_config(void)
{
  int i;
  int i_pix_depth;
  int o_pix_depth;
  int i_data_size;
  int o_data_size;
  int data_size;
  int ww, wh;
  int depth;
  int csp;

  if (in_file == NULL || out_file == NULL)
    return -1;

  printf("\nINPUT FILE: %s\n", in_file->name);

  dec_out_csp = enc_in_csp;
#if 0
  if (in_file->vcodec == VCODEC_RGB24)
    dec_out_csp = CSP_RGB24;
  else
    dec_out_csp = CSP_YV12;
#endif

  if (vcodec_decode_set_codec(in_file->vcodec) != VCODEC_OK) {
    fprintf(stderr, "video_config: vcodec_decode_set_codec failed.\n");
    video_quit();
    exit(1);
  }
  csp = vcodec_decode_csp_cap(dec_out_csp);
  if (csp != dec_out_csp) {
    dec_out_csp = vcodec_decode_csp_cap(CSP_UNKNOWN);
    if (dec_out_csp == CSP_UNKNOWN) {
      fprintf(stderr, "video_config: vcodec_decode_csp_cap failed.\n");
      video_quit();
      exit(1);
    }
  }
  if (vcodec_decode_init(in_file->vcodec, in_file->actual_width,
       	in_file->actual_height, dec_out_csp) != VCODEC_OK) {
    fprintf(stderr, "video_config: vcodec_decode_init failed.\n");
    video_quit();
    exit(1);
  }
  decode_flag = 1;

  i_pix_depth = csp_to_pixel_depth(dec_out_csp);
  if (out_file->vcodec == VCODEC_RGB24)
    o_pix_depth = 24;
  else
    o_pix_depth = 12;
//  i_data_size =in_file->actual_width * in_file->actual_height * i_pix_depth / 8;
//  o_data_size = out_file->width * out_file->height * o_pix_depth / 8;
  i_data_size =in_file->actual_width * in_file->actual_height * 24 / 8;
  o_data_size = out_file->width * out_file->height * 24 / 8;

  frame_buf_size = (i_data_size > o_data_size) ? i_data_size : o_data_size;

  printf("CODEC: %s\n", vcodec_get_codec_name(in_file->vcodec));

  if ((vframe == NULL) ||
      (interlaced_src_flag != in_file->interlaced_src_flag) ||
      (vframe[0]->width != in_file->actual_width) ||
      (vframe[0]->height != in_file->actual_height) ||
      (vframe[0]->csp != dec_out_csp)) {
    interlaced_src_flag = in_file->interlaced_src_flag;
    if (vframe) {
      vframe_free();
    }
    if (interlaced_src_flag)
      vframe_num = VFRAME_NUM_INTERLACED;
    else
      vframe_num = VFRAME_NUM_NORMAL;
    vframe = init_vframe(in_file->actual_width, in_file->actual_height,
	i_pix_depth, dec_out_csp);
    interlaced_buf = (unsigned char*)realloc(interlaced_buf, i_data_size);
    if (!interlaced_buf)
      mem_alloc_fail("video_config", 1);

    y_buf_size = in_file->actual_width * in_file->actual_height;
    y_buf_prev = (unsigned char*)realloc(y_buf_prev, y_buf_size);
    if (!y_buf_prev)
      mem_alloc_fail("video_config", 1);
    y_buf_cur = (unsigned char*)realloc(y_buf_cur, y_buf_size);
    if (!y_buf_cur)
      mem_alloc_fail("video_config", 1);

    if (interlaced_src_flag) {
      if (flicker_init(in_file->actual_width, in_file->actual_height, dec_out_csp, in_file->flicker_eval_per) != OK) {
	fprintf(stderr, "video_config: flicker_init failed.\n");
	exit(1);
      }
    }
  }
  if (interlaced_src_flag)
    printf("INTERLACED SOURCE: %s\n", interlaced_src_flag ? "ON":"OFF");

  for (i = 0; i < vframe_num; i++) {
    vframe[i]->intensity = MAX_UINT;
    vframe[i]->prev_intensity = MAX_UINT;
  }

  fps_ratio = out_file->fps / in_file->fps;
  fps_inv_ratio = in_file->fps / out_file->fps;

  last_actual_in_frame_pos = -1;
  last_actual_in_field_pos = -1;

  median_filt_flag = 0;
  if (in_file->median_filt_flag) {
    median_filt_flag = in_file->median_filt_flag;
    printf("MEDIAN FILTER: ON\n");
  }

  deinterlace_flag = 0;
  if (in_file->deinterlace_flag) {
    deinterlace_flag = 1;
#if 0
    if (in_file->deinterlace_y_only_flag)
      deinterlace_set_y_only(in_file->deinterlace_y_only_flag);
    if (in_file->deinterlace_uv_only_flag)
      deinterlace_set_uv_only(in_file->deinterlace_uv_only_flag);
#endif
    if (deinterlace_init(in_file->actual_width, in_file->actual_height, NULL,
	  in_file->deinterlace_type, dec_out_csp) == -1) {
      fprintf(stderr, "DEINTERLACE initialize failed.\n");
      fprintf(stderr, "DEINTERLACE disable.\n");
      deinterlace_flag = 0; 
    } else
      printf("DEINTERLACE ON: %s\n", deinterlace_get_type());
  }

  colour_flag = colour_init(in_file->actual_width, in_file->actual_height,
                            in_file->bright, in_file->hue, in_file->contrast,
			    in_file->color, dec_out_csp);
  if (colour_flag < 0) {
    colour_flag = 0;
    fprintf(stderr, "COLOUR initialize failed.\n");
    fprintf(stderr, "COLOUR disable.\n");
  }

  denoise_flag = in_file->denoise_flag;
  if (denoise_flag) {
    if (dec_out_csp != CSP_YV12 && dec_out_csp != CSP_I420 && dec_out_csp != CSP_YUV420P) {
      fprintf(stderr, "DENOISE is not support RGB24 colorspace.\n");
      fprintf(stderr, "DENOISE disable.\n");
      denoise_flag = 0;
    } else {
      printf("DENOISE ON: %d\n", denoise_flag);
      denoise_set_param(in_file->denoise_radius, in_file->denoise_threshold,
	  in_file->denoise_ppthreshold, in_file->denoise_delay,
	  in_file->denoise_sharpen, in_file->denoise_uv_flag,
	  in_file->denoise_uvthreshold);
      denoise_init(in_file->actual_width, in_file->actual_height, dec_out_csp);
      denoise_set_test(in_file->denoise_test);
    }
  }

  if (in_file->width != out_file->width ||
      in_file->height != out_file->height) {
    int crop_top;
    int crop_bottom;
    if (in_file->vcodec == VCODEC_RGB24) {
      crop_top = in_file->crop_bottom;
      crop_bottom = in_file->crop_top;
    } else {
      crop_top = in_file->crop_top;
      crop_bottom = in_file->crop_bottom;
    }
    zoom_flag = 1;
    zoom_init(in_file->actual_width, in_file->actual_height,
	in_file->crop_left, crop_top,
	in_file->actual_width - in_file->crop_right,
	in_file->actual_height - crop_bottom,
	out_file->width, out_file->height,
	0, 0, out_file->width, out_file->height,
	dec_out_csp);
    printf ("crop left %d, right %d, top %d, bottom %d, width %d, height %d\n",
            in_file->crop_left, in_file->crop_right,
	    in_file->crop_top, in_file->crop_bottom,
	    in_file->width, in_file->height);
    printf ("zoom horizontal scale = %f, vertical scale = %f, width %d, height %d\n",
          (double)out_file->width / (double)in_file->width,
          (double)out_file->height / (double)in_file->height,
          out_file->width, out_file->height);
  }

//  frame_shift = (double)in_file->frame_shift / 100.0;
//  debug("video_config: frame_shift %f\n", frame_shift);

  ww = (in_file->actual_width > out_file->width) ?
    in_file->actual_width : out_file->width;
  wh = (in_file->actual_height > out_file->height) ?
    in_file->actual_height : out_file->height;
  depth = (i_pix_depth > o_pix_depth) ? i_pix_depth : o_pix_depth;
//  data_size = ww * wh * depth / 8;
  data_size = ww * wh * 24 / 8;
  set_work_buf(data_size);

//  out_frames_num = (int)((double)(in_file->frames-1) * fps_ratio);
  out_frames_num = (int)((double)(in_file->frames) * fps_ratio);

  if (in_file->flicker_level != 0)
    flicker_level = in_file->flicker_level;
  else
    flicker_level = FLICKER_LEVEL;

  return 0;
}

int
video_set_in_file(int index)
{
  int ret;

  if (out_file == NULL)
    return 0;

  in_file = set_in_file_video(index);
  ret = video_config();
  if (ret < 0) {
    if (vframe)
      free(vframe);
    quit_iofile();
    exit(1);
  }
  return ret;
}

void
video_unset_in_file(int index)
{
  if (out_file == NULL)
    return;

  vcodec_decode_quit();
  unset_in_file_video(index);
}

static unsigned char *
video_get_interlaced_frame(int field_pos)
{
  unsigned char *frm_buf = NULL;
  int shift = 0;
  int shift2 = 0;
  int shift3 = 0;
  int expand = 0;
  int field_delta = 0;
  unsigned int its_m2;
  unsigned int its_m1;
  unsigned int its_0;
  unsigned int its_p1;
  unsigned int its_p2;
  unsigned int fits_0 = MAX_UINT;
  unsigned int fits_1 = MAX_UINT;
  int i;
  int c0, c1;
  int field_order = in_file->field_order;
  int csp = dec_out_csp;
//static int pos = 0;
//pos++;

  if (csp == CSP_RGB24)
    field_order = !field_order;

  field_delta = field_pos - last_actual_in_field_pos;

  if (field_delta >= 3)
    shift = 1;
  if (field_delta >= 4)
    shift2 = 1;
  if (field_delta <= 2)
    shift3 = 1;

  if (field_pos & 1) {
    fits_0 = vframe[1]->prev_intensity;
    if (shift)
      fits_1 = vframe[2]->intensity;
    else
      fits_1 = vframe[1]->intensity;
  } else {
    fits_0 = vframe[1]->intensity;
    if (shift)
      fits_1 = vframe[1]->prev_intensity;
    else
      fits_1 = vframe[0]->prev_intensity;
  }
#if 0
  {
    static int p = 0;
  printf("p %4d: 0 %u, 1 %u\n", p++, fits_0, fits_1);
  }
#endif
  for (i = c0 = c1 = 0; i < 4; i++) {
    if (fits_0 > vframe[i]->intensity)
      c0++;
    if (fits_0 > vframe[i]->prev_intensity)
      c0++;
    if (fits_1 > vframe[i]->intensity)
      c1++;
    if (fits_1 > vframe[i]->prev_intensity)
      c1++;
  }
#define debug printf
  if (c0 >= flicker_level && c1 >= flicker_level) {
    expand = 1;
//      debug("pos:%5d:field:%5d: icount %d: c0 %d, c1 %d\n", pos, field_pos, icount++, c0, c1);
  }

  its_m2 = MAX_UINT;
  its_m1 = MAX_UINT;
  its_0 = MAX_UINT;
  its_p1 = MAX_UINT;
  its_p2 = MAX_UINT;


  if (field_pos & 1) {
    if (shift2 && expand)
      its_m2 = vframe[2]->prev_intensity;

    if (shift || (expand && !shift3))
      its_m1 = vframe[2]->intensity;

    its_0 = vframe[1]->prev_intensity;

    if (!shift || (expand && !shift2))
      its_p1 = vframe[1]->intensity;

    if (shift3 && expand)
      its_p2 = vframe[0]->prev_intensity;

//    if (expand)
//    debug("pos:%5d:field:%5d: its_m2 %u, its_m1 %u, its_0 %u, its_p1 %u, its_p2 %u\n", pos, field_pos-last_actual_in_field_pos, its_m2, its_m1, its_0, its_p1, its_p2);
//    debug("ki: shift2 %d, shift %d shift3 %d, expand %d\n", shift2, shift, shift3, expand);

    if ((its_0 < its_m2) && (its_0 < its_m1) &&
	(its_0 < its_p1) && (its_0 < its_p2)) {
      field_interleave(interlaced_buf, vframe[2]->data, vframe[1]->data,
	  in_file->actual_width, in_file->actual_height,
	  csp, field_order);
      last_actual_in_field_pos = field_pos;
      frm_buf = interlaced_buf;
    } else if ((its_m1 < its_m2) && (its_m1 < its_p1) && (its_m1 < its_p2)) {
      last_actual_in_field_pos = field_pos - 1;
      frm_buf = vframe[2]->data;
    } else if ((its_p1 < its_m2) && (its_p1 < its_p2)) {
      last_actual_in_field_pos = field_pos + 1;
      frm_buf = vframe[1]->data;
    } else if (its_m2 < its_p2) {
      field_interleave(interlaced_buf, vframe[3]->data, vframe[2]->data,
	  in_file->actual_width, in_file->actual_height,
	  csp, field_order);
      last_actual_in_field_pos = field_pos - 2;
      frm_buf = interlaced_buf;
    } else {
      field_interleave(interlaced_buf, vframe[1]->data, vframe[0]->data,
	  in_file->actual_width, in_file->actual_height,
	  csp, field_order);
      last_actual_in_field_pos = field_pos + 2;
      frm_buf = interlaced_buf;
    }
  } else {
    if (shift2 && expand)
      its_m2 = vframe[2]->intensity;

    if (shift || (expand && !shift3))
      its_m1 = vframe[1]->prev_intensity;

    its_0 = vframe[1]->intensity;

    if (!shift || (expand && !shift2))
      its_p1 = vframe[0]->prev_intensity;

    if (shift3 && expand)
      its_p2 = vframe[0]->intensity;

//    if (expand)
//    debug("pos:%5d:field:%5d: its_m2 %u, its_m1 %u, its_0 %u, its_p1 %u, its_p2 %u\n", pos, field_pos-last_actual_in_field_pos, its_m2, its_m1, its_0, its_p1, its_p2);
//    debug("gu: shift2 %d, shift %d shift3 %d, expand %d\n", shift2, shift, shift3, expand);

    if ((its_0 < its_m2) && (its_0 < its_m1) &&
	(its_0 < its_p1) && (its_0 < its_p2)) {
      last_actual_in_field_pos = field_pos;
      frm_buf = vframe[1]->data;
    } else if ((its_m1 < its_m2) && (its_m1 < its_p1) && (its_m1 < its_p2)) {
      field_interleave(interlaced_buf, vframe[2]->data, vframe[1]->data,
	  in_file->actual_width, in_file->actual_height,
	  csp, field_order);
      last_actual_in_field_pos = field_pos - 1;
      frm_buf = interlaced_buf;
    } else if ((its_p1 < its_m2) && (its_p1 < its_p2)) {
      field_interleave(interlaced_buf, vframe[1]->data, vframe[0]->data,
	  in_file->actual_width, in_file->actual_height,
	  csp, field_order);
      last_actual_in_field_pos = field_pos + 1;
      frm_buf = interlaced_buf;
    } else if (its_m2 < its_p2) {
      last_actual_in_field_pos = field_pos - 2;
      frm_buf = vframe[2]->data;
    } else {
      last_actual_in_field_pos = field_pos + 2;
      frm_buf = vframe[0]->data;
    }
  }

  return frm_buf;
}

static void
clear_vframe(void)
{
  rotate_vframe();
  vframe[0]->keyframe = 0;
  vframe[0]->intensity = MAX_UINT;
  vframe[0]->prev_intensity = MAX_UINT;
  memset(vframe[0]->data, 0, vframe[0]->data_size);
}

static int
load_vframe(void)
{
  int width = in_file->actual_width;
  int height = in_file->actual_height;
  int size, flag;

  rotate_vframe();
  if (decode_flag) {
    unsigned char *read_buf;
    read_buf = get_work_buf();
    size = read_video_data(in_file, read_buf, &vframe[0]->keyframe);
    if (size > 0) {
      flag = 0;
      vcodec_decode(vframe[0]->data, read_buf, size, &flag);
#if 0
      while (vcodec_decode(vframe[0]->data, read_buf, size, &flag) == 0) {
        size = read_video_data(in_file, read_buf, &vframe[0]->keyframe);
	if (size <= 0 && vframe_num > 1) {
          memcpy(vframe[0]->data, vframe[1]->data, vframe[1]->data_size);
	  break;
	}
	flag = 0;
      }
#endif
    } else if (vframe_num > 1) {
      memcpy(vframe[0]->data, vframe[1]->data, vframe[1]->data_size);
    }
  } else {
    size = read_video_data(in_file, vframe[0]->data, &vframe[0]->keyframe);
    if ((size <= 0) && (vframe_num > 1))
      memcpy(vframe[0]->data, vframe[1]->data, vframe[1]->data_size);
  }

  if (interlaced_src_flag) {
    unsigned char *prev_data;
    unsigned char *cur_data;
    int field_order = in_file->field_order;
    int csp;

    if (dec_out_csp == CSP_RGB24) {
      unsigned char *tmp;
      tmp = y_buf_prev;
      y_buf_prev = y_buf_cur;
      y_buf_cur = tmp;
      rgb2y(y_buf_cur, vframe[0]->data, width, height);
      prev_data = y_buf_prev;
      cur_data = y_buf_cur;
      field_order = !field_order;
      csp = CSP_YV12;
    } else {
      prev_data = vframe[1]->data;
      cur_data = vframe[0]->data;
      csp = dec_out_csp;
    }

#if 0
    vframe[0]->prev_intensity = flicker_func(prev_data, cur_data, width, height,
       	csp, field_order);
    vframe[0]->intensity = flicker_func(cur_data, cur_data, width, height,
	csp, field_order);
#else
    vframe[0]->prev_intensity = flicker(prev_data, cur_data);
    vframe[0]->intensity = flicker(cur_data, cur_data);
#endif
  }
  return 0;
}

static int
video_move_in_position(int position, int last_position)
{
  int i;
  int pos;
  int last_pos;
  int read_pos;
  int keyframe;
  int key_pos;
  int size;
  unsigned char *read_buf;
  unsigned char *dummy_buf = NULL;

  last_position++;
  if (position == last_position)
    return position;

  if (interlaced_src_flag) {
    pos = position;
    last_pos = last_position;
    read_pos = last_pos;
    if ((pos - last_pos) > (vframe_num-1) || last_pos > pos) {
      read_pos = pos - (vframe_num - 1);
      if (read_pos < 0)
	for (; read_pos < 0; read_pos++)
	  clear_vframe();
      if (!decode_flag) {
        set_in_file_video_frame_position(in_file, read_pos);
      } else {
	key_pos = prev_key_frame(in_file, read_pos);
	if (key_pos > last_pos || last_pos > pos) {
          set_in_file_video_frame_position(in_file, key_pos);
	  i = key_pos;
	} else {
	  i = last_pos;
	}
        read_buf = get_work_buf();
	dummy_buf = get_work_buf();
	for (; i < read_pos; i++) {
	  int flag;
          size = read_video_data(in_file, read_buf, &keyframe);
	  flag = VCODEC_NULL_DECODE;
	  if (size > 0)
	    vcodec_decode(dummy_buf, read_buf, size, &flag);
#if 0
	  int ret = 0;
          while (!ret) {
            size = read_video_data(in_file, read_buf, &keyframe);
	    flag = VCODEC_NULL_DECODE;
	    if (size == 0)
	      break;
	    ret = vcodec_decode(dummy_buf, read_buf, size, &flag);
	  }
#endif
	}
      }
    }

    for (i = read_pos; i < pos; i++) {
      load_vframe();
    }
  } else {
    pos = position;
    last_pos = last_position;
    if (!decode_flag) {
      set_in_file_video_frame_position(in_file, pos);
    } else {
      key_pos = prev_key_frame(in_file, position);
      if (key_pos > last_pos || last_pos > pos) {
        set_in_file_video_frame_position(in_file, key_pos);
        i = key_pos;
      } else {
        i = last_pos;
      }
      read_buf = get_work_buf();
      dummy_buf = get_work_buf();
      for (; i < pos; i++) {
	int flag;
          size = read_video_data(in_file, read_buf, &keyframe);
	  flag = VCODEC_NULL_DECODE;
	  if (size > 0)
	    vcodec_decode(dummy_buf, read_buf, size, &flag);
#if 0
	int ret = 0;
        while (!ret) {
          size = read_video_data(in_file, read_buf, &keyframe);
	  if (size == 0)
	    break;
	  flag = VCODEC_NULL_DECODE;
	  ret = vcodec_decode(dummy_buf, read_buf, size, &flag);
	}
#endif
      }
    }
  }
  return pos;
}

static int
video_load_frame(int position)
{
  int ifrm_pos;
  int ifield_pos = 0;
  int actual_pos;
  double frame_time;
  double field_time;

  frame_time = (double)position * fps_inv_ratio + frame_shift;
  if (interlaced_src_flag) {
    field_time = frame_time * 2;
    ifield_pos = (int)(field_time + 0.5);
    ifrm_pos = (ifield_pos + 3) / 2;
  } else {
    ifrm_pos = (int)(frame_time + 0.5);
  }
  //printf("video_load_frame: ifrm_pos %d\n", ifrm_pos);

  actual_pos = get_actual_in_position(ifrm_pos);
  if (actual_pos >= in_file->actual_frames) {
    fprintf(stderr,"video_load_frame: position invalid pos %d, total_frames %d\n", actual_pos, in_file->actual_frames);
    //return -1;
  }
#if 0
#ifndef NDEVIEW_SOURCE
  if (actual_pos < last_actual_in_frame_pos) {
    fprintf(stderr,"video_load_frame: position invalid pos %d, total_frames %d\n", actual_pos, in_file->actual_frames);
    return -1;
  }
#endif
#endif
  if (actual_pos != last_actual_in_frame_pos) {
    if (actual_pos != last_actual_in_frame_pos+1) {
      video_move_in_position(actual_pos, last_actual_in_frame_pos);
    }
    load_vframe();
    if (interlaced_src_flag) {
      video_frame_buf = video_get_interlaced_frame(ifield_pos);
    } else {
      video_frame_buf = vframe[0]->data;
    }
  }
  last_actual_in_frame_pos = actual_pos;
  return actual_pos;
}

static int
video_put_frame(unsigned char *frm_buf)
{
  int keyframe, flag;
  unsigned char *dest_buf = NULL;
  int size = 0;

  dest_buf = get_work_buf();
  flag = 0;
  size = vcodec_encode(frm_buf, dest_buf, frame_buf_size, &flag);
  if (size < 0) {
    fprintf(stderr, "video_put_frame: vcodec_encode failed.\n");
    video_quit();
    exit(1);
  }
  if (flag & VCODEC_IS_INTRA)
    keyframe = 1;
  else
    keyframe = 0;

  write_video_frame(out_file, dest_buf, size, keyframe);
  return size;
}

int
video_get_frames_num(void)
{
  if (out_file == NULL)
    return 0;
  return out_frames_num;
}

int
video_frame_process(int frame_count)
{
  int data_size;
  int pos;
  unsigned char *src;
  unsigned char *dest;

  if (out_file == NULL)
    return 0;

  pos = video_load_frame(frame_count);
  if (pos < 0)
    return -1;
  src = video_frame_buf;

  if (median_filt_flag) {
    dest = get_work_buf();
    median_filter(dest, src, in_file->actual_width, in_file->actual_height,
	in_file->actual_width, dec_out_csp);
    src = dest;
  }

  if (deinterlace_flag) {
    dest = get_work_buf();
    deinterlace(dest, src);
    src = dest;
  }

  if (colour_flag) {
    dest = get_work_buf();
    colour(dest, src, 0);
    src = dest;
  }

  if (denoise_flag) {
    dest = get_work_buf();
    denoise(dest, src);
    src = dest;
  }

#if 0
  if (colour_flag) {
    dest = get_work_buf();
    colour(dest, src, 1);
    src = dest;
  }
#endif

  if (zoom_flag) {
    dest = get_work_buf();
    zoom(dest, src);
    src = dest;
  }

  data_size = video_put_frame(src);
  return data_size;
}

void
video_quit(void)
{
  if (out_file == NULL)
    return;
  vcodec_decode_quit();
  vcodec_encode_quit();
  quit_iofile();
  free_work_buf();
  vframe_free();
}

int
video_init(void)
{
  int csp;

  out_file = get_video_out_file();
  if (out_file == NULL)
    return 0;

  ofile_set_video(out_file);

  if (vcodec_encode_set_codec(out_file->vcodec) != VCODEC_OK) {
    fprintf(stderr, "video_init: vcodec_encode_set_codec failed.\n");
    video_quit();
    exit(1);
  }
  csp = vcodec_encode_csp_cap(enc_in_csp);
  if (csp == CSP_UNKNOWN) {
    fprintf(stderr, "video_init: vcodec_encode_csp_cap failed.\n");
    video_quit();
    exit(1);
  }
  if (vcodec_encode_init(out_file->vcodec, out_file->width, out_file->height,
	out_file->fps, csp) != VCODEC_OK) {
    fprintf(stderr, "video_init: vcodec_encode_init failed.\n");
    video_quit();
    exit(1);
  }
  vcodec_encode_print_param();
  encode_flag = 1;

  return 0;
}

