/* ofile.c */

#include <stdio.h>
#include <stdlib.h>

#include "video.h"
#include "audio.h"
#include "parseopt.h"
#include "ifile.h"
#include "ofile.h"
#include "deinterlace.h"
#include "util.h"
#include "vutil.h"
#include "vcodec.h"
#include "zoom.h"
#include "acodec.h"

#ifdef NDEVIEW_SOURCE
#include "xwin.h"
//static int output_type = OUTPUT_XWIN;
static char const *default_outfile_name = "xwin";
static int vo_type = 0;
#else
//static int output_type = OUTPUT_FILE;
static const char *suff_avi = "avi";
static const char *suff_mov = "mov";
static const char *default_outfile_name = "out";
static int out_file_format = FILE_FORMAT_UNKNOWN;
#endif /* NDEVIEW_SOURCE */

static OUT_FILE *out_file = NULL;
static int out_video_codec = VCODEC_UNKNOWN;
static int out_width = 0;
static int out_height = 0;
static int out_write_frames = 0;
static double out_fps = 0.0;

static int video_disable_flag = 0;
static int audio_disable_flag = 0;

#define MAX_INT (2147483647)

double
ofile_get_fps(void)
{
  if (out_file && out_file->fps != 0.)
    return out_file->fps;
  return out_fps;
}

static int opt_frame_rate(const char *arg)
{
    out_fps = strtod(arg, 0);
    return 0;
}

static int opt_frame_size(const char *arg)
{
  char *p;
  char *ep;
  int width;
  int height;

  p = (char*)arg;
  width = strtol(p, &ep, 10);
  if (ep == NULL) {
    fprintf (stderr, "option frame size error. '%s'\n", arg);
    exit(1);
  }
  p = ++ep;
  height = strtol(p, &ep, 10);
  if (width <= 0 || height <= 0) {
    fprintf(stderr, "Incorrect frame size width %d, height %d\n", width,height);
    exit(1);
//    return -1;
  }
  if ((width % 16) != 0) {
    fprintf(stderr, "Frame width must be a multiple of 16, %d\n", width);
    exit(1);
//    return -1;
  }
  if ((height % 16) != 0) {
    fprintf(stderr, "Frame height must be a multiple of 16, %d\n", height);
    exit(1);
  }
  out_width = width;
  out_height = height;
  return 0;
}

#ifndef NDEVIEW_SOURCE
static SEL_COMPO video_codec_sel[] = {
#ifdef HAVE_LIBXVIDCORE
  { "XVID", VCODEC_XVID },
#endif /* HAVE_LIBXVIDCORE */
  { "YV12", VCODEC_YV12 },
  { "RGB", VCODEC_RGB24 },
#ifdef HAVE_RTJPEG
  { "RTJPEG", VCODEC_RTJPEG },
#endif /* HAVE_RTJPEG */
};
#else
static SEL_COMPO vo_type_sel[] = {
  { "XV", XWIN_VO_XV },
  { "XSHM", XWIN_VO_XSHM },
  { "X11", XWIN_VO_X11 }
};
#endif

#ifndef NDEVIEW_SOURCE
static SEL_COMPO file_format_sel[] = {
  { "AVI", FILE_FORMAT_AVI },
  { "MOV", FILE_FORMAT_MOV },
};
#endif /* NDEVIEW_SOURCE */

static int
set_out_filename (const char *name)
{
  int len;

  if (!name)
    return 0;

  if (out_file) {
    if (out_file->name)
      free(out_file->name);
    free(out_file);
    out_file = NULL;
  }
  out_file = (OUT_FILE*) malloc(sizeof(OUT_FILE));
  if (out_file == NULL)
    goto memfail;

  memset(out_file, 0, sizeof(OUT_FILE));

  len = strlen(name);
  if (len > MAX_FILENAME_LENGTH-1)
    len = MAX_FILENAME_LENGTH-1;
  out_file->name = (char*) malloc(len+1);
  if (out_file->name == NULL)
    goto memfail;

  memcpy(out_file->name, name, len);
  out_file->name[len] = '\0';

  out_file->avifile = NULL;
  out_file->qt = NULL;
  out_file->vqtrk = NULL;
  out_file->aqtrk = NULL;

  return 0;

memfail:
  fprintf(stderr,"out_out_filename: malloc error.\n");
  exit(1);
}

void
close_out_file(OUT_FILE *ofile)
{
#ifndef NDEVIEW_SOURCE
  if (ofile->file_format == FILE_FORMAT_AVI) {
    if (ofile->avifile) {
      AVI_close(ofile->avifile);
      ofile->avifile = NULL;
    }
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    if (ofile->qt) {
      qtime_close(ofile->qt);
      ofile->qt = NULL;
      ofile->vqtrk = NULL;
      ofile->aqtrk = NULL;
    }
  } else {
    fprintf(stderr, "close_out_file: unknown file format.\n");
  }
#else
  if (ofile->avifile || ofile->qt) {
    fprintf(stderr, "close_out_file: invalid file format.\n");
  }
#endif /* NDEVIEW_SOURCE */
}

void
free_out_files(void)
{
  if (out_file == NULL)
    return;

  close_out_file(out_file);

  if (out_file->name)
    free(out_file->name);
  free(out_file);
  out_file = NULL;
}

int
opt_filter_type(const char *arg)
{
  zoom_set_xfilt(arg);
  zoom_set_yfilt(arg);
  return 0;
}

int
opt_window_type(const char *arg)
{
  zoom_set_xwindow(arg);
  zoom_set_ywindow(arg);
  return 0;
}

#if 0
int
opt_flicker_func(const char *arg)
{
  flicker_func_num = atoi(arg);
  set_flicker_func(flicker_func_num);
  return 0;
}
#endif

OptionDef out_file_param[] = {
#ifndef NDEVIEW_SOURCE
    { "frames", NULL, HAS_ARG, {(void*)&out_write_frames}, {0}, {0}, MAX_INT, "number of frames to write", "number"},
    { "file-format", "FILE_FORMAT", HAS_ARG|OPT_SEL, {(void*)&out_file_format}, {FILE_FORMAT_MOV}, {(int)file_format_sel}, sizeof(file_format_sel)/sizeof(SEL_COMPO), "output file format", "format"},
    { "vcodec", "VIDEO_CODEC", HAS_ARG|OPT_SEL, {(void*)&out_video_codec}, {VCODEC_UNKNOWN}, {(int)video_codec_sel}, sizeof(video_codec_sel)/sizeof(SEL_COMPO), "output video codec", "codec"},
#endif
    { "fps", "FPS", HAS_ARG|OPT_FUNC, {(void*)opt_frame_rate}, {(int)"0"}, {0}, 0, "output frame rate", "r"},
    { "s", "FRAME_SIZE", HAS_ARG|OPT_FUNC, {(void*)opt_frame_size}, {0}, {0}, 0, "set output frame size (WidthxHeight)", "WxH"},
//    { "flicker_level", "FLICKER_LEVEL", HAS_ARG, {(void*)flicker_level}, {0}, {0}, 6, "flicker level", "level"},
//    { "flicker_func", "FLICKER_LEVEL", HAS_ARG|OPT_FUNC, {(void*)opt_flicker_func}, {(int)"2"}, {0}, 0, "flicker func (1 or 2)", "num"},
    { "zoom-filter-type", "ZOOM_FILTER_TYPE", HAS_ARG|OPT_FUNC, {(void*)opt_filter_type}, {0}, {0}, 0, "zoom filter type", "type"},
    { "zoom-window-type", "ZOOM_WINDOW_TYPE", HAS_ARG|OPT_FUNC, {(void*)opt_window_type}, {0}, {0}, 0, "zoom window type", "type"},
#ifndef NDEVIEW_SOURCE
//    { "o", NULL, HAS_ARG|OPT_FUNC, {(void*)set_out_filename}, {(int)"out.mov"}, {0}, 0, "output file name", "name"},
    { "o", NULL, HAS_ARG|OPT_FUNC, {(void*)set_out_filename}, {(int)0}, {0}, 0, "output file name", "name"},
#else
    { "vo", NULL, HAS_ARG|OPT_SEL, {(void*)&vo_type}, {(int)XWIN_VO_XSHM}, {(int)vo_type_sel}, sizeof(vo_type_sel)/sizeof(SEL_COMPO), "video output type", "type"},
#endif
};

int out_file_param_num = sizeof(out_file_param) / sizeof(OptionDef);

int
open_out_file(OUT_FILE *ofile)
{
#ifndef NDEVIEW_SOURCE
  ofile->file_format = out_file_format;

  if (ofile->file_format == FILE_FORMAT_AVI) {
    ofile->avifile = AVI_open_output_file(ofile->name);
    if (ofile->avifile == NULL) {
      AVI_print_error(ofile->name);
      exit(1);
    }
    return ofile->file_format;
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    ofile->qt = qtime_open_write(ofile->name);
    if (ofile->qt == NULL) {
      qtime_close(ofile->qt);
      ofile->qt = NULL;
      ofile->vqtrk = NULL;
      ofile->aqtrk = NULL;
      fprintf(stderr, "file open error, %s.\n", ofile->name);
      exit(1);
    }
    return ofile->file_format;
  } else {
    fprintf(stderr, "open_out_file: unknown file format.\n");
  }
  return -1;
#else
  ofile->file_format = FILE_FORMAT_UNKNOWN;
  return 0;
#endif /* NDEVIEW_SOURCE */
}

static int
check_out_file(void)
{
  char buf[2048];
  check_in_files();

  if (!out_file) {
#ifndef NDEVIEW_SOURCE
    const char *suf = "put";
    if (out_file_format == FILE_FORMAT_AVI)
      suf = suff_avi;
    else if (out_file_format == FILE_FORMAT_MOV)
      suf = suff_mov;
    sprintf(buf, "%s.%s", default_outfile_name, suf);
#else
    sprintf(buf, "%s", default_outfile_name);
#endif /* NDEVIEW_SOURCE */
    set_out_filename(buf);
  }

#ifndef NDEVIEW_SOURCE
  if (open_out_file(out_file) < 0) {
    exit(1);
  }
#else
  out_file->avifile = NULL;
  out_file->qt = NULL;
#endif

#ifdef NDEVIEW_SOURCE
  out_file->vcodec = VCODEC_YV12;
#else
  out_file->vcodec = out_video_codec;
#endif /* NDEVIEW_SOURCE */
  out_file->compressor = NULL;
  if (out_video_codec != VCODEC_UNKNOWN) {
    if (vcodec_encode_set_codec(out_video_codec) != VCODEC_OK) {
      fprintf(stderr, "check_out_file: invalid video codec.\n");
      exit(1);
    } else {
      out_file->vcodec = out_video_codec;
      out_file->compressor = vcodec_codec_to_fourcc_str(out_video_codec);
    }
  }
  out_file->frames = out_write_frames;
  out_file->width = out_width;
  out_file->height = out_height;
  out_file->fps = out_fps;

  return 0;
}

int
ofile_set_video(OUT_FILE *ofile)
{
  if (video_disable_flag) {
    ofile->vqtrk = NULL;
    return -1;
  }

  if (ofile->file_format == FILE_FORMAT_AVI) {
    AVI_set_video(ofile->avifile, ofile->width, ofile->height,
                         ofile->fps, (char*)ofile->compressor);
    return 0;
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    if (!ofile->vqtrk) {
      ofile->vqtrk = qtime_create_track(ofile->qt);
      if (ofile->vqtrk == NULL) {
        qtime_close(ofile->qt);
        ofile->qt = NULL;
	ofile->vqtrk = ofile->aqtrk = NULL;
        fprintf(stderr, "ofile_set_video: create track error, %s.\n", ofile->name);
        exit(1);
      }
    }
    return qtime_track_set_video(ofile->vqtrk, ofile->width, ofile->height,
	                         ofile->fps, (char*)ofile->compressor);
  } else {
    fprintf(stderr, "ofile_set_video: unknown file format.\n");
  }
  return -1;
}

int
ofile_set_audio(OUT_FILE *ofile)
{
  int ret;

  if (audio_disable_flag) {
    ofile->aqtrk = NULL;
    return -1;
  }

  if (ofile->file_format == FILE_FORMAT_AVI) {
    ofile->audio_be = 0;
    ofile->format = acodec_codec_to_code(ofile->acodec);
    AVI_set_audio (ofile->avifile, ofile->channels, ofile->rate,
                          ofile->bits, ofile->format, ofile->bitrate);
    return 0;
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    if (!ofile->aqtrk) {
      ofile->aqtrk = qtime_create_track(ofile->qt);
      if (ofile->aqtrk == NULL) {
        qtime_close(ofile->qt);
        ofile->qt = NULL;
	ofile->vqtrk = ofile->aqtrk = NULL;
        fprintf(stderr, "ofile_set_audio: create track error, %s.\n", ofile->name);
        exit(1);
      }
    }
    ofile->compressor = acodec_codec_to_fourcc(ofile->acodec);
    ret = qtime_track_set_audio(ofile->aqtrk, ofile->channels, ofile->rate,
                                 ofile->bits, (char*)ofile->compressor);
    if (ofile->acodec == ACODEC_MP3) {
      qtime_track_set_audio_ext(ofile->aqtrk, 1, -2, ofile->frame_size, 0, 0, ofile->bytespspl);
    }
    return ret;
  }
  return -1;
}

int
init_ofile(void)
{
  int total_audio_bytes = 0;
  IN_FILE *i_file;

  if (check_in_files() < 0) {
    fprintf(stderr, "init_ofile: check_in_files error.\n");
    quit_ifile();
    quit_ofile();
    exit(1);
  }
  if (check_out_file() < 0) {
    fprintf(stderr, "init_ofile: check_out_files error.\n");
    quit_ifile();
    quit_ofile();
    exit(1);
  }

  i_file = ifile_get_in_file(0);

  if (out_file->width == 0)
    out_file->width = i_file->actual_width;
  if (out_file->height == 0)
    out_file->height = i_file->actual_height;
  if (out_file->vcodec == VCODEC_UNKNOWN) {
    out_file->vcodec = i_file->vcodec;
    if (vcodec_encode_set_codec(out_file->vcodec) != VCODEC_OK) {
      fprintf(stderr, "init_ofile: invalid video codec.\n");
      exit(1);
    }
    out_file->compressor = vcodec_codec_to_fourcc_str(out_file->vcodec);
  }
  if (out_file->fps == 0.0)
    out_file->fps = i_file->fps;

  if (out_file->frames == 0)
    out_file->frames = ifile_calc_total_frames(out_file->fps);

  total_audio_bytes = ifile_calc_total_audio_bytes();
  if (total_audio_bytes != 0) {
    out_file->acodec = i_file->acodec;
    out_file->channels = i_file->channels;
    out_file->format = i_file->format;
    out_file->rate = i_file->rate;
    out_file->bits = i_file->bits;
    out_file->bitrate = i_file->bitrate;
  } else {
    audio_disable_flag = TRUE;
  }

#ifndef NDEVIEW_SOURCE
//  ofile_set_video(out_file);
//  ofile_set_audio(out_file);
  printf("compressor %s\n", out_file->compressor);
#else
  {
    int depth = 0;
    if (out_file->vcodec != VCODEC_YV12 && out_file->vcodec != VCODEC_RGB24) {
      out_file->vcodec = VCODEC_YV12;
      depth = 12;
    } else if (out_file->vcodec == VCODEC_YV12) {
      depth = 12;
    } else if (out_file->vcodec == VCODEC_RGB24) {
      depth = 24;
    }
    xwin_set_vo_type(NULL, vo_type);
    xwin_init(out_file->width, out_file->height, depth);
  }
#endif

  return 0;
}

void
quit_ofile(void)
{
  if (out_file) {
    close_out_file(out_file);
    free_out_files();
  }
#ifdef NDEVIEW_SOURCE
  xwin_quit();
#endif
}

OUT_FILE*
get_video_out_file(void)
{
  if (video_disable_flag)
    return NULL;
  return out_file;
}

OUT_FILE*
get_audio_out_file(void)
{
  if (audio_disable_flag)
    return NULL;
  return out_file;
}

int
write_video_frame(OUT_FILE *ofile, unsigned char *frm_buf, int data_size,
    int keyframe)
{
#ifndef NDEVIEW_SOURCE
  if (ofile->file_format == FILE_FORMAT_AVI) {
    return AVI_write_frame(ofile->avifile, frm_buf, data_size, keyframe);
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    return qtime_track_write_video(ofile->vqtrk, frm_buf, data_size, keyframe);
  }
#else
  x_put_pic(ofile->width, ofile->height, frm_buf);
//  x_dispatch_event();

  // avoid warnning, unnecessary
  ofile = NULL; data_size = keyframe = 0;
  // avoid warnning, unnecessary
#endif
  return 0;
}

int
write_audio_data(OUT_FILE *ofile, unsigned char *buf, int data_size, int num_samples)
{
#ifndef NDEVIEW_SOURCE
  if (ofile->file_format == FILE_FORMAT_AVI) {
    return AVI_write_audio(ofile->avifile, buf, data_size);
  } else if (ofile->file_format == FILE_FORMAT_MOV) {
    return qtime_track_write_audio(ofile->aqtrk, buf, data_size, num_samples);
  }
#else
  // avoid warnning, unnecessary
  ofile = NULL; buf = NULL; data_size = num_samples = 0;
  // avoid warnning, unnecessary
#endif
  return 0;
}

int
init_iofile(void)
{
  init_ifile();
  init_ofile();
  return 0;
}

void
quit_iofile(void)
{
  quit_ifile();
  quit_ofile();
}

