/* tkhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "matrix.h"
#include "tkhd.h"

void
tkhd_init(tkhd_t *tkhd)
{
  int i;

  atom_init(&tkhd->atom);
  tkhd->atom.type = QTIME_TYPE_TKHD;
  tkhd->version = 0;
//  qtime_flags_set(tkhd->flags, FLAG_TRACK_ENABLED);
  qtime_flags_set(tkhd->flags, 15);
  tkhd->creation_time = qtime_current_time();
  tkhd->modification_time = qtime_current_time();
  tkhd->track_id = 0;
  tkhd->reserved1 = 0;
  tkhd->duration = 0;
  for (i = 0; i < 8; i++) tkhd->reserved2[i] = 0;
  tkhd->layer = 0;
  tkhd->alternate_group = 0;
//  tkhd->volume = qtime_float_to_fixed16(0.996094);
  tkhd->volume = qtime_float_to_fixed16(0.);
  tkhd->reserved3 = 0;
  matrix_init(&tkhd->matrix);
  tkhd->track_width = qtime_float_to_fixed32(0);
  tkhd->track_height = qtime_float_to_fixed32(0);
}

void
tkhd_clean(tkhd_t *tkhd)
{
  qtime_error_type_check(QTIME_TYPE_TKHD, tkhd->atom.type);
  tkhd_init(tkhd);
}

tkhd_t*
tkhd_new(void)
{
  tkhd_t *tkhd;
  tkhd = (tkhd_t*)qtime_malloc(sizeof(tkhd_t));
  if (!tkhd)
    return NULL;
  tkhd_init(tkhd);
  return tkhd;
}

void
tkhd_delete(tkhd_t *tkhd)
{
  qtime_error_type_check(QTIME_TYPE_TKHD, tkhd->atom.type);
  qtime_free(tkhd);
}

tkhd_t*
tkhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, tkhd_t *tkhd_ptr)
{
  int i;
  tkhd_t *tkhd;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_TKHD, atom_head->type, NULL);

  if (tkhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_TKHD, tkhd_ptr->atom.type, NULL);
    tkhd = tkhd_ptr;
    tkhd_clean(tkhd);
  } else {
    if ((tkhd = tkhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  tkhd->atom.size = atom_head->size;
  tkhd->atom.type = QTIME_TYPE_TKHD;
  tkhd->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &tkhd->version, 1);
  qtime_io_read(qtio, tkhd->flags, 3);
  qtime_io_read32(qtio, &tkhd->creation_time);
  qtime_io_read32(qtio, &tkhd->modification_time);
  qtime_io_read32(qtio, &tkhd->track_id);
  qtime_io_read32(qtio, &tkhd->reserved1);
  qtime_io_read32(qtio, &tkhd->duration);
  qtime_io_read(qtio, tkhd->reserved2, 8);
  qtime_io_read16(qtio, &tkhd->layer);
  qtime_io_read16(qtio, &tkhd->alternate_group);
  qtime_io_read16(qtio, &tkhd->volume);
  qtime_io_read16(qtio, &tkhd->reserved3);
  for (i = 0; i < 9; i++)
    qtime_io_read32(qtio, &tkhd->matrix.values[i]);
  qtime_io_read32(qtio, &tkhd->track_width);
  qtime_io_read32(qtio, &tkhd->track_height);

  return tkhd;
}

tkhd_t*
tkhd_create(tkhd_t *tkhd_ptr)
{
  tkhd_t *tkhd;
  if (tkhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_TKHD, tkhd_ptr->atom.type, NULL);
    tkhd = tkhd_ptr;
    tkhd_clean(tkhd);
  } else {
    if ((tkhd = tkhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return tkhd;
}

int64_t
tkhd_calc_size(tkhd_t *tkhd)
{
  int64_t size;
  qtime_error_type_check_v(QTIME_TYPE_TKHD, tkhd->atom.type, 0);
  size = 8 + TKHD_PROP_SIZE;
  tkhd->atom.size = size;
  return size;
}

int
tkhd_write_atom(qtime_io_t *qtio, tkhd_t *tkhd)
{
  atom_head_t atom_head;
  int i;

  qtime_error_type_check_v(QTIME_TYPE_TKHD, tkhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM);

  atom_head_init(&atom_head);
  atom_head.size = tkhd->atom.size;
  atom_head.type = tkhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TKHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &tkhd->version, 1);
  qtime_io_write(qtio, tkhd->flags, 3);
  qtime_io_write32(qtio, &tkhd->creation_time);
  qtime_io_write32(qtio, &tkhd->modification_time);
  qtime_io_write32(qtio, &tkhd->track_id);
  qtime_io_write32(qtio, &tkhd->reserved1);
  qtime_io_write32(qtio, &tkhd->duration);
  qtime_io_write(qtio, tkhd->reserved2, 8);
  qtime_io_write16(qtio, &tkhd->layer);
  qtime_io_write16(qtio, &tkhd->alternate_group);
  qtime_io_write16(qtio, &tkhd->volume);
  qtime_io_write16(qtio, &tkhd->reserved3);
  for (i = 0; i < 9; i++)
    qtime_io_write32(qtio, &tkhd->matrix.values[i]);
  qtime_io_write32(qtio, &tkhd->track_width);
  qtime_io_write32(qtio, &tkhd->track_height);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TKHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
tkhd_dump(const char *parent_types, tkhd_t *tkhd)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;
  char buf[64];

  qtime_type_to_str(tkhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(tkhd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)tkhd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, tkhd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: creation time, %s", types, qtime_ctime_r(tkhd->creation_time,buf));
  fprintf(stdout, "%s: modification time, %s", types, qtime_ctime_r(tkhd->modification_time,buf));
  fprintf(stdout, "%s: track id     %d\n", types, tkhd->track_id);
  fprintf(stdout, "%s: duration     %u\n", types, tkhd->duration);
  fprintf(stdout, "%s: layer        %d\n", types, tkhd->layer);
  fprintf(stdout, "%s: alternate group %d\n", types, tkhd->alternate_group);
  fprintf(stdout, "%s: volume       %f\n", types, qtime_fixed16_to_float(tkhd->volume));
  for (i = 0; i < 9; i++)
    fprintf(stdout, "%s: matrix[%d]   %f\n", types, i, qtime_fixed32_to_float(tkhd->matrix.values[i]));
  fprintf(stdout, "%s: track width  %f\n", types, qtime_fixed32_to_float(tkhd->track_width));
  fprintf(stdout, "%s: track height %f\n", types, qtime_fixed32_to_float(tkhd->track_height));
}

int
tkhd_set_duration(tkhd_t *tkhd, uint32_t duration)
{
  qtime_error_type_check_v(QTIME_TYPE_TKHD, tkhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM);
  tkhd->duration = duration;
  return QTIME_OK;
}

