/* qtime_sample_desc.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "qtime_sample_desc.h"
#include "sample_desc.h"
#include "video_desc.h"
#include "sound_desc.h"


void
qtime_sample_sample_desc_to_desc(sample_desc_t *sample_desc, void *desc, uint32_t media_type)
{
  qtime_sample_desc_video_t *qvd;
  qtime_sample_desc_sound_t *qsd;
  video_desc_t *vd;
  sound_desc_t *sd;
  int len;
  char fcc[5];

  switch (media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      qvd = desc;
      vd = (video_desc_t*)sample_desc;
      qvd->dref_id = vd->sample_desc.data_reference_index;
      memcpy(qvd->data_format, &vd->sample_desc.data_format, 4);
      qvd->data_format[4] = '\0';
      qvd->version = vd->version;
      memcpy(qvd->vendor, &vd->vendor, 4);
      qvd->temporal_quality = vd->temporal_quality;
      qvd->spatial_quality = vd->spatial_quality;
      qvd->width = vd->width;
      qvd->height = vd->height;
      qvd->horizontal_resolution = qtime_fixed32_to_float(vd->horizontal_resolution);
      qvd->vertical_resolution = qtime_fixed32_to_float(vd->vertical_resolution);
      qvd->frame_count = vd->frame_count;
      len = vd->compressor_name[0];
      memcpy(qvd->compressor_name, &vd->compressor_name[1], len);
      qvd->compressor_name[len] = '\0';
      qvd->depth = vd->depth;
      qvd->color_table_id = vd->color_table_id;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      qsd = desc;
      sd = (sound_desc_t*)sample_desc;
      qsd->dref_id = sd->sample_desc.data_reference_index;
      memcpy(qsd->data_format, &sd->sample_desc.data_format, 4);
      qsd->data_format[4] = '\0';
      qsd->version = sd->version;
      memcpy(qsd->vendor, &sd->vendor, 4);
      qsd->number_of_channels = sd->number_of_channels;
      qsd->sample_size = sd->sample_size;
      qsd->compression_id = sd->compression_id;
      qsd->sample_rate = qtime_fixed32_to_float(sd->sample_rate);
      qsd->samples_per_packet = sd->samples_per_packet;
      qsd->bytes_per_packet = sd->bytes_per_packet;
      qsd->bytes_per_frame = sd->bytes_per_frame;
      qsd->bytes_per_sample = sd->bytes_per_sample;
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_type_to_str(media_type, fcc);
      fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: unsupported media_type '%4s'.\n", fcc);
      break;
  }
  return;
}

void
qtime_sample_desc_to_sample_desc(void *desc, sample_desc_t *sample_desc, uint32_t media_type)
{
  qtime_sample_desc_video_t *qvd;
  qtime_sample_desc_sound_t *qsd;
  video_desc_t *vd;
  sound_desc_t *sd;
  int len;
  char fcc[5];

  switch (media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      qvd = desc;
      vd = (video_desc_t*)sample_desc;
      vd->sample_desc.data_reference_index = qvd->dref_id;
      memcpy(&vd->sample_desc.data_format, qvd->data_format, 4);
      vd->version = qvd->version;
      memcpy(&vd->vendor, qvd->vendor, 4);
      vd->temporal_quality = qvd->temporal_quality;
      vd->spatial_quality = qvd->spatial_quality;
      vd->width = qvd->width;
      vd->height = qvd->height;
      vd->horizontal_resolution = qtime_float_to_fixed32(qvd->horizontal_resolution);
      vd->vertical_resolution = qtime_float_to_fixed32(qvd->vertical_resolution);
      vd->frame_count = qvd->frame_count;
      len = strlen(qvd->compressor_name);
      memcpy(&vd->compressor_name, qvd->compressor_name, len);
      vd->compressor_name[0] = len;
      vd->depth = qvd->depth;
      vd->color_table_id = qvd->color_table_id;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      qsd = desc;
      sd = (sound_desc_t*)sample_desc;
      sd->sample_desc.data_reference_index = qsd->dref_id;
      memcpy(&sd->sample_desc.data_format, qsd->data_format, 4);
      sd->version = qsd->version;
      memcpy(&sd->vendor, qsd->vendor, 4);
      sd->number_of_channels = qsd->number_of_channels;
      sd->sample_size = qsd->sample_size;
      sd->compression_id = qsd->compression_id;
      sd->sample_rate = qtime_float_to_fixed32(qsd->sample_rate);
      sd->samples_per_packet = qsd->samples_per_packet;
      sd->bytes_per_packet = qsd->bytes_per_packet;
      sd->bytes_per_frame = qsd->bytes_per_frame;
      sd->bytes_per_sample = qsd->bytes_per_sample;
    default:
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_type_to_str(media_type, fcc);
      fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: unsupported media_type '%4s'.\n", fcc);
      break;
  }
  return;
}

sample_desc_t*
qtime_sample_get_first_desc(qtime_sample_t *qtsp)
{
  sample_desc_t *sample_desc;

  sample_desc = stsd_get_sample_desc(qtsp->stsd, 1);
  if (!sample_desc) {
    switch (qtsp->media_type) {
      case QTIME_MEDIA_TYPE_VIDEO:
        sample_desc = (sample_desc_t*)video_desc_create(NULL);
        break;
      case QTIME_MEDIA_TYPE_SOUND:
        sample_desc = (sample_desc_t*)sound_desc_create(NULL);
        break;
      default:
        qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
        qtime_error_invalid_media(QTIME_TYPE_QTSP,qtsp->media_type,QTIME_MEDIA_TYPE_UNKNOWN);
	sample_desc = NULL;
        break;
    }
    stsd_add_sample_desc(qtsp->stsd, sample_desc);
  }
  return sample_desc;
}

int
qtime_sample_set_width(qtime_sample_t *qtsp, int width)
{
  sample_desc_t *sample_desc;
  video_desc_t *vd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_VIDEO);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_VIDEO) {
    fprintf(stderr, "qtime_sample_set_width: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  vd = (video_desc_t*)sample_desc;
  vd->width = width;

  return QTIME_OK;
}

int
qtime_sample_set_height(qtime_sample_t *qtsp, int height)
{
  sample_desc_t *sample_desc;
  video_desc_t *vd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_VIDEO);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_VIDEO) {
    fprintf(stderr, "qtime_sample_set_height: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  vd = (video_desc_t*)sample_desc;
  vd->height = height;

  return QTIME_OK;
}

int
qtime_sample_set_compressor(qtime_sample_t *qtsp, uint8_t *compressor, uint32_t media_type)
{
  sample_desc_t *sample_desc;

  if (media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    fprintf(stderr, "qtime_sample_set_compressor: unknown media_type.\n");
    return QTIME_ERROR;
  }
  if (media_type != QTIME_MEDIA_TYPE_VIDEO &&
      media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_compressor: unsupported media_type.\n");
    return QTIME_ERROR;
  }


  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_sample_set_media_type(qtsp, media_type);
  } else if (qtsp->media_type != media_type) {
    fprintf(stderr, "qtime_sample_set_compressor: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sample_desc->data_format = *((uint32_t*)compressor);

  if (media_type == QTIME_MEDIA_TYPE_SOUND) {
    qtsp->sound_desc = (sound_desc_t*)sample_desc;
    qtsp->bytespspl = qtsp->sound_desc->number_of_channels * (qtsp->sound_desc->sample_size/8);
  }
  qtsp->desc_id = 1;

  return QTIME_OK;
}

int
qtime_sample_set_sound_channels(qtime_sample_t *qtsp, int channels)
{
  sample_desc_t *sample_desc;
  sound_desc_t *sd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_SOUND);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_sound_channels: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sd = (sound_desc_t*)sample_desc;
  sd->number_of_channels = channels;

  qtsp->desc_id = 1;
  qtsp->sound_desc = sd;
  qtsp->bytespspl = sd->number_of_channels * (sd->sample_size/8);

  return QTIME_OK;
}

int
qtime_sample_set_sound_bits(qtime_sample_t *qtsp, int bits)
{
  sample_desc_t *sample_desc;
  sound_desc_t *sd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_SOUND);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_sound_bits: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sd = (sound_desc_t*)sample_desc;
  sd->sample_size = bits;

  qtsp->sound_desc = sd;
  qtsp->bytespspl = qtsp->sound_desc->number_of_channels * (qtsp->sound_desc->sample_size/8);
  qtsp->desc_id = 1;

  return QTIME_OK;
}

int
qtime_sample_set_sound_rate(qtime_sample_t *qtsp, int rate)
{
  sample_desc_t *sample_desc;
  sound_desc_t *sd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_SOUND);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_sound_rate: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sd = (sound_desc_t*)sample_desc;
  sd->sample_rate = qtime_float_to_fixed32((float)rate);

  qtsp->sound_desc = sd;
  qtsp->bytespspl = qtsp->sound_desc->number_of_channels * (qtsp->sound_desc->sample_size/8);
  qtsp->desc_id = 1;

  return QTIME_OK;
}

int
qtime_sample_set_sound_ext(qtime_sample_t *qtsp, qtime_sample_desc_sound_ext_t *sound_ext)
{
  sample_desc_t *sample_desc;
  sound_desc_t *sd;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_SOUND);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_sound_ext: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sd = (sound_desc_t*)sample_desc;
  sd->version = sound_ext->version;
  sd->compression_id = sound_ext->compression_id;
  sd->samples_per_packet = sound_ext->samples_per_packet;
  sd->bytes_per_packet = sound_ext->bytes_per_packet;
  sd->bytes_per_frame = sound_ext->bytes_per_frame;
  sd->bytes_per_sample = sound_ext->bytes_per_sample;

  qtsp->sound_desc = sd;
  qtsp->bytespspl = qtsp->sound_desc->number_of_channels * (qtsp->sound_desc->sample_size/8);
  qtsp->desc_id = 1;

  return QTIME_OK;
}

int
qtime_sample_get_sound_ext(qtime_sample_t *qtsp, qtime_sample_desc_sound_ext_t *sound_ext)
{
  sample_desc_t *sample_desc;
  sound_desc_t *sd;

  sound_ext->version = 0;
  sound_ext->compression_id = 0;
  sound_ext->samples_per_packet = 0;
  sound_ext->bytes_per_packet = 0;
  sound_ext->bytes_per_frame = 0;
  sound_ext->bytes_per_sample = 0;

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtime_sample_set_media_type(qtsp, QTIME_MEDIA_TYPE_SOUND);

  if (qtsp->media_type != QTIME_MEDIA_TYPE_SOUND) {
    fprintf(stderr, "qtime_sample_set_sound_ext: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }

  sample_desc = qtime_sample_get_first_desc(qtsp);
  sd = (sound_desc_t*)sample_desc;
  sound_ext->version = sd->version;
  sound_ext->compression_id = sd->compression_id;
  if (sd->version > 0) {
    sound_ext->samples_per_packet = sd->samples_per_packet;
    sound_ext->bytes_per_packet = sd->bytes_per_packet;
    sound_ext->bytes_per_frame = sd->bytes_per_frame;
    sound_ext->bytes_per_sample = sd->bytes_per_sample;
  }

  qtsp->sound_desc = sd;
  qtsp->bytespspl = qtsp->sound_desc->number_of_channels * (qtsp->sound_desc->sample_size/8);
  qtsp->desc_id = 1;

  return QTIME_OK;
}

