/* mvhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "matrix.h"
#include "mvhd.h"

void
mvhd_init(mvhd_t *mvhd)
{
  int i;
  atom_init(&mvhd->atom);
  mvhd->atom.type = QTIME_TYPE_MVHD;
  mvhd->version = 0;
  qtime_flags_set(mvhd->flags, 0);
//  mvhd->creation_time = qtime_current_time();
  mvhd->creation_time = 0;
//  mvhd->modification_time = qtime_current_time();
  mvhd->modification_time = 0;
  mvhd->time_scale = 0;
  mvhd->duration = 0;
  mvhd->preferred_rate = qtime_float_to_fixed32(1.0);
  mvhd->preferred_volume = qtime_float_to_fixed16(0.996094);
  for (i = 0; i < 10; i++) mvhd->reserved[i] = 0;
  matrix_init(&mvhd->matrix);
  mvhd->preview_time = 0;
  mvhd->preview_duration = 0;
  mvhd->poster_time = 0;
  mvhd->selection_time = 0;
  mvhd->selection_duration = 0;
  mvhd->current_time = 0;
  mvhd->next_track_id = 1;
}

void
mvhd_clean(mvhd_t *mvhd)
{
  qtime_error_type_check(QTIME_TYPE_MVHD, mvhd->atom.type)
  mvhd_init(mvhd);
}

mvhd_t*
mvhd_new(void)
{
  mvhd_t *mvhd;
  mvhd = qtime_malloc(sizeof(mvhd_t));
  if (!mvhd) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return NULL;
  }
  mvhd_init(mvhd);
  return mvhd;
}

void
mvhd_delete(mvhd_t *mvhd)
{
  qtime_error_type_check(QTIME_TYPE_MVHD, mvhd->atom.type)
  qtime_free(mvhd);
}

mvhd_t*
mvhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, mvhd_t *mvhd_ptr)
{
  int i;
  mvhd_t *mvhd;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_MVHD, atom_head->type, NULL)
  if (mvhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd_ptr->atom.type, NULL)
    mvhd = mvhd_ptr;
    mvhd_clean(mvhd);
  } else {
    if ((mvhd = mvhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  mvhd->atom.size = atom_head->size;
  mvhd->atom.type = QTIME_TYPE_MVHD;
  mvhd->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &mvhd->version, 1);
  qtime_io_read(qtio, mvhd->flags, 3);
  qtime_io_read32(qtio, &mvhd->creation_time);
  qtime_io_read32(qtio, &mvhd->modification_time);
  qtime_io_read32(qtio, &mvhd->time_scale);
  qtime_io_read32(qtio, &mvhd->duration);
  qtime_io_read32(qtio, &mvhd->preferred_rate);
  qtime_io_read16(qtio, &mvhd->preferred_volume);
  qtime_io_read(qtio, mvhd->reserved, 10);

  for (i = 0; i < 9; i++) {
    qtime_io_read32(qtio, &mvhd->matrix.values[i]);
  }

  qtime_io_read32(qtio, &mvhd->preview_time);
  qtime_io_read32(qtio, &mvhd->preview_duration);
  qtime_io_read32(qtio, &mvhd->poster_time);
  qtime_io_read32(qtio, &mvhd->selection_time);
  qtime_io_read32(qtio, &mvhd->selection_duration);
  qtime_io_read32(qtio, &mvhd->current_time);
  qtime_io_read32(qtio, &mvhd->next_track_id);

  return mvhd;
}

mvhd_t*
mvhd_create(mvhd_t *mvhd_ptr)
{
  mvhd_t *mvhd;

  if (mvhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd_ptr->atom.type, NULL)
    mvhd = mvhd_ptr;
    mvhd_clean(mvhd);
  } else {
    if ((mvhd = mvhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return mvhd;
}

int64_t
mvhd_calc_size(mvhd_t *mvhd)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, 0)
  mvhd->atom.size = 8 + MVHD_PROP_SIZE;
  return mvhd->atom.size;
}

int
mvhd_write_atom(qtime_io_t *qtio, mvhd_t *mvhd)
{
  atom_head_t atom_head;
  int i;

  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = mvhd->atom.size;
  atom_head.type = mvhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MVHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &mvhd->version, 1);
  qtime_io_write(qtio, mvhd->flags, 3);
  qtime_io_write32(qtio, &mvhd->creation_time);
  qtime_io_write32(qtio, &mvhd->modification_time);
  qtime_io_write32(qtio, &mvhd->time_scale);
  qtime_io_write32(qtio, &mvhd->duration);
  qtime_io_write32(qtio, &mvhd->preferred_rate);
  qtime_io_write16(qtio, &mvhd->preferred_volume);
  qtime_io_write(qtio, mvhd->reserved, 10);

  for (i = 0; i < 9; i++) {
    qtime_io_write32(qtio, &mvhd->matrix.values[i]);
  }

  qtime_io_write32(qtio, &mvhd->preview_time);
  qtime_io_write32(qtio, &mvhd->preview_duration);
  qtime_io_write32(qtio, &mvhd->poster_time);
  qtime_io_write32(qtio, &mvhd->selection_time);
  qtime_io_write32(qtio, &mvhd->selection_duration);
  qtime_io_write32(qtio, &mvhd->current_time);
  qtime_io_write32(qtio, &mvhd->next_track_id);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MVHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
mvhd_dump(const char *parent_types, mvhd_t *mvhd)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;
  char buf[64];

  qtime_type_to_str(mvhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(mvhd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)mvhd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, mvhd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
//  fprintf(stdout, "%s: creation time %u\n", types, mvhd->creation_time);
  fprintf(stdout, "%s: creation time, %s", types, qtime_ctime_r(mvhd->creation_time,buf));
//  fprintf(stdout, "%s: modification time %u\n", types, mvhd->modification_time);
  fprintf(stdout, "%s: modification time, %s", types, qtime_ctime_r(mvhd->modification_time,buf));
  fprintf(stdout, "%s: time scale   %u\n", types, mvhd->time_scale);
  fprintf(stdout, "%s: duration     %u\n", types, mvhd->duration);
  fprintf(stdout, "%s: preferred rate %f\n", types, qtime_fixed32_to_float(mvhd->preferred_rate));
  fprintf(stdout, "%s: preferred volume %f\n", types, qtime_fixed16_to_float(mvhd->preferred_volume));
  for (i = 0; i < 9; i++)
    fprintf(stdout, "%s: matrix[%d]   %f\n", types, i, qtime_fixed32_to_float(mvhd->matrix.values[i]));
  fprintf(stdout, "%s: preview time %u\n", types, mvhd->preview_time);
  fprintf(stdout, "%s: preview duration %u\n", types, mvhd->preview_duration);
  fprintf(stdout, "%s: poster time  %u\n", types, mvhd->poster_time);
  fprintf(stdout, "%s: selection time %u\n", types, mvhd->selection_time);
  fprintf(stdout, "%s: selection duration %u\n", types, mvhd->selection_duration);
  fprintf(stdout, "%s: current time %u\n", types, mvhd->current_time);
  fprintf(stdout, "%s: next track id %u\n", types, mvhd->next_track_id);
}

int
mvhd_set_next_track_id(mvhd_t *mvhd, int next_track_id)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  if (next_track_id <= 0) return QTIME_ERROR;
  mvhd->next_track_id = next_track_id;
  return next_track_id;
}

int
mvhd_get_next_track_id(mvhd_t *mvhd)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  return mvhd->next_track_id;
}

int
mvhd_set_time_scale(mvhd_t *mvhd, uint32_t time_scale)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  mvhd->time_scale = time_scale;
  return QTIME_OK;
}

int
mvhd_set_duration(mvhd_t *mvhd, uint32_t duration)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  mvhd->duration = duration;
  return QTIME_OK;
}

int
mvhd_update_modification_time(mvhd_t *mvhd)
{
  qtime_error_type_check_v(QTIME_TYPE_MVHD, mvhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  mvhd->modification_time = qtime_current_time();
  if (mvhd->creation_time == 0)
    mvhd->creation_time = mvhd->modification_time;
  return QTIME_OK;
}

