/* elst.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "elst.h"

void
elst_init(elst_t *elst)
{
  atom_init(&elst->atom);
  elst->atom.type = QTIME_TYPE_ELST;
  elst->version = 0;
  qtime_flags_set(elst->flags, 0);
  elst->number_of_entries = 0;
  elst->edit_list = NULL;
}

void
elst_clean(elst_t *elst)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return;
  }
#endif
  if (elst->edit_list)
    qtime_free(elst->edit_list);
  elst_init(elst);
}

elst_t*
elst_new(void)
{
  elst_t *elst;
  elst = (elst_t*)qtime_malloc(sizeof(elst_t));
  if (!elst) return NULL;
  elst_init(elst);
  return elst;
}

void
elst_delete(elst_t *elst)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return;
  }
#endif
  if (elst->edit_list)
    qtime_free(elst->edit_list);
  qtime_free(elst);
}

elst_t*
elst_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, elst_t *elst_ptr)
{
  elst_t *elst;
  int i, num_ent, table_byte_size;
  edit_list_t *table;

#ifdef DEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (elst_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_ELST, elst_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    elst = elst_ptr;
    elst_clean(elst);
  } else {
    if ((elst = elst_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
 
  elst->atom.size = atom_head->size;
  elst->atom.type = QTIME_TYPE_ELST;
  elst->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &elst->version, 1);
  qtime_io_read(qtio, elst->flags, 3);
  qtime_io_read32(qtio, &elst->number_of_entries);

  if (elst->number_of_entries == 0)
    return elst;

  num_ent = elst->number_of_entries;
  table_byte_size = sizeof(edit_list_t) * num_ent;
  table = (edit_list_t*)qtime_malloc(table_byte_size);
  if (!table) {
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  elst->edit_list = table;

  for (i = 0; i < num_ent; i++) {
    qtime_io_read32(qtio, &table[i].track_duration);
    qtime_io_read32(qtio, &table[i].media_time);
    qtime_io_read32(qtio, &table[i].media_rate);
  }

  return elst;

fail:
  if (elst_ptr)
    elst_clean(elst);
  else
    elst_delete(elst);
  qtime_error_atom_read(QTIME_TYPE_ELST);
  return NULL;
}

elst_t*
elst_create(elst_t *elst_ptr)
{
  elst_t *elst;

  if (elst_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_ELST, elst_ptr->atom.type)) {
      return NULL;
    }
#endif
    elst = elst_ptr;
    elst_clean(elst);
  } else {
    if ((elst = elst_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return elst;
}

int64_t
elst_calc_size(elst_t *elst)
{
  int64_t size = 0;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return 0;
  }
#endif

  if (elst->number_of_entries <= 0)
    return 0;

  size = ELST_PROP_SIZE;
  size += elst->number_of_entries * EDIT_LIST_SIZE;
  if (size & SIZE64MASK)
    size += 8 + 8;
  else
    size += 8;
  elst->atom.size = size;
  return size;
}

int
elst_write_atom(qtime_io_t *qtio, elst_t *elst)
{
  atom_head_t atom_head;
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (elst->number_of_entries <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = elst->atom.size;
  atom_head.type = elst->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_ELST);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &elst->version, 1);
  qtime_io_write(qtio, elst->flags, 3);
  qtime_io_write32(qtio, &elst->number_of_entries);
  for (i = 0; i < elst->number_of_entries; i++) {
    qtime_io_write32(qtio, &elst->edit_list[i].track_duration);
    qtime_io_write32(qtio, &elst->edit_list[i].media_time);
    qtime_io_write32(qtio, &elst->edit_list[i].media_rate);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_ELST);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
elst_dump(const char *parent_types, elst_t *elst)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;
  int i;

//  if (elst->number_of_entries <= 0)
//    return;

  qtime_type_to_str(elst->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(elst->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)elst->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, elst->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number of entries %d\n", types, elst->number_of_entries);
  for (i = 0; i < elst->number_of_entries; i++) {
    fprintf(stdout, "%s: list[%d] track_duration %u, media_time %u, media_rate %f\n", types, i, elst->edit_list[i].track_duration, elst->edit_list[i].media_time, qtime_fixed32_to_float(elst->edit_list[i].media_rate));
  }
}

int
elst_add_table(elst_t *elst, uint32_t track_duration, uint32_t media_time, float media_rate)
{
  int num_ent;
  edit_list_t *elst_table;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  num_ent = elst->number_of_entries;
  elst_table = (edit_list_t*)qtime_realloc(elst->edit_list,
      sizeof(edit_list_t) * (num_ent+1));
  if (!elst_table) {
    return QTIME_ERROR_MEMORY;
  }
  elst_table[num_ent].track_duration = track_duration;
  elst_table[num_ent].media_time     = media_time;
  elst_table[num_ent].media_rate     = qtime_float_to_fixed32(media_rate);
  elst->edit_list = elst_table;
  elst->number_of_entries++;
  return QTIME_OK;
}

int
elst_get_number_of_entries(elst_t *elst)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_ELST, elst->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif
  return elst->number_of_entries;
}

