/* atom.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "mdat.h"


void
atom_init(atom_t *atom)
{
  atom->size             = 0;
  atom->type             = 0;
  atom->flag             = 0;
  atom->index            = 0;
  atom->number_of_childs = 0;
  atom->childs           = NULL;
  atom->parent           = NULL;
}

void
atom_clean(atom_t *atom)
{
#if 0
#ifndef NDEBUG
  char fcc[5];
//  if (atom->childs || atom->number_of_childs) {
  if (atom->number_of_childs) {
    qtime_type_to_str(atom->type, fcc);
    fprintf(stderr, "QTIME_ERROR: childs are remains, %d, %p, in '%4s' atom.\n", atom->number_of_childs, atom->childs, fcc);
  }
#endif
#endif
  if (atom->childs)
    qtime_free(atom->childs);
  atom->size             = 0;
  atom->type             = 0;
  atom->flag             = 0;
  atom->index            = 0;
  atom->number_of_childs = 0;
  atom->childs           = NULL;
  atom->parent           = NULL;
}

void
atom_head_init(atom_head_t *atom_head)
{
  atom_head->size         = 0;
  atom_head->type         = 0;
  atom_head->media_type   = 0;
  atom_head->body_size    = 0;
  atom_head->start_offset = 0;
  atom_head->end_offset   = 0;
  atom_head->parent       = NULL;
  atom_head->flag         = 0;
  atom_head->error_code   = QTIME_OK;
}

int
atom_read_header(qtime_io_t *qtio, atom_head_t *atom_head)
{
  uint32_t size32;

  atom_head->size = 0;
  atom_head->type = QTIME_TYPE_UNKNOWN;
  atom_head->start_offset = qtime_io_get_offset(qtio);
  if (atom_head->start_offset == -1)
    return QTIME_ERROR_IO;
  else if (atom_head->start_offset >= qtio->end_offset-4)
    return QTIME_END_OF_FILE;
  atom_head->end_offset = atom_head->start_offset;
  qtime_io_read32(qtio, &size32);
  qtime_io_read_type(qtio, &atom_head->type);
  if (size32 == 1) {
    qtime_io_read64(qtio, &atom_head->size);
    atom_head->end_offset = atom_head->start_offset + atom_head->size;
    atom_head->body_size = atom_head->size - 4 - 4 - 8;
  } else if (size32 == 0) {
    atom_head->end_offset = qtime_io_end_offset(qtio);
    atom_head->size = atom_head->end_offset - atom_head->start_offset;
    atom_head->body_size = atom_head->size - 4 - 4;
  } else {
    atom_head->size = size32;
    atom_head->end_offset = atom_head->start_offset + atom_head->size;
    atom_head->body_size = atom_head->size - 4 - 4;
  }

//  fprintf(stderr, "'%.4s' size = %lld.\n", (char*)&atom_head->type, atom_head->size);

#ifndef NDEBUG
#if ATOM_SIZE_T_LENGTH == 4
  if (atom_head->size & SIZE64MASK && atom_head->type != QTIME_TYPE_MDAT) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: atom size too large, %lld, atom size max %u.\n", atom_head->size, ATOM_SIZE_MAX);
  }
#endif
#endif

  return QTIME_OK;
}

int
atom_read_footer(qtime_io_t *qtio, atom_head_t *atom_head)
{
//  if (atom_head->type == QTIME_TYPE_UNKNOWN || atom_head->size == 0)
  if (atom_head->size == 0)
    return QTIME_OK;

  if (qtime_io_set_offset(qtio, atom_head->end_offset)!=atom_head->end_offset) {
    qtime_error_debug_info(QTIME_ERROR_IO);
    qtime_error_io(qtio->name);
    return QTIME_ERROR_IO;
  }
  return QTIME_OK;
}

int
atom_write_header(qtime_io_t *qtio, atom_head_t *atom_head)
{
  uint32_t size32;

#ifndef NDEBUG
#if ATOM_SIZE_T_LENGTH == 4
  if (atom_head->size & SIZE64MASK) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: atom size too large, %lld, atom size max %u.\n", atom_head->size, ATOM_SIZE_MAX);
  }
#endif
#endif

  atom_head->start_offset = qtime_io_get_offset(qtio);
  if (atom_head->start_offset == -1)
    return QTIME_ERROR_IO;

  if (atom_head->size & SIZE64MASK) {
    size32 = 1;
    qtime_io_write32(qtio, &size32);
    qtime_io_write_type(qtio, &atom_head->type);
    qtime_io_write64(qtio, &atom_head->size);
  } else {
    size32 = (uint32_t)atom_head->size;
    qtime_io_write32(qtio, &size32);
    qtime_io_write_type(qtio, &atom_head->type);
  }

  return QTIME_OK;
}

int
atom_write_footer(qtime_io_t *qtio, atom_head_t *atom_head)
{
  uint32_t size32;
  int64_t size;

  if (atom_head->start_offset < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(atom_head->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  atom_head->end_offset = qtime_io_get_offset(qtio);
  if (atom_head->end_offset < -1) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(atom_head->type);
    return QTIME_ERROR_IO;
  }

  size = atom_head->end_offset - atom_head->start_offset;
#ifndef NDEBUG
  if (size < 0) {
    fprintf(stderr, "QTIME_ERROR: atom_write_footer: size is negative %lld\n", size);
    return QTIME_ERROR_ATOM_WRITE;
  }

#if ATOM_SIZE_T_LENGTH == 4
  if (size & SIZE64MASK) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: atom size too large, %lld, atom size max %u.\n", size, ATOM_SIZE_MAX);
  }
#endif
#endif

  if (size != atom_head->size) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: '%.4s' atom size doesn't suit, %lld, %lld.\n", (char*)&atom_head->type, atom_head->size, size);
#endif
    if (qtime_io_set_offset(qtio, atom_head->start_offset) !=
	atom_head->start_offset) {
      qtime_error_debug_info(QTIME_ERROR_IO);
      qtime_error_io(qtio->name);
      return QTIME_ERROR_IO;
    }

    if (atom_head->size & SIZE64MASK) {
      size32 = 1;
      qtime_io_write32(qtio, &size32);
      qtime_io_write_type(qtio, &atom_head->type);
      qtime_io_write64(qtio, &size);
    } else {
      if (size & SIZE64MASK) {
        qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
        qtime_error_io(qtio->name);
	fprintf(stderr, "QTIME_ERROR: atom_write_footer: cannot write atom header, %lld, %lld.\n", atom_head->size, size);
        return QTIME_ERROR_ATOM_WRITE;
      }
      size32 = (uint32_t)size;
      qtime_io_write32(qtio, &size32);
      qtime_io_write_type(qtio, &atom_head->type);
    }

    if (qtime_io_set_offset(qtio, atom_head->end_offset) !=
	atom_head->end_offset) {
      qtime_error_debug_info(QTIME_ERROR_IO);
      qtime_error_io(qtio->name);
      return QTIME_ERROR_IO;
    }
  }

  return QTIME_OK;
}

void
atom_list_free(atom_list_t atom_list)
{
  if (!atom_list) return;
  qtime_free(atom_list);
}

int
atom_add_child(atom_t *parent, atom_t *child)
{
  atom_list_t atom_list;
  int num_atom;

#ifndef NDEBUG
  if (parent == NULL || child == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: atom_add_child(): parent or child is NULL, %p, %p.\n", parent, child);
//    return QTIME_ERROR;
    exit(1);
  }
#endif
  num_atom = parent->number_of_childs;
  atom_list = parent->childs;
  atom_list =(atom_list_t)qtime_realloc(atom_list,sizeof(atom_t*)*(num_atom+1));
  if (!atom_list) return QTIME_ERROR_MEMORY;
  atom_list[num_atom] = child;
  num_atom++;
  parent->childs = atom_list;
  parent->number_of_childs = num_atom;
  child->parent = parent;

  return num_atom;
}

int
atom_insert_child(atom_t *parent, atom_t *child, int idx)
{
  int i;
  atom_list_t atom_list;
  int num_atom;

#ifndef NDEBUG
  if (parent == NULL || child == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: atom_add_child(): parent or child is NULL, %p, %p.\n", parent, child);
//    return QTIME_ERROR;
    exit(1);
  }
#endif
  if (idx < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: atom_add_child(): idx is negative, %d.\n", idx);
    idx = 0;
  }

  num_atom = parent->number_of_childs;
  atom_list = parent->childs;

  num_atom++;
  atom_list =(atom_list_t)qtime_realloc(atom_list,sizeof(atom_t*)*(num_atom));
  if (!atom_list) return QTIME_ERROR_MEMORY;

  for (i = num_atom-1; i > idx; i--) {
    atom_list[i] = atom_list[i-1];
  }
  atom_list[i] = child;

  parent->childs = atom_list;
  parent->number_of_childs = num_atom;
  child->parent = parent;

  return num_atom;
}

int
atom_remove_child(atom_t *parent, atom_t *child)
{
  atom_list_t atom_list;
  int num_atom;
  int i;

#ifndef NDEBUG
  if (parent == NULL || child == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: atom_remove_child(): parent or child is NULL, %p, %p.\n", parent, child);
//    return QTIME_ERROR;
    exit(1);
  }
#endif

  num_atom = parent->number_of_childs;
  atom_list = parent->childs;
  for (i = 0; i < num_atom; i++) {
    if (atom_list[i] == child)
      break;
  }
  if (i == num_atom) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: atom_remove_child(): cannot find child.\n");
    return QTIME_ERROR;
  }
  num_atom--;
  if (num_atom <= 0) {
    qtime_free(parent->childs);
    parent->childs = NULL;
    parent->number_of_childs = 0;
    return 0;
  }
  for (; i < num_atom; i++)
    atom_list[i] = atom_list[i+1];

  atom_list =(atom_list_t)qtime_realloc(atom_list,sizeof(atom_t*)*(num_atom));
  if (!atom_list) return QTIME_ERROR_MEMORY;
  parent->childs = atom_list;
  parent->number_of_childs = num_atom;

  return num_atom;
}

int
atom_remove_child_all(atom_t *atom)
{
  if (atom->childs)
    qtime_free(atom->childs);
  atom->number_of_childs = 0;
  atom->childs = NULL;
  return 0;
}

atom_t*
atom_get_child(atom_t *parent, int idx)
{
  if (idx < 0 || idx >= parent->number_of_childs)
    return NULL;
  return parent->childs[idx];
}

atom_t*
atom_find(atom_t *atom, uint32_t type, int flag)
{
  int i;
  atom_t *atom_ret;

  for (i = 0; i < atom->number_of_childs; i++)
    if (atom->childs[i]->type == type)
      return atom->childs[i];
    else if (flag == ATOM_FIND_RECURSIVE)
      if ((atom_ret = atom_find(atom->childs[i], type, flag)) != NULL)
	return atom_ret;
  return NULL;
}

atom_t*
atom_find_child(atom_t *atom, uint32_t type)
{
  int i;
  for (i = 0; i < atom->number_of_childs; i++)
    if (atom->childs[i]->type == type)
      return atom->childs[i];
  return NULL;
}

atom_t*
atom_find_recursive(atom_t *atom, uint32_t type)
{
  int i;
  atom_t *atom_ret;
  for (i = 0; i < atom->number_of_childs; i++)
    if (atom->childs[i]->type == type)
      return atom->childs[i];
    else
      if ((atom_ret = atom_find_recursive(atom->childs[i], type)) != NULL)
	return atom_ret;
  return NULL;
}

static int
find_atoms(atom_t *parent, atom_t *atoms)
{
  int i, ret;

  for (i = 0; i < parent->number_of_childs; i++) {
    if (atoms->type == QTIME_TYPE_UNKNOWN ||
       	atoms->type == parent->childs[i]->type) {
      if ((ret = atom_add_child(atoms, parent->childs[i])) < 0) {
	//atom_remove_child_all(atoms);
	return ret;
      }
    }
    if (atoms->flag == ATOM_FIND_RECURSIVE) {
      if ((ret = find_atoms(parent->childs[i], atoms)) != QTIME_OK)
	return ret;
    }
  }
  return QTIME_OK;
}

atom_list_t
atom_find_atoms(atom_t *atom, uint32_t type, uint32_t flag, int *num_atoms_ret)
{
  int ret;
  atom_t atoms;

  atom_init(&atoms);
  atoms.type = type;
  atoms.flag = flag;

  if ((ret = find_atoms(atom, &atoms)) != QTIME_OK) {
    atom_remove_child_all(&atoms);
    *num_atoms_ret = ret;
    return NULL;
  }
  *num_atoms_ret = atoms.number_of_childs;
  return atoms.childs;
}

atom_list_t
atom_find_child_atoms(atom_t *atom, uint32_t type, int *num_atoms_ret)
{
  int i, ret;
  atom_t atoms;
  atom_list_t childs;

  atoms.number_of_childs = 0;
  atoms.childs = NULL;
  //atom_init(&atoms);
  childs = atom->childs;

  for (i = 0; i < atom->number_of_childs; i++) {
    if (type == QTIME_TYPE_UNKNOWN || type == childs[i]->type) {
      if ((ret = atom_add_child(&atoms, childs[i])) < 0) {
	atom_remove_child_all(&atoms);
	*num_atoms_ret = ret;
	return NULL;
      }
    }
  }
  *num_atoms_ret = atoms.number_of_childs;
  return atoms.childs;
}

static int
find_recursive_atoms(atom_t *parent, atom_t *atoms)
{
  int i, ret;

  for (i = 0; i < parent->number_of_childs; i++) {
    if (atoms->type == QTIME_TYPE_UNKNOWN ||
       	atoms->type == parent->childs[i]->type) {
      if ((ret = atom_add_child(atoms, parent->childs[i])) < 0) {
	//atom_remove_child_all(atoms);
	return ret;
      }
    }
    if ((ret = find_recursive_atoms(parent->childs[i], atoms)) != QTIME_OK)
      return ret;
  }
  return QTIME_OK;
}

atom_list_t
atom_find_recursive_atoms(atom_t *atom, uint32_t type, int *num_atoms_ret)
{
  int ret;
  atom_t atoms;

  atoms.number_of_childs = 0;
  atoms.childs = NULL;
  //atom_init(&atoms);
  atoms.type = type;

  if ((ret = find_recursive_atoms(atom, &atoms)) != QTIME_OK) {
    atom_remove_child_all(&atoms);
    *num_atoms_ret = ret;
    return NULL;
  }
  *num_atoms_ret = atoms.number_of_childs;
  return atoms.childs;
}

int
atom_count_child(atom_t *atom, uint32_t type)
{
  int i;
  int count = 0;

  for (i = 0; i < atom->number_of_childs; i++) {
    if (type == QTIME_TYPE_UNKNOWN)
      count++;
    else if (atom->childs[i]->type == type)
      count++;
  }
  return count;
}

int
atom_count_recursive(atom_t *atom, uint32_t type)
{
  int i;
  int count = 0;

  for (i = 0; i < atom->number_of_childs; i++) {
    if (type == QTIME_TYPE_UNKNOWN)
      count++;
    else if (atom->childs[i]->type == type)
      count++;
    count += atom_count_recursive(atom->childs[i], type);
  }
  return count;
}

#if 0
int
atom_find_atoms_child(atom_t *parent, uint32_t type, atom_t *atoms)
{
  int i, ret;
  atom_list_t childs;

  atom_remove_child_all(atoms);
  atoms->type = type;
  //atom_clean(atoms);
  childs = parent->childs;

  for (i = 0; i < parent->number_of_childs; i++) {
    if (type == QTIME_TYPE_UNKNOWN || type == childs[i]->type) {
      if ((ret = atom_add_child(atoms, childs[i])) < 0) {
	atom_remove_child_all(atoms);
	return ret;
      }
    }
  }
  return atoms->number_of_childs;
}

static int
find_atoms_recursive(atom_t *parent, atom_t *atoms)
{
  int i, ret;

  for (i = 0; i < parent->number_of_childs; i++) {
    if (atoms->type == QTIME_TYPE_UNKNOWN ||
       	atoms->type == parent->childs[i]->type) {
      if ((ret = atom_add_child(atoms, parent->childs[i])) < 0) {
	//atom_remove_child_all(atoms);
	return ret;
      }
    }
    if ((ret = find_recursive_atoms(parent->childs[i], atoms)) != QTIME_OK)
      return ret;
  }
  return QTIME_OK;
}

int
atom_find_atoms_recursive(atom_t *parent, uint32_t type, atom_t *atoms)
{
  int ret;

  atom_remove_child_all(atoms);
  //atom_init(&atoms);
  atoms->type = type;

  if ((ret = find_atoms_recursive(parent, atoms)) != QTIME_OK) {
    atom_remove_child_all(atoms);
    return ret;
  }
  return atoms->number_of_childs;
}

#endif

