/* ac_mp3lame.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LIBMP3LAME


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#ifdef MMX
#undef MMX
#endif

#ifdef HAVE_LAME_H
#include <lame.h>
#endif

#ifdef HAVE_LAME_LAME_H
#include <lame/lame.h>
#endif

#include "parseopt.h"
#include "acodec.h"
#include "ac_mp3lame.h"
#include "mp3_frame.h"
#include "util.h"

#if 0
static MP3LAME_CONFIG mp3lame_config = {
/* option parameter */
/* mp3_bitrate */ 64,
/* mp3_quality */ 5,
/* mp3_mode */    JOINT_STEREO,
/* mp3_vbr_mode */    vbr_off,
/* mp3_vbr_quality */ 5,

/* internal only */
/* rate */         0,
/* channels */     0,
/* bits */         0,
/* frame_size */   0,

/* gfp */          NULL,
};
#else
static MP3LAME_CONFIG mp3lame_config;
#endif

static MP3LAME_CONFIG mp3lame_dec_config;


static SEL_COMPO mp3_mode[] = {
  {"stereo", STEREO},
  {"joint", JOINT_STEREO},
  {"mono", MONO},
};

static SEL_COMPO mp3_vbr_mode[] = {
  {"off",  vbr_off },
  {"default", vbr_default },
  {"mt",   vbr_mt },
  {"rh",   vbr_rh },
  {"abr",  vbr_abr },
  {"mtrh", vbr_mtrh },
};

OptionDef mp3_param[] = {
  {"mp3-bitrate", "MP3_BITRATE", HAS_ARG, {(void*)&mp3lame_config.bitrate}, {64}, {0}, 320, "mp3lame bitrate", "b"},
  {"mp3-quality", "MP3_QUALITY", HAS_ARG, {(void*)&mp3lame_config.quality}, {5}, {0}, 9, "mp3lame quality", "q"},
  {"mp3-mode", "MP3_MODE", HAS_ARG|OPT_SEL, {(void*)&mp3lame_config.mode}, {JOINT_STEREO}, {(int)mp3_mode}, sizeof(mp3_mode)/sizeof(SEL_COMPO), "mp3lame mode", "m"},
  {"mp3-vbr-mode", "MP3_VBR_MODE", HAS_ARG|OPT_SEL, {(void*)&mp3lame_config.vbr_mode}, {vbr_off}, {(int)mp3_vbr_mode}, sizeof(mp3_vbr_mode)/sizeof(SEL_COMPO), "mp3lame VBR mode", "sel"},
  {"mp3-vbr-quality", "MP3_VBR_QUALITY", HAS_ARG, {(void*)&mp3lame_config.vbr_quality}, {5}, {0}, 9, "mp3lame VBR quality", "d"},
};

int mp3_param_num = sizeof(mp3_param) / sizeof(OptionDef);

const char*
errcode_to_errstr(int code)
{
  const char *str;
  switch (code) {
    case -1:
      str = "mp3buf was too small";
      break;
    case -2:
      str = "memory allocation proglem";
      break;
    case -3:
      str = "lame_init_params() not called";
      break;
    case -4:
      str = "psycho acoustic problems";
      break;
    case -5:
      str = "ogg cleanup encoding error";
      break;
    case -6:
      str = "ogg frame encoding error";
      break;
    default:
      str = "UNKNOWN";
      break;
  }
  return str;
}

const char*
mp3lame_get_codec_name(void)
{
  return "MP3LAME";
}

uint32_t
mp3lame_get_cap(uint32_t codec)
{
  if (codec == ACODEC_MP3)
    return ACODEC_CAP_ANALYSIS;
  return ACODEC_CAP_NONE;
}

uint32_t
mp3lame_fourcc_to_codec(const char *fourcc)
{
  if (strncmp(fourcc, ".mp3", 4) == 0) {
    return ACODEC_MP3;
  }
  if (strncmp(fourcc, ".MP3", 4) == 0) {
    return ACODEC_MP3;
  }
  return ACODEC_UNKNOWN;
}

uint32_t
mp3lame_code_to_codec(int code)
{
  if (code == 0x55)
    return ACODEC_MP3;
  return ACODEC_UNKNOWN;
}

const char*
mp3lame_codec_to_fourcc(uint32_t codec)
{
  if (codec == ACODEC_MP3) {
#if 0
    if (mp3lame_config.vbr_mode != vbr_off)
      return ".MP3";
    else
      return ".mp3";
#else
    return ".mp3";
#endif
  }
  return NULL;
}

int
mp3lame_codec_to_code(uint32_t codec)
{
  if (codec == ACODEC_MP3)
    return 0x55;
  return 0;
}


/* encode functions */

static int
mp3lame_frame_parse(MP3LAME_CONFIG *conf, int *num_samples, uint8_t *stream_buf,int stream_buf_size, int one_frame_flag)
{
  uint8_t *in_stream, *out_stream;
  int frame_length, frame_samples;
  int length, size, start, samples;

  length = conf->stream_buf_pos;
  in_stream = conf->stream_buf;
  out_stream = stream_buf;
  frame_length = conf->frame_length;
  frame_samples = conf->frame_samples;
  samples = 0;
  size = 0;
  start = 0;

  while (length > 4) {
    if (frame_length <= 0) {
      frame_length = mp3_frame_header(&conf->mp3frame,in_stream,length,&start);
      frame_samples = conf->mp3frame.frame_samples;
      in_stream += start;
      length -= start;
    } else if (frame_length <= length) {
      memcpy(out_stream, in_stream, frame_length);
      in_stream  += frame_length;
      length     -= frame_length;
      out_stream += frame_length;
      size       += frame_length;
      samples    += frame_samples;
      frame_length = 0;
      frame_samples = 0;
      if (one_frame_flag)
	break;
    } else {
      break;
    }
  }

  if (conf->stream_buf_pos > length) {
    memmove(conf->stream_buf, in_stream, length);
    conf->stream_buf_pos = length;
  }
  conf->frame_length   = frame_length;
  conf->frame_samples  = frame_samples;

  *num_samples = samples;

  if (size > stream_buf_size) {
    fprintf(stderr, "mp3lame_frame_parse: stream_buf size is too small.\n");
  }

  return size;
}

int
mp3lame_encode (void *pcm_buf[], int *num_samples, uint8_t *stream_buf,int stream_buf_size)
{
  MP3LAME_CONFIG *conf = &mp3lame_config;
  int length;

  if (*num_samples > 0) {
    if (conf->channels == 2) {
      length = lame_encode_buffer(conf->gfp,
                   (const short int*)pcm_buf[0], (const short int*)pcm_buf[1],
                   *num_samples,
	           conf->stream_buf + conf->stream_buf_pos,
	           conf->stream_buf_size - conf->stream_buf_pos);
    } else {
      const short dummy[*num_samples+32];
      length = lame_encode_buffer(conf->gfp,
                   (const short int*)pcm_buf[0], dummy,
                   *num_samples,
	           conf->stream_buf + conf->stream_buf_pos,
	           conf->stream_buf_size - conf->stream_buf_pos);
    }
    if (length < 0) {
      fprintf (stderr, "mp3lame_encode: lame_encode_buffer() error %s, ",
                        errcode_to_errstr(length));
      fprintf (stderr, "spl_length %d, stream_remain %d\n",
                        *num_samples, stream_buf_size);
      return ACODEC_FAIL;
    }
    conf->stream_buf_pos += length;
  }

  length = mp3lame_frame_parse(conf, num_samples,stream_buf,stream_buf_size,1);

  //fprintf (stderr, "mp3lame_encode: length %d, num_samples %d.\n", length, *num_samples);
  return length;
}

int
mp3lame_encode_flush (int *num_samples, uint8_t *stream_buf, int stream_buf_size)
{
  MP3LAME_CONFIG *conf = &mp3lame_config;
  int length;

  if (!conf->flushed) {
    length = lame_encode_flush(conf->gfp,
                               conf->stream_buf + conf->stream_buf_pos,
                               conf->stream_buf_size - conf->stream_buf_pos);
    if (length < 0) {
      fprintf (stderr, "mp3lame_flush: lame_encode_flush() error %s, stream_buf_size %d\n", errcode_to_errstr(length), stream_buf_size);
      return -1;
    }
    conf->stream_buf_pos += length;
    conf->flushed = 1;
  }

  length = mp3lame_frame_parse(conf, num_samples,stream_buf,stream_buf_size,1);

  return length;;
}

int
mp3lame_encode_rate_cap(int rate)
{
  if (rate < 100)
    rate *= 1000;

  if (rate < 11025)
    rate = 11025;
  else if (rate < 22050)
    rate = 11025;
  else if (rate < 44100)
    rate = 22050;
  else if (rate < 48000)
    rate = 44100;
  else if (rate >= 48000)
    rate = 48000;
  return rate;
}

#if 0
int
mp3lame_encode_rate_cap(int rate)
{
  if (rate < 100)
    rate *= 1000;

  if (rate < 8000)
    rate = 8000;
  else if (rate < 11025)
    rate = 8000;
  else if (rate < 12000)
    rate = 11025;
  else if (rate < 16000)
    rate = 12000;
  else if (rate < 22050)
    rate = 16000;
  else if (rate < 24000)
    rate = 22050;
  else if (rate < 32000)
    rate = 24000;
  else if (rate < 44100)
    rate = 32000;
  else if (rate < 48000)
    rate = 44100;
  else if (rate >= 48000)
    rate = 48000;
  return rate;
}
#endif

int
mp3lame_encode_channels_cap(int channels)
{
  int ret;
  switch (channels) {
    case 1:
      ret = 1;
      break;
    default:
    case 2:
      ret = 2;
      break;
  }
  return ret;
}

int
mp3lame_encode_bits_cap(int bits)
{
#if 0
  int ret;
  switch (bits) {
    default:
    case 16:
      ret = 16;
      break;
  }
  return ret;
#else
  bits = 16;
  return bits;
#endif
}

int
mp3lame_encode_get_frame_size(void)
{
  return mp3lame_config.frame_size;
}

int
mp3lame_encode_get_bitrate(void)
{
  return mp3lame_config.bitrate;
}

int
mp3lame_encode_get_out_rate(void)
{
  if (mp3lame_config.gfp == NULL) {
    fprintf(stderr, "mp3lame_encode_get_out_rate: codec doesn't initialized.\n");
    return 0;
  }
  return lame_get_out_samplerate(mp3lame_config.gfp);
}

int
mp3lame_encode_quit (void)
{
  MP3LAME_CONFIG *conf = &mp3lame_config;

  if (conf->gfp) {
    lame_close(conf->gfp);
    conf->gfp = NULL;
  }
  if (conf->stream_buf)
    free(conf->stream_buf);
  conf->stream_buf = NULL;
  conf->stream_buf_size = 0;
  conf->stream_buf_pos = 0;
  conf->frame_length = 0;
  conf->frame_samples = 0;
  conf->flushed = 0;

  return ACODEC_OK;
}

int
mp3lame_encode_init (ACODEC *codec)
{
  static int first = 1;
  MP3LAME_CONFIG *conf = &mp3lame_config;
  int ret, buf_size;
  int need_config = 0;

  if (first) {
    conf->rate = 0;
    conf->channels = 0;
    conf->bits = 0;
    conf->frame_size = 0;
    conf->gfp = NULL;
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
    conf->frame_length = 0;
    conf->frame_samples = 0;
    conf->flushed = 0;
    first = 0;
    need_config = 1;
  }

  if (conf->gfp == NULL)
    need_config = 1;
  else if (conf->rate != codec->rate)
    need_config = 1;
  else if (conf->channels != codec->channels)
    need_config = 1;
  else if (conf->bits != codec->bits)
    need_config = 1;

  if (!need_config) {
    return ACODEC_OK;
  }

  conf->rate = codec->rate;
  conf->channels = codec->channels;
  conf->bits = codec->bits;

  if (conf->gfp) {
    lame_close(conf->gfp);
    conf->gfp = NULL;
  }

  switch (conf->channels) {
    case 1:
      if (conf->mode != MONO)
        conf->mode = MONO;
      break;
    case 2:
      if ((conf->mode != STEREO) && (conf->mode != JOINT_STEREO) &&
	  (conf->mode != DUAL_CHANNEL))
	conf->mode = STEREO;
      break;
    default:
      fprintf (stderr, "mp3lame_init(): not support %d channel enccode. channel must be 1 or 2.\n", conf->channels);
      return ACODEC_FAIL;
      break;
  }

  conf->gfp = lame_init();

//  lame_set_preset(mp3lame_config.gfp, preset);
//  lame_set_asm_optimizations(mp3lame_config.gfp, SSE, SSE);

  lame_set_in_samplerate(conf->gfp, conf->rate);
  lame_set_num_channels(conf->gfp, conf->channels);
  lame_set_brate(conf->gfp, conf->bitrate);
  lame_set_mode(conf->gfp, conf->mode);
  lame_set_quality(conf->gfp, conf->quality);

  if (conf->vbr_mode != vbr_off) {
    lame_set_VBR (conf->gfp, conf->vbr_mode);
    lame_set_VBR_q(conf->gfp, conf->vbr_quality);
//  if (conf->vbr_mode == vbr_abr) {
//    lame_set_VBR_mean_bitrate_kbps (conf->gfp, bitrate);
//    lame_set_VBR_min_bitrate_kbps (conf->gfp, 32);
//    lame_set_VBR_max_bitrate_kbps (conf->gfp, bitrate);
//  }
  }

  ret = lame_init_params(conf->gfp);
  if (ret < 0) {
    fprintf (stderr, "mp3lame_init(): lame_init_params() error\n");
    return ACODEC_FAIL;
  }
  conf->frame_size = lame_get_framesize(conf->gfp);

//  buf_size = conf->frame_size * conf->channels * ((conf->bits+7)/8);
  buf_size = conf->frame_size * conf->channels * ((conf->bits+7)/8) * 4;
  if (conf->stream_buf_size < buf_size) {
    uint8_t *buf;
    buf = (uint8_t*)realloc(conf->stream_buf, buf_size);
    if (!buf)
      return ACODEC_FAIL;
    conf->stream_buf = buf;
    conf->stream_buf_size = buf_size;
  }
  conf->stream_buf_pos = 0;
  conf->frame_length = 0;
  conf->frame_samples = 0;
  conf->flushed = 0;

  return ACODEC_OK;
}

void
mp3lame_encode_print_param (void)
{
  MP3LAME_CONFIG *conf = &mp3lame_config;
  SEL_COMPO *sel;
  int i;

  fprintf (stderr, "ENCPARAM MP3LAME\n");
  fprintf (stderr, "bitrate:                 %d\n", conf->bitrate);
  if (conf->vbr_mode == vbr_off) {
    fprintf (stderr, "quality:                 %d\n", conf->quality);
  } else if (conf->vbr_mode != vbr_off) {
    sel = mp3_vbr_mode;
    for (i=0; (uint32_t)i < sizeof(mp3_vbr_mode)/sizeof(SEL_COMPO); i++, sel++){
      if (sel->id == conf->vbr_mode) {
        fprintf (stderr, "vbr_mode:                %s\n", sel->name);
        break;
      }
    }
    fprintf (stderr, "vbr_quality:             %d\n", conf->vbr_quality);
  }

  sel = mp3_mode;
  for (i=0; (uint32_t)i < sizeof(mp3_mode)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == conf->mode) {
      fprintf (stderr, "mode:                    %s\n", sel->name);
      break;
    }
  }
  if (i == sizeof(mp3_mode)/sizeof(SEL_COMPO))
    fprintf (stderr, "mode:                    %s\n", "UNKNOWN");

  fprintf (stderr, "frame size:              %d\n", conf->frame_size);

  fflush (stderr);
}


/* decode functions */

int
mp3lame_decode (void *pcm_buf[], uint8_t *stream, int stream_size)
{
  int num_spl;
//  MP3LAME_CONFIG *conf = &mp3lame_dec_config;

  num_spl = lame_decode(stream, stream_size,
                   (short*)pcm_buf[0], (short*)pcm_buf[1]);
  return num_spl;
}

int
mp3lame_decode_get_rate(void)
{
  return mp3lame_dec_config.rate;
}

int
mp3lame_decode_get_channels(void)
{
  return mp3lame_dec_config.channels;
}

int
mp3lame_decode_get_bits(void)
{
  return 16;
}

int
mp3lame_decode_get_bitrate(void)
{
  return mp3lame_dec_config.bitrate;
}

int
mp3lame_decode_quit (void)
{
  MP3LAME_CONFIG *conf = &mp3lame_dec_config;

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
  }
  conf->rate = 0;
  conf->channels = 0;
  conf->bits = 0;
//  conf->bitrate = 0;
  conf->analyzed = 0;

  return ACODEC_OK;
}

int
mp3lame_decode_analyze_stream(ACODEC *acodec, uint8_t *stream, int stream_size)
{
  MP3LAME_CONFIG *conf = &mp3lame_dec_config;
  int start = 0;
  int vbr = 0;
  uint8_t *strm;
  int size;
  int buf_pos;
  int frame_length;

  buf_pos = conf->stream_buf_pos;
  size = buf_pos + stream_size;

  strm = conf->stream_buf;
  if (size > conf->stream_buf_size) {
    strm = (uint8_t*)realloc(conf->stream_buf, size);
    if (!strm) {
      fprintf(stderr,"mp3lame_decode_analyze_stream: memory allocate error.\n");
      exit(1);
    }
    conf->stream_buf_size = size;
  }
  memcpy(strm + buf_pos, stream, stream_size);
  buf_pos += stream_size;
  conf->stream_buf = strm;
  conf->stream_buf_pos = buf_pos;

  vbr = mp3_frame_header_vbr(&conf->mp3frame, strm, buf_pos, &start);
  if (vbr < 0) {
    if (start > 0)
      memmove(strm, strm + start, buf_pos - start);
    conf->stream_buf_pos = buf_pos - start;
    return ACODEC_NEED_MORE_DATA;
  } else if (vbr > 0) {
//    conf->vbr = 1;
    if (buf_pos < (start+vbr)) {
      if (start > 0)
        memmove(strm, strm + start, buf_pos - start);
      conf->stream_buf_pos = buf_pos - start;
      return ACODEC_NEED_MORE_DATA;
    }
    memmove(strm, strm + start + vbr, buf_pos - start - vbr);
    buf_pos -= (start + vbr);
    conf->stream_buf_pos = buf_pos;
  }

  frame_length = mp3_frame_header(&conf->mp3frame, strm, buf_pos, &start);
  if (frame_length <= 0) {
    if (start > 0)
      memmove(strm, strm + start, buf_pos - start);
    buf_pos -= start;
    conf->stream_buf_pos = buf_pos;
    return ACODEC_NEED_MORE_DATA;
  }

  acodec->rate = conf->rate = conf->mp3frame.frequency;
  switch (conf->mp3frame.mode) {
    case MP3_MODE_STEREO:
    case MP3_MODE_JOINT:
    case MP3_MODE_DUAL:
      acodec->channels = conf->channels = 2;
      break;
    case MP3_MODE_MONO:
      acodec->channels = conf->channels = 1;
      break;
    default:
      fprintf(stderr, "mp3lame_decode_analyze_stream: cannot be specified channels. %d\n", conf->mp3frame.mode);
      break;
  }
  acodec->bits = conf->bits = 16;
  conf->bitrate = conf->mp3frame.bitrate;
  conf->analyzed = 1;

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_pos = 0;
    conf->stream_buf_size = 0;
  }

  return ACODEC_OK;
}

int
mp3lame_decode_init (ACODEC *codec)
{
  static int first = 1;
  MP3LAME_CONFIG *conf = &mp3lame_dec_config;

  if (first) {
    conf->rate = 0;
    conf->channels = 0;
    conf->bits = 0;
    conf->bitrate = 0;
    conf->analyzed = 0;
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
    first = 0;
  }

  if (!conf->analyzed) {
    conf->rate = codec->rate;
    conf->channels = codec->channels;
    conf->bits = codec->bits;
  }

  if (conf->stream_buf) {
    free(conf->stream_buf);
    conf->stream_buf = NULL;
    conf->stream_buf_size = 0;
    conf->stream_buf_pos = 0;
  }

  lame_decode_init();

  return ACODEC_OK;
}

void
mp3lame_decode_print_param (void)
{
  fprintf (stderr, "DECPARAM MP3LAME\n");
  fprintf (stderr, "rate:                    %d\n", mp3lame_dec_config.rate);
  fprintf (stderr, "channels:                %d\n",mp3lame_dec_config.channels);
  fprintf (stderr, "bits:                    %d\n", mp3lame_dec_config.bits);
  fprintf (stderr, "bitrate:                 %d\n", mp3lame_dec_config.bitrate);
  fflush (stderr);
}


/* function structure */

ACODEC_FUNCS mp3lame_funcs = {
  mp3lame_get_codec_name,
  mp3lame_get_cap,
  mp3lame_fourcc_to_codec,
  mp3lame_code_to_codec,
  mp3lame_codec_to_fourcc,
  mp3lame_codec_to_code,

  mp3lame_encode_init,
  mp3lame_encode_quit,
  mp3lame_encode,
  mp3lame_encode_flush,
  mp3lame_encode_rate_cap,
  mp3lame_encode_channels_cap,
  mp3lame_encode_bits_cap,
  mp3lame_encode_get_frame_size,
  mp3lame_encode_get_bitrate,
  mp3lame_encode_get_out_rate,
  mp3lame_encode_print_param,

  mp3lame_decode_init,
  mp3lame_decode_quit,
  mp3lame_decode,
  mp3lame_decode_get_rate,
  mp3lame_decode_get_channels,
  mp3lame_decode_get_bits,
  mp3lame_decode_get_bitrate,
  mp3lame_decode_analyze_stream,
  mp3lame_decode_print_param,
};

#endif /* HAVE_LIBMP3LAME */

