/* Copyright 2013 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <ncursesw/ncurses.h>

#include "env.h"
#include "error.h"
#include "nt_string.h"
#include "_2ch/_2ch.h"
#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "ui/disp.h"


static BOOL read_editor(FILE *fp, nt_write_data_tp writep,
		char *buf_reuse, size_t buf_len);

BOOL disp_editor(nt_write_data_tp writep)
{
	FILE *tmp_fp;
	pid_t pid;
	char fname[512];
	char buf[1024*2];
	int len, ret;

	assert(writep);

	writep->status_msg = NULL;

	pid = getpid();

	sprintf(fname, "%s/msg%d.tmp", LOG_PATH, pid);
	tmp_fp = fopen(fname, "w");
	if(!tmp_fp)
		return FALSE;
	
	strcpy(buf, "name: ");
	if(writep->name)
		strcat(buf, writep->name);
	strcat(buf, "\n");
	len = strlen(buf);	
	if(len > fwrite(buf, sizeof(char), len, tmp_fp))
		goto ERROR_TRAP1;

	sprintf(buf, "mail: %s\n\n", 
			(writep->mail) ? writep->mail : "sage");
	len = strlen(buf);	
	if(len > fwrite(buf, sizeof(char), len, tmp_fp))
		goto ERROR_TRAP1;
	
	fprintf(tmp_fp, "Please write your message here\n");
	fclose(tmp_fp);

	sprintf(buf, "%s %s", EDITOR_CMD, fname);
	ret = system(buf);
	if (WIFSIGNALED(ret) &&
         	(WTERMSIG(ret) == SIGINT || 
			 WTERMSIG(ret) == SIGQUIT))
		goto ERROR_TRAP2;

	tmp_fp = fopen(fname, "r");
	if(!tmp_fp)
		goto ERROR_TRAP2;

	if(!read_editor(tmp_fp, writep, buf, sizeof(buf)))
		goto ERROR_TRAP1;
	

	writep->status_msg = NT_INFO_WRITE_MSG_SUCCESS;
	fclose(tmp_fp);
	unlink(fname);
	return TRUE;

ERROR_TRAP1:
	fclose(tmp_fp);
ERROR_TRAP2:
	unlink(fname);
	return FALSE;
}

BOOL read_editor(FILE *fp, nt_write_data_tp writep, 
		char *buf_reuse, size_t buf_len)
{
	char *cptr, *cptr2;
	char buf2[64];
	int len;
	int offset, nread;

	/* Read headers. */
	while(fgets(buf_reuse, buf_len, fp)){
		if(buf_reuse[0] == '\n' ||
				buf_reuse[0] == '\0')
			break;
		cptr = strchr(buf_reuse, ':');
		if(!cptr)
			break;

		*cptr = '\0';
		cptr2 = nt_trim2(buf_reuse, buf2, sizeof(buf2));
		if(!cptr2)
			continue;

		if(0 == strcmp(cptr2, "name")){
			if(writep->name)
				free(writep->name);
			writep->name = nt_trim(cptr+1);
		}else if(0 == strcmp(cptr2, "mail")){
			if(writep->mail)
				free(writep->mail);
			writep->mail = nt_trim(cptr+1);
		}
	}
	/* Read message to send */
	buf_reuse[0] = '\0';
	offset = 0;
	while(fgets(buf_reuse+offset, buf_len-offset, fp)){
		nread = strlen(buf_reuse+offset);
		offset += nread;
	}
	len = strlen(buf_reuse);
	if(len == 0){
		writep->status_msg = NT_ERR_MSG_WRITE_MSG_LENGTH;
		return FALSE;
	}
	if(!strcmp(buf_reuse, "Please write your message here\n")){
		writep->status_msg = NT_ERR_MSG_WRITE_MSG_ABORTED;
		return FALSE;
	}
	cptr = malloc(len+1);
	strcpy(cptr, buf_reuse);
	writep->msg = cptr;
	return TRUE;
}

