#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <wchar.h>
#include <ncursesw/ncurses.h>
#include <regex.h>

#include "env.h"
#include "error.h"
#include "utils/nt_std_t.h"
#include "utils/nt_usr_db.h"
#include "_2ch/_2ch.h"
#include "_2ch/model_2ch.h"
#include "utils/text.h"
#include "ui/disp.h"
#include "ui/disp_string.h"

typedef struct tag_ctx_threadlist_t *ctx_threadlist_tp;
typedef struct tag_ctx_threadlist_t
{
	int thread_num;
	int cursor_pos;
	int scroll_pos;

	regex_t regex;
	BOOL regex_init;
	int sel_thread_no;

	nt_link_tp thread_data_list;

	nt_link_tp thread_disp_list;

	int sort_type;

} ctx_threadlist_t;

#define NT_CMD_ERR -1
#define NT_CMD_NONE 0
#define NT_CMD_SORT_NUMBER 1
#define NT_CMD_SORT_READ 2
#define NT_CMD_SORT_UNREAD 3

static int parse_cmd1(const char *param);
static int get_thread_index_by_seqno(nt_link_tp linkp, int seq_no);
static BOOL search_line_asc(regex_t *regexp, 
				nt_link_tp threadlistp, int *sel_thread_no);
static BOOL search_line_desc(regex_t *regexp, 
				nt_link_tp threadlistp, int *sel_thread_no);
static ctx_threadlist_tp init_context(nt_2ch_model_tp modelp);


static int sort_by_number(void *lhs, void *rhs);
static int sort_by_unread(void *lhs, void *rhs);
static int sort_by_read(void *lhs, void *rhs);
static ctx_threadlist_tp g_ctxp;

static void sort_thread_title(int sort_type, ctx_threadlist_tp ctxp)
{
	g_ctxp = ctxp;
	switch(sort_type){
	case NT_THREAD_SORT_BY_NUMBER:
		nt_link_sort(&ctxp->thread_disp_list, sort_by_number);
		break;
	case NT_THREAD_SORT_BY_UNREAD:
		nt_link_sort(&ctxp->thread_disp_list, sort_by_unread);
		break;
	case NT_THREAD_SORT_BY_READ:
		nt_link_sort(&ctxp->thread_disp_list, sort_by_read);
		break;
	}
}

int disp_threadlist(nt_window_tp wp, nt_2ch_model_tp modelp,
				void *db_handle)
{
	ctx_threadlist_tp ctxp;
	nt_board_tp boardp;
	nt_thread_tp threadp;
	nt_link_tp clistp;
	int i, rows, nwrite, len, ch;
	int btm_cur, num, wlines;
	wchar_t buf[16];
	attr_t attr;
	BOOL search_asc;
	int read_count, column;
	int cmd;
	BOOL sort;

	sort = FALSE;

	ctxp = (ctx_threadlist_tp)wp->data;
	if(!ctxp){
		ctxp = init_context(modelp);
		if(!ctxp)
			return DISP_STATE_ERROR;
		wp->data = ctxp;
		sort = TRUE;
	}

	boardp = modelp->selected_boardp;

	if(!ctxp->thread_data_list){
		ctxp->thread_data_list =
			nt_usr_db_query_read_count_list(
					db_handle, boardp->name);
	}

	if(sort){
		sort_thread_title(THREAD_SORT_TYPE, ctxp);
	}

	ch = wp->key;

	switch(ch){
	case NT_KEY_REFRESH: 
		return DISP_STATE_REFRESH;
	case NT_KEY_CLOSE:
	case KEY_LEFT:
		return DISP_STATE_BOARDMENU;
	case NT_KEY_BOTTOM:
	case KEY_END:
		ctxp->cursor_pos = ctxp->thread_num - 1;

		wlines = ctxp->thread_num * 2;
		wlines -= wp->lines;
		if(wlines < 0){
			ctxp->scroll_pos = 0;
			break;
		}
		ctxp->scroll_pos = wlines;
		//ctxp->scroll_pos += wlines%2;
		break;
	case NT_KEY_COMMAND2:
	case NT_KEY_COMMAND3:
		search_asc = (ch == NT_KEY_COMMAND2);
		if(wp->cmd_param && wp->cmd_param[0] != '\0'){
			if(0 != regcomp(&(ctxp->regex),
					wp->cmd_param, REG_EXTENDED)){
				if(ctxp->regex_init){
					regfree(&(ctxp->regex));
					ctxp->regex_init = FALSE;
					break;
				}
			}
			if(!ctxp->regex_init)
			    ctxp->regex_init = TRUE;
		}
		if(!ctxp->regex_init)
		    break;
		if(search_asc){
		    if(!search_line_asc(&(ctxp->regex), ctxp->thread_disp_list,
			        &ctxp->sel_thread_no)){
				wp->status_msg = NT_ERR_MSG_SEARCH_NOT_FOUND;
				break;
			}
		}else{
		    if(!search_line_desc(&(ctxp->regex), ctxp->thread_disp_list,
			        &ctxp->sel_thread_no)){
				wp->status_msg = NT_ERR_MSG_SEARCH_NOT_FOUND;
				break;
			}
		}
		num = ctxp->sel_thread_no;
		//fall through
	case NT_KEY_COMMAND1:
		if(ch == NT_KEY_COMMAND1){
			assert(wp->cmd_param);
			cmd = parse_cmd1(wp->cmd_param);
			switch(cmd){
			case NT_CMD_SORT_NUMBER:
				THREAD_SORT_TYPE = NT_THREAD_SORT_BY_NUMBER;
				sort_thread_title(THREAD_SORT_TYPE, ctxp);
				return DISP_STATE_REFRESH;
			case NT_CMD_SORT_READ:
				THREAD_SORT_TYPE = NT_THREAD_SORT_BY_READ;
				sort_thread_title(THREAD_SORT_TYPE, ctxp);
				return DISP_STATE_REFRESH;
			case NT_CMD_SORT_UNREAD:
				THREAD_SORT_TYPE = NT_THREAD_SORT_BY_UNREAD;
				sort_thread_title(THREAD_SORT_TYPE, ctxp);
				return DISP_STATE_REFRESH;
			default:
				break;
			}
			num = atoi(wp->cmd_param);
			if(0 == num)
				break;
			num = get_thread_index_by_seqno(
					ctxp->thread_disp_list, num);
			//num--;
		}
		wlines = ctxp->thread_num - num;
		if(wlines <= 0)
			break;

		wlines *= 2;
		ctxp->cursor_pos = num;
		if(wp->lines < wlines){
			ctxp->scroll_pos = num * 2;
		}else{
			ctxp->scroll_pos =
				ctxp->thread_num * 2 - wp->lines;
		}
		break;
	case NT_KEY_UP:
	case KEY_UP:
		ctxp->cursor_pos--;
		if(ctxp->cursor_pos < 0)
			ctxp->cursor_pos = 0;
		if(ctxp->scroll_pos > (ctxp->cursor_pos*2)){
			ctxp->scroll_pos = ctxp->cursor_pos*2;
		}
		break;
	case NT_KEY_DOWN:
	case KEY_DOWN:
		ctxp->cursor_pos++;
		if(ctxp->cursor_pos >= ctxp->thread_num)
			ctxp->cursor_pos = ctxp->thread_num - 1;
		btm_cur = ctxp->cursor_pos * 2 + 1;
		if(btm_cur >= (ctxp->scroll_pos + wp->lines)){
			ctxp->scroll_pos = btm_cur;
			ctxp->scroll_pos -= wp->lines;
			ctxp->scroll_pos++;
		}
		break;
	case NT_KEY_PAGEUP:
	case KEY_PPAGE:
		if(wp->lines >= ctxp->thread_num*2){
			ctxp->scroll_pos = 0;
			break;
		}
		ctxp->scroll_pos -= wp->lines;
		if(ctxp->scroll_pos < 0){
			ctxp->cursor_pos *= 2;
			ctxp->cursor_pos -= ctxp->scroll_pos + wp->lines;
			ctxp->scroll_pos =  0;
			//ctxp->cursor_pos += ctxp->scroll_pos;
			ctxp->cursor_pos /= 2;
		}else{
			ctxp->cursor_pos -= wp->lines / 2;
			if(ctxp->cursor_pos * 2 < ctxp->scroll_pos)
				ctxp->cursor_pos = ctxp->scroll_pos / 2;
			if(ctxp->cursor_pos * 2 >= ctxp->scroll_pos + wp->lines){
				ctxp->cursor_pos = (ctxp->scroll_pos + wp->lines) / 2;
				ctxp->cursor_pos -= (ctxp->scroll_pos + wp->lines) % 2;
			}
		}
		break;
	case NT_KEY_PAGEDOWN:
	case KEY_NPAGE:
		if(wp->lines >= ctxp->thread_num*2){
			ctxp->scroll_pos = 0;
			break;
		}
		ctxp->scroll_pos += wp->lines;
		if(ctxp->scroll_pos + wp->lines > ctxp->thread_num*2){
			ctxp->cursor_pos *= 2;
			ctxp->cursor_pos -= ctxp->scroll_pos - wp->lines;
			ctxp->scroll_pos =  (ctxp->thread_num * 2) - wp->lines;
			ctxp->cursor_pos += ctxp->scroll_pos;
			ctxp->cursor_pos /= 2;
		}else{
			ctxp->cursor_pos += wp->lines / 2;
			if(ctxp->cursor_pos * 2 < ctxp->scroll_pos)
				ctxp->cursor_pos = ctxp->scroll_pos / 2;
			if(ctxp->cursor_pos * 2 >= ctxp->scroll_pos + wp->lines){
				ctxp->cursor_pos = (ctxp->scroll_pos + wp->lines) / 2;
				ctxp->cursor_pos -= (ctxp->scroll_pos + wp->lines) % 2;
			}
		}
		break;
	case NT_KEY_SELECT:
	case KEY_RIGHT:
		threadp = nt_link_get_by_index(ctxp->thread_disp_list,
				ctxp->cursor_pos);
		if(!threadp)
			break;
		nt_set_selected_thread(modelp, threadp);
		return DISP_STATE_RESLIST;
	}

	clistp = ctxp->thread_disp_list;
	rows = 0;
	for(i = 0; i < ctxp->thread_num; i++){
		if(rows >= (wp->lines+ctxp->scroll_pos))
			break;
		if(ctxp->cursor_pos == i)
			attr = WA_BOLD;
		else
			attr = 0;
		threadp = (nt_thread_tp)clistp->data;
		len = wcslen(threadp->name);
		if(rows >= ctxp->scroll_pos){
			wmove(wp->wp, rows - ctxp->scroll_pos, 0);
			if(-1 == swprintf(buf, sizeof(buf)-1, 
							L"%5d.", threadp->seq_no))
				continue;

			nt_add_wstr(wp->wp, buf, attr);
			nwrite = nt_add_wnstr(wp->wp, threadp->name, attr, 
						wp->cols - 5); 
		}else{
			nwrite = nt_get_wc_count_within_colmns(
					threadp->name, wp->cols - 5);
		}
		rows++;
		if(rows >= (wp->lines+ctxp->scroll_pos))
			break;
		if(rows >= ctxp->scroll_pos){
			wmove(wp->wp, rows - ctxp->scroll_pos, 0);
			nt_add_wnch(wp->wp, L' ', WA_UNDERLINE | attr, wp->cols);
			wmove(wp->wp, rows - ctxp->scroll_pos, 5);
			if(nwrite < len){
				nwrite = nt_add_wnstr(wp->wp, 
					threadp->name + nwrite, 
					WA_UNDERLINE | attr, wp->cols - 5); 
			}
			if(ctxp->thread_data_list){
				read_count = nt_usr_db_get_read_count_by_dat_name(
						ctxp->thread_data_list, threadp->file_name);
				if(read_count < 0)
					read_count = 0;
			}else{
				read_count = 0;
			}
			buf[0] = L'\0';
			if(read_count == 0){
				if(-1 < swprintf(
					buf, sizeof(buf)-1, L"(%d)  ", threadp->num_res)){
				}
			}else{
				if(-1 < swprintf(
					buf, sizeof(buf)-1, L"(%d) 未読:%d ", 
						threadp->num_res, 
						threadp->num_res - read_count)){
				}
			}
			column = nt_get_column_length(buf);
			if(column > 0){
				if(read_count > 0 && read_count < threadp->num_res)
					attr |= WA_REVERSE;
				wmove(wp->wp, 
					rows - ctxp->scroll_pos, wp->cols - column);
				nwrite = nt_add_wstr(wp->wp, buf, WA_UNDERLINE | attr);
			}
		}
		rows++;
		clistp = clistp->next;
	}/* end for */
	
	return DISP_STATE_THREADTITLE; 
}

static ctx_threadlist_tp init_context(nt_2ch_model_tp modelp)
{
	ctx_threadlist_tp ctxp;
	nt_board_tp boardp;
	ctxp = (ctx_threadlist_tp)calloc(1,sizeof(ctx_threadlist_t));

	if(!ctxp)
		return NULL;

	boardp = modelp->selected_boardp;
	ctxp->thread_num = nt_link_num(boardp->threadlistp);

	ctxp->regex_init = FALSE;
	ctxp->sel_thread_no = -1;

	ctxp->thread_data_list = NULL;
	ctxp->thread_disp_list = 
		nt_link_copy(boardp->threadlistp);
	if(!ctxp->thread_disp_list){
		free(ctxp);
		return NULL;
	}
	ctxp->sort_type = NT_CMD_SORT_NUMBER;
	return ctxp;
}

void init_threadlist_ctx(void *ptr)
{
	ctx_threadlist_tp ctxp;
	if(!ptr)
		return;
	ctxp = (ctx_threadlist_tp)ptr;
	if(ctxp->thread_data_list){
		nt_all_link_free(ctxp->thread_data_list,
			nt_usr_db_thread_data_free);
		ctxp->thread_data_list = NULL;
	}
}

void free_threadlist_ctx(void *ptr)
{
	ctx_threadlist_tp ctxp;
	if(!ptr)
		return;
	ctxp = (ctx_threadlist_tp)ptr;
	if(ctxp->thread_data_list){
		nt_all_link_free(ctxp->thread_data_list,
			nt_usr_db_thread_data_free);
	}
	if(ctxp->thread_disp_list){
		nt_all_link_free(ctxp->thread_disp_list, NULL);
	}
	free(ctxp);
}


static BOOL search_line_asc(regex_t *regexp, 
				nt_link_tp threadlistp, int *sel_thread_no)
{
	wchar_t *cptr;
	nt_thread_tp threadp;
	nt_link_tp clistp;
	size_t nmatch = 5;
	regmatch_t pmatch[5];
	char buf[256];
	int cur;

	clistp = threadlistp;
	cur = 0;
	do{
		if(cur > *sel_thread_no){
			threadp = (nt_thread_tp)clistp->data;
			cptr = threadp->name;
			if(0 < wcstombs(buf, cptr, sizeof(buf))){
			    if(0 == regexec(regexp, buf,
					    nmatch, pmatch, 0)){
					*sel_thread_no = cur;
					return TRUE;
				}
			}
		}
		cur++;
		clistp = clistp->next;
	}while(clistp != threadlistp);
	clistp = threadlistp;
	cur = 0;
	do{
		if(cur > *sel_thread_no)
			break;

		threadp = (nt_thread_tp)clistp->data;
		cptr = threadp->name;
		if(0 < wcstombs(buf, cptr, sizeof(buf))){
		    if(0 == regexec(regexp, buf,
				    nmatch, pmatch, 0)){
				*sel_thread_no = cur;
				return TRUE;
			}
		}
		cur++;
		clistp = clistp->next;
	}while(clistp != threadlistp);
	return FALSE;
}
static BOOL search_line_desc(regex_t *regexp, 
				nt_link_tp threadlistp, int *sel_thread_no)
{
	wchar_t *cptr;
	nt_thread_tp threadp;
	nt_link_tp clistp;
	size_t nmatch = 5;
	regmatch_t pmatch[5];
	char buf[256];
	int cur;
	int num;

	clistp = threadlistp->prev;
	num = nt_link_num(threadlistp);
	cur = num - 1;
	do{
		if(cur < *sel_thread_no){
			threadp = (nt_thread_tp)clistp->data;
			cptr = threadp->name;
			if(0 < wcstombs(buf, cptr, sizeof(buf))){
			    if(0 == regexec(regexp, buf,
					    nmatch, pmatch, 0)){
					*sel_thread_no = cur;
					return TRUE;
				}
			}
		}
		cur--;
		clistp = clistp->prev;
	}while(clistp != threadlistp->prev);
	clistp = threadlistp->prev;
	cur = num;
	do{
		if(cur < *sel_thread_no)
			break;

		threadp = (nt_thread_tp)clistp->data;
		cptr = threadp->name;
		if(0 < wcstombs(buf, cptr, sizeof(buf))){
		    if(0 == regexec(regexp, buf,
				    nmatch, pmatch, 0)){
				*sel_thread_no = cur;
				return TRUE;
			}
		}
		cur--;
		clistp = clistp->prev;
	}while(clistp != threadlistp->prev);
	return FALSE;
}

static int get_thread_index_by_seqno(nt_link_tp linkp, int seq_no)
{
	int idx;
	nt_link_tp workp;
	nt_thread_tp threadp;

	workp = linkp;
	idx = 0;
	do{
		threadp = (nt_thread_tp)workp->data;
		if(threadp->seq_no == seq_no)
			return idx;
		workp = workp->next;
		idx++;
	}while(workp != linkp);
	return -1;
}


static int parse_cmd1(const char *param)
{
	int offset, len;
	const char *start, *end;

	assert(param);
	if(0 == strncmp(NT_COMMAND1_SORT_1, param,
			strlen(NT_COMMAND1_SORT_1))){
		offset = strlen(NT_COMMAND1_SORT_1);
	}else if( 0 == strncmp(NT_COMMAND1_SORT_2, param,
			strlen(NT_COMMAND1_SORT_2))){
		offset = strlen(NT_COMMAND1_SORT_2);
	}else{
		return NT_CMD_NONE;
	}
	if(!nt_strtok(param+offset, ' ', &start, &end))
		return NT_CMD_ERR;
	
	len = end - start;
	if(len <= 0)
		return NT_CMD_ERR;

	if(0 == strncmp(start, NT_COMMNAD1_SORT_NUMBER_1,len) ||
		0 == strncmp(start, NT_COMMNAD1_SORT_NUMBER_2,len)){
		return NT_CMD_SORT_NUMBER;
	}else if(0 == strncmp(start, NT_COMMNAD1_SORT_READ_1,len) ||
		0 == strncmp(start, NT_COMMNAD1_SORT_READ_2,len)){
		return NT_CMD_SORT_READ;
	}else if(0 == strncmp(start, NT_COMMNAD1_SORT_UNREAD_1,len) ||
		0 == strncmp(start, NT_COMMNAD1_SORT_UNREAD_2,len)){
		return NT_CMD_SORT_UNREAD;
	}
	return NT_CMD_ERR;
}


static int sort_by_number(void *lhs, void *rhs)
{
	int num1, num2;
	nt_thread_tp lthread, rthread;
	lthread = (nt_thread_tp)lhs;
	rthread = (nt_thread_tp)rhs;
	num1 = lthread->seq_no;
	num2 = rthread->seq_no;

	if(num1 == num2)
		return 0;
	else if(num1 > num2)
		return -1;
	else
		return 1;
}

static int sort_by_unread(void *lhs, void *rhs)
{
	int num1, num2;
	int unread1, unread2;
	nt_thread_tp lthread, rthread;
	lthread = (nt_thread_tp)lhs;
	rthread = (nt_thread_tp)rhs;
	num1 = nt_usr_db_get_read_count_by_dat_name(
		g_ctxp->thread_data_list, lthread->file_name);
	num2 = nt_usr_db_get_read_count_by_dat_name(
		g_ctxp->thread_data_list, rthread->file_name);
	unread1 = -1;
	unread2 = -1;
	if(num1 > 0)
		unread1 = lthread->num_res - num1;
	if(num2 > 0)
		unread2 = rthread->num_res - num2;

	if(unread1 == unread2)
		return 0;
	return (unread1 > unread2) ? 1 : -1;
}

static int sort_by_read(void *lhs, void *rhs)
{
	int num1, num2;
	nt_thread_tp lthread, rthread;
	lthread = (nt_thread_tp)lhs;
	rthread = (nt_thread_tp)rhs;
	num1 = nt_usr_db_get_read_count_by_dat_name(
		g_ctxp->thread_data_list, lthread->file_name);
	num2 = nt_usr_db_get_read_count_by_dat_name(
		g_ctxp->thread_data_list, rthread->file_name);

	if(num1 == num2)
		return 0;
	if(num1 >= 0)
		return (num2 >= 0) ? 0 : 1;
	return (num2 >= 0) ? -1 : 0;
}
