#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <wchar.h>
#include <ncursesw/ncurses.h>

#include "env.h"
#include "utils/nt_std_t.h"
#include "_2ch/model_2ch.h"
#include "ui/disp.h"
#include "ui/disp_string.h"

#define FOLDER_OPEN 1
#define IS_SET_F(a,b) ((a) & (b))
#define SET_F(a,b) ((a) |= (b))
#define UNSET_F(a,b) ((a) &= (~b))

typedef struct tag_category_data_t *category_data_tp;
typedef struct tag_category_data_t{
	int flag;
	nt_category_tp categoryp;
}category_data_t;

typedef struct tag_ctx_board_menu_t *ctx_board_menu_tp;
typedef struct tag_ctx_board_menu_t{
	int category_num;
	int cursor_pos;
	int scroll_pos;
	category_data_tp category_list;
}ctx_board_menu_t;

static BOOL disp_board(nt_window_tp wp, ctx_board_menu_tp ctxp,
		int *cur_line, nt_category_tp categoryp,
		BOOL selected, nt_board_tp *sel_boardpp);
static ctx_board_menu_tp init_context(nt_2ch_model_tp modelp);


static int get_menu_disp_row(ctx_board_menu_tp ctxp);
static int increment_cursor(ctx_board_menu_tp ctxp);
static int decrement_cursor(ctx_board_menu_tp ctxp);
static int adjust_scroll_pos(nt_window_tp wp, ctx_board_menu_tp ctxp);
static int move_cursor_right_colmn(nt_window_tp wp, 
		ctx_board_menu_tp ctxp);
static int move_cursor_left_colmn(nt_window_tp wp, 
		ctx_board_menu_tp ctxp);
static int get_display_folder_contents_count(category_data_tp datap);
static BOOL adjust_scroll_pos2(nt_window_tp wp, 
		ctx_board_menu_tp ctxp, category_data_tp  datap);
static int page_up(nt_window_tp wp, ctx_board_menu_tp ctxp);
static int page_down(nt_window_tp wp, ctx_board_menu_tp ctxp);


int disp_board_menu(nt_window_tp wp, nt_2ch_model_tp modelp)
{
	int cur_line = 0;
	nt_link_tp clistp;
	nt_category_tp categoryp;
	nt_board_tp sel_boardp;
	ctx_board_menu_tp ctxp;
	category_data_tp  datap;
	int i;
	int offset_y, offset_x;
	BOOL selected = FALSE;

	wchar_t *cname;
	attr_t attr;

	assert(modelp);
	assert(wp);

	ctxp = (ctx_board_menu_tp)wp->data;
	if(!ctxp){
		ctxp = init_context(modelp);
		if(!ctxp)
			return DISP_STATE_ERROR;
		wp->data = ctxp;
	}
	switch(wp->key){
	case NT_KEY_UP:
		decrement_cursor(ctxp);
		adjust_scroll_pos(wp, ctxp);
		break;
	case NT_KEY_DOWN: 
		increment_cursor(ctxp);
		adjust_scroll_pos(wp, ctxp);
		break;
	case NT_KEY_LEFT:
		move_cursor_left_colmn(wp, ctxp);
		break;
	case NT_KEY_RIGHT:
		move_cursor_right_colmn(wp, ctxp);
		break;
	case NT_KEY_PAGEUP:
		page_up(wp, ctxp);
		break;
	case NT_KEY_PAGEDOWN:
		page_down(wp, ctxp);
		break;
	case NT_KEY_SELECT: 
		selected = TRUE;
		break;
	}

	clistp = modelp->categorylistp;
	if(clistp == NULL)
		return TRUE;
	for(i = 0; i < ctxp->category_num; i++){
		offset_y = cur_line - ctxp->scroll_pos;
		if(offset_y == (wp->lines*2)){
			return TRUE;
		}else if(offset_y >= wp->lines){
			offset_y -= wp->lines;
			offset_x = wp->cols / 2;
		}else{
			offset_x = 0;
		}

		datap = ctxp->category_list + i;
		categoryp = datap->categoryp;
		if(cur_line == ctxp->cursor_pos){
			attr = WA_BOLD | WA_REVERSE;
			if(selected){
				if(IS_SET_F(datap->flag,FOLDER_OPEN)){
					UNSET_F(datap->flag, FOLDER_OPEN);
				}else{
					SET_F(datap->flag, FOLDER_OPEN);
					if(adjust_scroll_pos2(wp, ctxp, datap)){
						i = -1;
						cur_line = 0;
						selected = FALSE;
						continue;
					}
				}
			}
		}else{
			attr = WA_BOLD;
		}

		wmove(wp->wp, offset_y, offset_x); 

		cname = categoryp->name;
		if(cur_line >= ctxp->scroll_pos){
			nt_add_wstr(wp->wp, L" + ", attr);
			nt_add_wstr(wp->wp, cname, attr);
			nt_add_wstr(wp->wp, L"\n", 0);
		}
		cur_line++;
		if(!IS_SET_F(datap->flag, FOLDER_OPEN))
			continue;
		sel_boardp = NULL;
		if(!disp_board(wp, ctxp, &cur_line, categoryp,
					selected, &sel_boardp))
			return FALSE;
		if(selected && sel_boardp){
			nt_set_selected_board(modelp, categoryp, sel_boardp);
			return DISP_STATE_THREADTITLE;
		}
	}

	touchwin(wp->wp);
	wrefresh(wp->wp);

	return DISP_STATE_BOARDMENU;
}

static BOOL disp_board(nt_window_tp wp, ctx_board_menu_tp ctxp,
		int *cur_line, nt_category_tp categoryp, 
		BOOL selected, nt_board_tp *sel_boardpp)
{
	nt_link_tp blistp, blist_tmp;
	nt_board_tp boardp;
	wchar_t *bname;
	int tmp;
	int offset_y;
	attr_t attr;

	assert(categoryp);

	blistp = categoryp->boardlistp;

	if(blistp == NULL)
		return TRUE;
	blist_tmp = blistp;
	do{
		offset_y = *cur_line - ctxp->scroll_pos;
		if(offset_y == (wp->lines*2)){
			return TRUE;
		}else if(offset_y >= wp->lines){
			offset_y -= wp->lines;
			wmove(wp->wp, offset_y, wp->cols / 2); 
		}else{
			wmove(wp->wp, offset_y, 0); 
		}
		if(*cur_line == ctxp->cursor_pos){
			if(selected){
				*sel_boardpp = (nt_board_tp)blist_tmp->data;
				return TRUE;
			}
			attr = WA_REVERSE;
		}else{
			attr = 0;
		}
		boardp = (nt_board_tp)blist_tmp->data;
		bname = boardp->name;
		if((*cur_line) >= ctxp->scroll_pos){
			nt_add_wstr(wp->wp, L"    - ", attr);
			nt_add_wstr(wp->wp, bname, attr);
			nt_add_wstr(wp->wp, L"\n", attr);
		}
		tmp = *cur_line;
		*cur_line = tmp + 1;
		blist_tmp = blist_tmp->next;
	}while(blist_tmp != blistp);
	


	return TRUE;
}


static int adjust_scroll_pos(nt_window_tp wp, 
			ctx_board_menu_tp ctxp)
{
	int scr_lines = wp->lines;
	int cursor_pos = ctxp->cursor_pos;
	int scroll_pos = ctxp->scroll_pos;

	if(cursor_pos < scroll_pos){
		scroll_pos = cursor_pos;
		ctxp->scroll_pos = scroll_pos;
	}else if((scroll_pos + (scr_lines*2)) <= cursor_pos){
		scroll_pos = cursor_pos - (scr_lines*2) + 1;
		ctxp->scroll_pos = scroll_pos;
	}
	return 0;
}

static BOOL adjust_scroll_pos2(nt_window_tp wp, 
		ctx_board_menu_tp ctxp, category_data_tp  datap)
{
	int scr_lines = wp->lines;
	int cursor_pos = ctxp->cursor_pos;
	int scroll_pos = ctxp->scroll_pos;
	int contents_count = get_display_folder_contents_count(datap);
	if(contents_count == 0)
		return FALSE;
	cursor_pos += contents_count;

	if(cursor_pos < scroll_pos){
		scroll_pos = cursor_pos;
		ctxp->scroll_pos = scroll_pos;
		return TRUE;
	}else if((scroll_pos + (scr_lines*2)) <= cursor_pos){
		scroll_pos = cursor_pos - (scr_lines*2) + 1;
		ctxp->scroll_pos = scroll_pos;
		return TRUE;
	}
	return FALSE;
}


static int page_up(nt_window_tp wp, ctx_board_menu_tp ctxp)
{
	int scroll_pos = ctxp->scroll_pos;
	int cursor_pos = ctxp->cursor_pos - scroll_pos;

	scroll_pos -= wp->lines;
	if(scroll_pos < 0)
		scroll_pos = 0;
	ctxp->scroll_pos = scroll_pos;
	ctxp->cursor_pos = scroll_pos + cursor_pos;
	return 0;
}
static int page_down(nt_window_tp wp, ctx_board_menu_tp ctxp)
{
	int scr_lines = wp->lines;
	int scroll_pos = ctxp->scroll_pos;
	int cursor_pos = ctxp->cursor_pos - scroll_pos;
	int menu_disp_row = get_menu_disp_row(ctxp);

	menu_disp_row -= scr_lines*2;
	if(menu_disp_row <= 0){
		ctxp->scroll_pos = 0;
		ctxp->cursor_pos = cursor_pos;
		return 0;
	}
	scroll_pos += wp->lines;
	if(scroll_pos >= menu_disp_row)
		scroll_pos = menu_disp_row -1;
	ctxp->scroll_pos = scroll_pos;
	ctxp->cursor_pos = scroll_pos + cursor_pos;
	return 0;
}


static ctx_board_menu_tp init_context(nt_2ch_model_tp modelp)
{
	int idx;
	nt_link_tp clistp;
	ctx_board_menu_tp ctxp = 
		(ctx_board_menu_tp)calloc(1,sizeof(ctx_board_menu_t));
	if(!ctxp)
		return NULL;

	clistp = modelp->categorylistp->next;
	if(!clistp)
		return ctxp;

	ctxp->category_num++;
	while(modelp->categorylistp != clistp){
		ctxp->category_num++;
		clistp = clistp->next;
	}
	/* Valgrind's error messages suppress. */
	ctxp->category_list = 
		calloc(ctxp->category_num, sizeof(category_data_t));
	if(!ctxp->category_list){
		free(ctxp);
		return NULL;
	}

	idx = 0;
	clistp = modelp->categorylistp;
	ctxp->category_list[idx++].categoryp = 
			(nt_category_tp)clistp->data;
	clistp = clistp->next;
	while(modelp->categorylistp != clistp){
		ctxp->category_list[idx++].categoryp = 
			(nt_category_tp)clistp->data;
		clistp = clistp->next;
	}
	assert(idx == ctxp->category_num);	

	return ctxp;
}

void free_board_menu_ctx(void *ptr)
{
	if(!ptr)
		return;
	ctx_board_menu_tp ctxp = 
			(ctx_board_menu_tp)ptr;
	free(ctxp->category_list);
	free(ctxp);
}


static int move_cursor_right_colmn(nt_window_tp wp, 
		ctx_board_menu_tp ctxp)
{
	int scr_lines = wp->lines;
	int cur_pos = ctxp->cursor_pos;
	int scroll_pos = ctxp->scroll_pos;
	int menu_disp_num;

	if(cur_pos - scroll_pos >= scr_lines)
		return cur_pos;
	cur_pos += scr_lines;
	menu_disp_num = get_menu_disp_row(ctxp);
	if(menu_disp_num > cur_pos)
		ctxp->cursor_pos = cur_pos;
	else 
		ctxp->cursor_pos = menu_disp_num - 1;
	return ctxp->cursor_pos;
}

static int move_cursor_left_colmn(nt_window_tp wp, 
		ctx_board_menu_tp ctxp)
{
	int scr_lines = wp->lines;
	int cur_pos = ctxp->cursor_pos;
	int scroll_pos = ctxp->scroll_pos;
	
	if(cur_pos - scroll_pos < scr_lines)
		return cur_pos;
	cur_pos -= scr_lines;

	if(0 > cur_pos-scroll_pos)
		ctxp->cursor_pos = scroll_pos;
	else 
		ctxp->cursor_pos = cur_pos;
	return ctxp->cursor_pos;
}

static int increment_cursor(ctx_board_menu_tp ctxp)
{
	int cur_pos = ctxp->cursor_pos + 1;
	int menu_disp_num = get_menu_disp_row(ctxp);
	if(menu_disp_num > cur_pos)
		ctxp->cursor_pos = cur_pos;
	return ctxp->cursor_pos;
}

static int decrement_cursor(ctx_board_menu_tp ctxp)
{
	int cur_pos = ctxp->cursor_pos - 1;
	if(0 <= cur_pos)
		ctxp->cursor_pos = cur_pos;
	return ctxp->cursor_pos;
}

static int get_display_folder_contents_count(category_data_tp datap)
{
	if(!IS_SET_F(datap->flag, FOLDER_OPEN))
		return 0;
	
	if(!datap->categoryp->boardlistp)
		return 0;
	
	return nt_link_num(datap->categoryp->boardlistp);
}

static int get_menu_disp_row(ctx_board_menu_tp ctxp)
{
	int i;
	int num = 0;
	category_data_tp datap;

	for(i = 0; i < ctxp->category_num; i++){
		num++;
		datap = ctxp->category_list + i;
		if(IS_SET_F(datap->flag, FOLDER_OPEN)){
			if(datap->categoryp->boardlistp){
				num += nt_link_num(
					datap->categoryp->boardlistp);
			}
		}
	}
	return num;
}

