#include <iostream>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/ublas/vector.hpp>

#include <nsim/dopri853.h>
#include <nsim/hiha54.h>
#include "model.h"
#include "handle.h"

typedef boost::numeric::ublas::vector<double> vector_type;

template <typename Driver>
struct main_
{
    Driver driver;

    main_(int argc, char** argv, int start = 1) : driver()
    {
        for (int i(start); i < argc; ++i) {
            std::string arg(argv[i]);
            if (arg == "-tol" && i+1 < argc) {
                driver.set_tolerance(boost::lexical_cast<double>(argv[++i]));
            } else if (arg == "-safe" && i+1 < argc) {
                driver.set_safety(boost::lexical_cast<double>(argv[++i]));
            } else if (arg == "-min_fac" && i+1 < argc) {
                driver.set_min_fac(boost::lexical_cast<double>(argv[++i]));
            } else if (arg == "-max_fac" && i+1 < argc) {
                driver.set_max_fac(boost::lexical_cast<double>(argv[++i]));
            } else if (arg == "-min_step" && i+1 < argc) {
                driver.set_min_step(boost::lexical_cast<double>(argv[++i]));
            } else if (arg == "-max_step" && i+1 < argc) {
                driver.set_max_step(boost::lexical_cast<double>(argv[++i]));
            } else {
                throw std::runtime_error(
std::string("Usage: ") + argv[0] + " [driver_type] [options]\n"
"driver_type:\n"
"    -dopri853\n"
"    -hia54\n"
"options:\n"
"    -tol val\n"
"    -safe val\n"
"    -min_fac val\n"
"    -max_fac val\n"
"    -min_step val\n"
"    -max_step val\n"
"Report bugs to <takekawa@users.sourceforge.jp>.");
            }
        }
    }

    void run()
    {
        vector_type x(3);
        x[0] = 1.0;

        test::model model;
        test::handle<vector_type> handle(x);
        driver.integrate(model, handle);
        for (double t(0.0); t < 1000.0; t += 0.1) {
            test::handle<vector_type>::state_type x(handle.x(t));
            std::cout << t <<
                '\t' << x(0) << '\t' << x(1) << '\t' << x(2) << '\n';
        }
    }
};

int main(int argc, char** argv)
try {
    std::string arg1(argc < 2 ? "" : argv[1]);
    if (arg1 == "-dopri853") {
        main_<nsim::dopri853::integrator<vector_type> > app(argc, argv, 2);
        app.run();
    } else if (arg1 == "-hiha54") {
        main_<nsim::hiha54::integrator<vector_type> > app(argc, argv, 2);
        app.run();
    } else {
        main_<nsim::hiha54::integrator<vector_type> > app(argc, argv);
        app.run();
    }
    return 0;
} catch (std::exception& err) {
    std::cerr << err.what() << '\n';
    return 1;
}
