// $Id: phasemodel.h 27 2004-07-03 07:03:56Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_PHASEMODEL_H
#define NSIM_PHASEMODEL_H

#include <cmath> // std::abs
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/matrix.hpp>

namespace nsim
{

template <typename M, typename L>
class phasemodel
{
public:
    typedef std::size_t size_type;
    typedef M basemodel_type;
    typedef L limitcycle_type;

    phasemodel(basemodel_type const& basemodel,
            limitcycle_type const& limitcycle, double ascale = 1e-6) :
        basemodel_(basemodel), limitcycle_(limitcycle), ascale_(ascale) {}

    void set_ascale(double val) { ascale_ = val; }

    size_type size() const { return basemodel_.size(); }
    double cycle() const { return limitcycle_.cycle(); }
    basemodel_type const& basemodel() const { return basemodel_; }
    limitcycle_type const& limitcycle() const { return limitcycle_; }

    template <typename V>
        double scale(double t, V const& z) const
        {
            return inner_prod(f(t), z);
        }

    template <typename V>
        void init(double t, V& z) const
        {
            typedef boost::numeric::ublas::scalar_vector<double> svector;
            z = svector(size(), 1.0);
            z /= scale(t, z);
        }

    template <typename V1, typename V2>
        void deriv(double t, V1 const& z, V2& dzdt) const
        {
            dzdt.assign(-prod(z, j(t)));
        }

    template <typename V1, typename V2>
        double norm(V1& e, double t, V2 const& z) const
        {
            double s(std::abs(scale(t, z)-1.0));
            for (typename V1::size_type i(0); i < e.size(); ++i)
                e[i] /= std::abs(z[i]) + ascale_;
            return s + norm_2(e);
        }

private:
    typedef typename limitcycle_type::state_type state_type;
    typedef typename limitcycle_type::deriv_type deriv_type;
    typedef boost::numeric::ublas::vector<double> vector_type;
    typedef boost::numeric::ublas::matrix<double> matrix_type;

    basemodel_type const& basemodel_;
    limitcycle_type const& limitcycle_;
    double ascale_;
    mutable vector_type x_;
    mutable matrix_type j_;

    state_type x(double t) const
    {
        return limitcycle_.x(t);
    }

    deriv_type f(double t) const
    {
        return limitcycle_.d(t);
    }

    matrix_type const& j(double t) const
    {
        x_ = x(t);
        return j(t, x_);
    }

    matrix_type const& j(double t, vector_type const& x) const
    {
        j_.resize(x.size(), x.size());
        basemodel_.jacob(t, x, j_);
        return j_;
    }
};

}

#endif
