// $Id: interpolator.h 27 2004-07-03 07:03:56Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_INTERPOLATOR_H
#define NSIM_INTERPOLATOR_H

#include <boost/numeric/ublas/iterator.hpp>
#include <boost/numeric/ublas/vector_expression.hpp>

namespace nsim
{

template <typename V> class state_proxy;
template <typename V> class deriv_proxy;

template <typename V>
class interpolator
{
public:
    typedef std::size_t size_type;
    typedef typename V::value_type value_type;
    typedef V vector_type;
    typedef state_proxy<V> state_type;
    typedef deriv_proxy<V> deriv_type;

    interpolator() {}

    template <typename result_type>
        interpolator(result_type const& r) :
            t_(r.t0()), h_(r.dt())
        {
            v_[0] = r.x0();
            v_[1] = r.dx();
            v_[2] = (r.d0() - v_[1]);
            v_[3] = (v_[1] - r.d1()) - v_[2];
        }

    interpolator(double t, double h,
            vector_type const& x0, vector_type const& dx,
            vector_type const& d0, vector_type const& d1) :
        t_(t), h_(h)
    {
        v_[0] = x0;
        v_[1] = dx;
        v_[2] = d0 - dx;
        v_[3] = dx - d1 - v_[2];
    }

    void set_t0(double val) { t_ = val; }
    void scale_x1(double val)
    {
        v_[1].assign(val*v_[1] - (1.0 - val)*v_[0]);
    }

    size_type size() const { return v_[0].size(); }

    double dt() const { return h_; }
    double t0() const { return t_; }
    double t1() const { return t_ + h_; }

    vector_type const& x0() const { return v_[0]; }

    value_type x0(size_type i) const { return v_[0](i); }
    value_type x1(size_type i) const { return v_[0](i) + h_*v_[1](i); }

    state_type x(double t) const;
    deriv_type d(double t) const;

    double find_time(size_type i, double xt) const
    {
        size_type const max_ = 20;
        double tt(t0());
        double xx(v_[0](i));
        double dd(v_[1](i) + v_[2](i));
        for (size_type n(1); n < max_; ++n) {
            double delta((xt - xx)/dd);
            tt += delta;
            if (std::abs(delta) < std::abs(tt*1e-15))
                return tt;
            if ((tt - t0())*(tt - t0() - dt()) > 0.0)
                tt = t0() + dt()/n;
            xx = x(tt)(i);
            dd = d(tt)(i);
        }
        throw std::runtime_error("newton");
    }

    void swap(interpolator& p)
    {
        std::swap(t_, p.t_);
        std::swap(h_, p.h_);
        for (size_type i(0); i < 4; ++i)
            v_[i].swap(p.v_[i]);
    }

private:
    friend class state_proxy<V>;
    friend class deriv_proxy<V>;
    double t_;
    double h_;
    vector_type v_[4];

    double x(double th, size_type i) const
    {
        return v_[0][i]
            + h_*th*(v_[1][i] + (1.0 - th)*(v_[2][i] + th*(v_[3][i])));
    }

    double d(double th, size_type i) const
    {
        return v_[1][i] + (1.0 - 2.0*th)*v_[2][i] + th*(2.0 - 3.0*th)*v_[3][i];
    }
};

template <typename V>
class state_proxy :
    public boost::numeric::ublas::vector_expression<state_proxy<V> >
{
    typedef state_proxy self_type;
    typedef boost::numeric::ublas::dense_random_access_iterator_tag
        iterator_tag;
    typedef boost::numeric::ublas::vector_expression<self_type> expression_type;

public:
    typedef V vector_type;
    typedef typename vector_type::value_type value_type;
    typedef typename vector_type::size_type size_type;
    typedef typename vector_type::difference_type difference_type;
    typedef typename vector_type::const_reference const_reference;
    typedef typename vector_type::reference reference;
    typedef typename vector_type::const_pointer const_pointer;
    typedef typename vector_type::pointer pointer;
    typedef self_type const const_closure_type;
    typedef boost::numeric::ublas::
        indexed_const_iterator<self_type, iterator_tag> const_iterator;
    typedef const_iterator iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    state_proxy(interpolator<V> const& ref, double t) :
        ref_(ref), th_((t - ref.t_)/ref.h_) {}

    size_type size() const { return ref_.size(); }

    value_type operator()(size_type i) const { return ref_.x(th_, i); }
    using expression_type::operator();

    value_type operator[](size_type i) const { return operator()(i); }

    const_iterator find(size_type i) const { return const_iterator(*this, i); }

    const_iterator begin() const { return find(0); }
    const_iterator end() const { return find(size()); }

    const_reverse_iterator rbegin() const
    {
        return const_reverse_iterator(end());
    }

    const_reverse_iterator rend() const
    {
        return const_reverse_iterator(begin());
    }

private:
    interpolator<V> const& ref_;
    double thh_;
    double th_;
};

template <typename V>
class deriv_proxy :
    public boost::numeric::ublas::vector_expression<deriv_proxy<V> >
{
    typedef deriv_proxy self_type;
    typedef boost::numeric::ublas::dense_random_access_iterator_tag
        iterator_tag;
    typedef boost::numeric::ublas::vector_expression<self_type> expression_type;

public:
    typedef V vector_type;
    typedef typename vector_type::value_type value_type;
    typedef typename vector_type::size_type size_type;
    typedef typename vector_type::difference_type difference_type;
    typedef typename vector_type::const_reference const_reference;
    typedef typename vector_type::reference reference;
    typedef typename vector_type::const_pointer const_pointer;
    typedef typename vector_type::pointer pointer;
    typedef self_type const const_closure_type;
    typedef boost::numeric::ublas::indexed_const_iterator
        <self_type, iterator_tag> const_iterator;
    typedef const_iterator iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    deriv_proxy(interpolator<V> const& ref, double t) :
        ref_(ref), th_((t - ref.t_)/ref.h_) {}

    size_type size() const { return ref_.size(); }

    value_type operator()(size_type i) const { return ref_.d(th_, i); }
    using expression_type::operator();

    value_type operator[](size_type i) const { return operator()(i); }

    const_iterator find(size_type i) const { return const_iterator(*this, i); }

    const_iterator begin() const { return find(0); }
    const_iterator end() const { return find(size()); }

    const_reverse_iterator rbegin() const
    {
        return const_reverse_iterator(end());
    }

    const_reverse_iterator rend() const
    {
        return const_reverse_iterator(begin());
    }

private:
    interpolator<V> const& ref_;
    double th_;
};

template <typename V>
inline state_proxy<V>
interpolator<V>::x(double t) const
{
    return state_type(*this, t);
}

template <typename V>
inline deriv_proxy<V>
interpolator<V>::d(double t) const
{
    return deriv_type(*this, t);
}

}

#endif
