// $Id: integrator_base.h 27 2004-07-03 07:03:56Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_INTEGRATOR_BASE_H
#define NSIM_INTEGRATOR_BASE_H

#include <limits>
#include <cmath>     // std::abs, std::pow
#include <algorithm> // std::min, std::max
#include <iostream>

namespace nsim
{

double const AUTOSTEP = std::numeric_limits<double>::infinity();
double const AUTOSTEP_BACKWARD = -AUTOSTEP;

template <typename R>
class integrator_base
{
public:
    typedef std::size_t size_type;
    typedef R result_type;
    typedef typename result_type::vector_type vector_type;
    
    static unsigned const ORDER = result_type::ORDER;

    integrator_base() :
        min_step_(1e-10), max_step_(AUTOSTEP), min_fac_(0.5), max_fac_(2.0),
        safety_(0.9), tol_(1e-6), max_rep_(0xffffffff) {}

    void set_min_step(double val) { min_step_ = val; }
    void set_max_step(double val) { max_step_ = val; }
    void set_min_fac(double val) { min_fac_ = val; }
    void set_max_fac(double val) { max_fac_ = val; }
    void set_safety(double val) { safety_ = val; }
    void set_tolerance(double val) { tol_ = val; }
    void set_max_rep(size_type val) { max_rep_ = val; }

    double min_step() const { return min_step_; }
    double max_step() const { return max_step_; }
    double min_fac() const { return min_fac_; }
    double max_fac() const { return max_fac_; }
    double safety() const { return safety_; }
    double tolerance() const { return tol_; }
    size_type max_rep() const { return max_rep_; }
    size_type success() const { return success_; }
    size_type failed() const { return failed_; }
    size_type trial() const { return success_ + failed_; }

    template <typename model_type, typename handle_type>
        void integrate(model_type& eq, handle_type& handle)
        {
            result_type result;
            handle.start(eq, result.t_, result.x_, result.h_);

            success_ = 0;
            failed_ = 0;
            result.init(eq);
            if (std::abs(result.h_) == AUTOSTEP)
                first_(eq, (result.h_ < 0.0), result);
            while (handle.is_active(eq, result.t_, result.x_)) {
                step_(eq, result);
                handle.accepted(eq, result);
                result.update();
                if (handle.modify(eq, result.t_, result.x_, result.h_))
                    result.reset(eq);
            }

            handle.finish(eq, result.t_, result.x_, result.h_);
            std::cerr << "# finish " << trial() << '\t' << success() << '\n';
        }

private:
    double min_step_, max_step_, min_fac_, max_fac_, safety_, tol_;
    size_type max_rep_;
    size_type success_;
    size_type failed_;

    template <typename model_type>
        void first_(model_type const& eq, bool backward, result_type& result)
        {
            result.xout_.assign(result.x_);
            double xs(eq.norm(result.xout_, result.t_, result.x_));

            result.xout_.assign(result.k_[0]);
            double ds(eq.norm(result.xout_, result.t_, result.x_));

            double h0_abs(((xs > 1e-5) || (ds > 1e-5)) ? 1e-2*xs/ds : 1e-6);
            double h0(backward ? -h0_abs : h0_abs);
            result.xout_.assign(result.x_ + h0*result.k_[0]);
            eq.deriv(result.t_ + h0, result.xout_, result.k_[1]);

            result.xout_.assign(result.k_[1] - result.k_[0]);
            double dds(eq.norm(result.xout_, result.t_, result.x_));

            double err(std::max(ds, dds));
            double h1_abs((err < 1e-15)
                    ? std::max(1e-3*h0_abs, 1e-6)
                    : safety()*std::pow(err, -1.0/ORDER));
            double h_abs(std::min(1e2*h0_abs, h1_abs));
            result.h_ = backward ? -h_abs : h_abs;
            //std::cerr << "# estimate step: " << result.h_ << '\n';
        }

    template <typename model_type>
        void step_(model_type const& eq, result_type& result)
        {
            for (;;) {
                if (std::abs(result.h_) < min_step())
                    result.h_ = (result.h_ < 0.0 ? -min_step() : min_step());
                else if (std::abs(result.h_) > max_step())
                    result.h_ = (result.h_ < 0.0 ? -max_step() : max_step());
                result.calc_k(eq);
                double err(result.error(eq)/tolerance());
                double fac(::isnan(err) ? min_fac()
                        : safety()*std::pow(err, -1.0/ORDER));
                if (fac >= safety()) {
                    ++success_;
                    result.calc_x(eq);
                    result.hnew_ = result.h_*std::min(fac, max_fac());
                    break;
                }
                ++failed_;
                if (std::abs(result.h_) == min_step()) {
                    if (::isnan(err))
                        throw std::runtime_error("bad step with min h");
                    result.calc_x(eq);
                    result.hnew_ = result.h_*std::max(fac, min_fac());
                    std::cerr << "# warning: min step at " << result.t_ << "\n";
                    break;
                } 
                result.h_ *= std::max(fac, min_fac());
                if (trial() >= max_rep())
                    throw std::runtime_error("max step num");
            }
        }
};

}

#endif
