// $Id: hiha54_result.h 26 2004-07-03 06:22:39Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_HIHA54_RESULT_H
#define NSIM_HIHA54_RESULT_H

#include <nsim/hiha54_const.h>

namespace nsim
{

template <typename R> class integrator_base;

namespace hiha54
{

template <typename V>
class result
{
public:
    typedef std::size_t size_type;
    typedef V vector_type;

    static size_type const ORDER = 5;
    static size_type const STAGE = 9;

    double t0() const { return t_; }
    double t1() const { return tout_; }
    double dt() const { return h_; }
    double hnew() const { return hnew_; }

    vector_type const& x0() const { return x_; }
    vector_type const& x1() const { return xout_; }
    vector_type const& dx() const { return k_[7]; }

    vector_type const& d0() const { return k_[0]; }
    vector_type const& d1() const { return k_[6]; }

    void update()
    {
        t_ += h_;
        h_ = hnew_;
        x_.swap(xout_);
        k_[0].swap(k_[6]);
    }

    template <typename model_type>
        void init(model_type const& eq)
        {
            xout_.resize(x_.size());
            for (size_type i(0); i < STAGE; ++i)
                k_[i].resize(x_.size());
            reset(eq);
        }

    template <typename model_type>
        void reset(model_type const& eq)
        {
            eq.deriv(t_, x_, k_[0]);
        }

    template <typename model_type>
        void calc_k(model_type const& eq);

    template <typename model_type>
        void calc_x(model_type const& eq);

    template <typename model_type>
        double error(model_type const& eq);

    double stiffness() const
    {
        return norm_inf(element_div(dt()*(d1() - d0()), x1() - x0()));
    }

private:
    friend class nsim::integrator_base<result>;
    double t_;
    double h_;
    double tout_;
    double hnew_;
    vector_type x_;
    vector_type xout_;
    vector_type k_[STAGE];
};

template <typename V>
template <typename model_type>
void result<V>::calc_k(model_type const& eq)
{
    k_[1].assign(A[1][0] * k_[0]);
    eq.deriv(t_ + h_*C[1], xout_.assign(x_ + h_*k_[1]), k_[1]);

    k_[2].assign(A[2][0] * k_[0]);
    k_[2].plus_assign(A[2][1] * k_[1]);
    eq.deriv(t_ + h_*C[2], xout_.assign(x_ + h_*k_[2]), k_[2]);

    k_[3].assign(A[3][0] * k_[0]);
    k_[3].plus_assign(A[3][2] * k_[2]);
    eq.deriv(t_ + h_*C[3], xout_.assign(x_ + h_*k_[3]), k_[3]);

    k_[4].assign(A[4][0] * k_[0]);
    for (size_type j(1); j < 4; ++j)
        k_[4].plus_assign(A[4][j] * k_[j]);
    eq.deriv(t_ + h_*C[4], xout_.assign(x_ + h_*k_[4]), k_[4]);

    k_[5].assign(A[5][0] * k_[0]);
    for (size_type j(1); j < 5; ++j)
        k_[5].plus_assign(A[5][j] * k_[j]);
    eq.deriv(tout_ = t_ + h_, k_[1].assign(x_ + h_*k_[5]), k_[5]);

    k_[7].assign(A[6][0] * k_[0]);
    for (size_type j(2); j < 6; ++j)
        k_[7].plus_assign(A[6][j] * k_[j]);
    eq.deriv(tout_, xout_.assign(x_ + h_*k_[7]), k_[6]);
}

template <typename V>
template <typename model_type>
void result<V>::calc_x(model_type const& eq)
{
}

template <typename V>
template <typename model_type>
double result<V>::error(model_type const& eq)
{
    k_[8].assign(E[0] * k_[0]);
    for (size_type j(2); j < 7; ++j)
        k_[8].plus_assign(E[j] * k_[j]);
    return eq.norm(k_[8] *= h_, tout_, xout_);
}

}
}

#endif
