// $Id: phasecycle.h 27 2004-07-03 07:03:56Z takekawa $
// Copyright (C) 2004  Takashi Takekawa
// This file is part of the Nsim Library.

// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// long with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307 USA.

#ifndef NSIM_PHASECYCLE_H
#define NSIM_PHASECYCLE_H

#include <nsim/orbit.h>

namespace nsim
{

template <typename V>
class phasecycle
{
public:
    typedef V vector_type;
    typedef typename vector_type::size_type size_type;
    typedef orbit<V> orbit_type;
    typedef typename orbit_type::interp_type interp_type;
    typedef typename interp_type::state_type state_type;
    typedef typename interp_type::deriv_type deriv_type;

    phasecycle(double eps = 1e-3, double delta = 0.0, double ascale = 1e-6) :
        eps_(eps), delta_(delta), ascale_(ascale) {}

    double cycle() const { return cycle_; }
    orbit_type orbit() const { return orbit_; }

    state_type x(double t) const
    {
        t = std::fmod(t-delta_, cycle_);
        if (t < 0.0)
            t += cycle_;
        return orbit_.x(t+delta_);
    }

    deriv_type d(double t) const
    {
        t = std::fmod(t-delta_, cycle_);
        if (t < 0.0)
            t += cycle_;
        return orbit_.d(t+delta_);
    }


    template <typename model_type>
        void start(model_type& eq, double& t, vector_type& z, double& h)
        {
            std::cerr << "# start phasecycle\n";
            orbit_.clear();
            cycle_ = 0.0;
            h = AUTOSTEP_BACKWARD;
            t = eq.cycle() + delta_;
            eq.init(t, z);
        }

    template <typename model_type>
        bool is_active(model_type const& eq,
                double t, vector_type const& z) const
        {
            return (t > delta_);
        }

    template <typename model_type, typename Result>
        void accepted(model_type const& eq, Result const& r)
        {
            orbit_.insert(r);
        }

    template <typename model_type>
        bool modify(model_type& eq, double& t, vector_type& z, double& h)
        {
            double scale(eq.scale(t, z));
            interp_type& interp(orbit_.begin()->second);
            interp.scale_x1(scale);
            z /= scale;
            if (t < delta_) {
                state_type z0(interp.x(delta_));
                state_type z1(orbit_.rbegin()->second.x(eq.cycle() + delta_));
                double diff(0.0);
                for (size_type j(0); j < z.size(); ++j)
                    diff = std::max(diff, norm_(z0(j), z1(j)));
                std::cerr << "# " << diff << '\n';
                if (diff > eps_) {
                    t += eq.cycle();
                    interp.set_t0(interp.t0() + eq.cycle());
                    orbit_type tmpo;
                    tmpo.insert_temporary(interp);
                    orbit_.swap(tmpo);
                }
            }
            return true;
        }

    template <typename model_type>
        void finish(model_type const& eq, double& t, vector_type& z, double& h)
        {
            cycle_ = eq.cycle();
        }

private:
    orbit_type orbit_;
    double cycle_;
    double eps_;
    double delta_;
    double ascale_;

    double norm_(double x, double y) const
    {
        return std::abs(x-y) / (std::max(std::abs(x), std::abs(y)) + ascale_);
    }
};

}

#endif
