﻿//  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer
//
//  Copyright(C) 2001-2006 Taku Kudo <taku@chasen.org>
//  Copyright(C) 2004-2006 Nippon Telegraph and Telephone Corporation
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace NMeCab.Core
{
    public static class Utils
    {
        public static double LogSumExp(double x, double y, bool flg)
        {
            const double MinusLogEpsilon = 50.0;

            if (flg) return y;  // init mode
            double vMin = Math.Min(x, y);
            double vMax = Math.Max(x, y);
            if (vMax > vMin + MinusLogEpsilon)
                return vMax;
            else
                return vMax + Math.Log(Math.Exp(vMin - vMax) + 1.0);
        }

        /// <summary>
        /// ビットフィールド値を取り出す
        /// </summary>
        /// <param name="bits">ビット列を表すUint32値</param>
        /// <param name="offset">開始ビット位置</param>
        /// <param name="len">ビット長</param>
        /// <returns>ビットフィールド値</returns>
        public static uint GetBitField(uint bits, int offset, int len)
        {
            //BitVector32構造体でもよいが、あえて異なる実装とする。
            uint mask = ~(uint.MaxValue << len);
            return (bits >> offset) & mask;
        }

        /// <summary>
        /// 特定位置のビット値を取り出す
        /// </summary>
        /// <param name="bits">ビット列を表すUint32値</param>
        /// <param name="offset">ビット位置</param>
        /// <returns>ビット値</returns>
        public static bool GetFlag(uint bits, int offset)
        {
            //BitVector32構造体でもよいが、あえて異なる実装とする。
            uint mask = 0x1u << offset;
            return (bits & mask) != 0;
        }

        /// <summary>
        /// ビット値の比較
        /// </summary>
        /// <param name="flags1">ビット列1を表すUint32値</param>
        /// <param name="flags2">ビット列2を表すUint32値</param>
        /// <param name="offset">開始ビット位置</param>
        /// <param name="len">ビット長</param>
        /// <returns>比較結果</returns>
        public static bool CompareFlags(uint flags1, uint flags2, int offset, int len)
        {
            uint mask = ~(uint.MaxValue << len) << offset;
            return (flags1 & flags2 & mask) != 0;
        }

        /// <summary>
        /// バイト配列の中から終端が\0で表された文字列を取り出す。
        /// バイト配列の長さはInt32.MaxValueを超えていても良い。
        /// </summary>
        /// <param name="bytes">バイト配列</param>
        /// <param name="enc">文字エンコーディング</param>
        /// <returns>文字列（\0は含まない）</returns>
        public static string GetString(byte[] bytes, Encoding enc)
        {
            return Utils.GetString(bytes, 0L, enc);
        }

        /// <summary>
        /// バイト配列の中から終端が\0で表された文字列を取り出す。
        /// バイト配列の長さはInt32.MaxValueを超えていても良い。
        /// </summary>
        /// <param name="bytes">バイト配列</param>
        /// <param name="index">オフセット位置</param>
        /// <param name="enc">文字エンコーディング</param>
        /// <returns>文字列（\0は含まない）</returns>
        public unsafe static string GetString(byte[] bytes, long offset, Encoding enc)
        {
            fixed (byte* pBytes = bytes)
            {
                return Utils.GetString(pBytes + offset, enc);
            }
        }

        /// <summary>
        /// バイト配列の中から終端が\0で表された文字列を取り出す。
        /// バイト配列の長さはInt32.MaxValueを超えていても良い。
        /// </summary>
        /// <param name="bytes">デコードする最初のバイトへのポインタ</param>
        /// <param name="enc">文字エンコーディング</param>
        /// <returns>文字列（\0は含まない）</returns>
        public unsafe static string GetString(byte* bytes, Encoding enc)
        {
            //バイト長のカウント
            int byteCount = 0;
            while (*bytes != 0x0)
            {
                checked { byteCount++; } //文字列のバイト長がInt32.MaxValueを超えたならエラー
                bytes++;
            }
            bytes -= byteCount;

            //生成されうる最大文字数のバッファを確保
            int maxCharCount = enc.GetMaxCharCount(byteCount);
            fixed (char* buff = new char[maxCharCount])
            {
                //バイト配列を文字列にデコード
                int len = enc.GetChars(bytes, byteCount, buff, maxCharCount);
                return new string(buff, 0, len);
            }
        }

        //public static unsafe uint TurnEndianness(uint value) 
        //{
        //    //System.Net.IPAddress.HostToNetworkOrderでもよいが、

        //    byte* pVal = (byte*)&value;

        //    byte* pDist = stackalloc byte[4];
        //    pDist += 3;

        //    *pDist-- = *pVal++;
        //    *pDist-- = *pVal++;
        //    *pDist-- = *pVal++;
        //    *pDist = *pVal;

        //    return *(uint*)pDist;
        //}
    }
}
