/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.automata.lr.ContextFreeGrammar;
import net.morilib.automata.lr.ContextFreeRule;
import net.morilib.automata.lr.GrammarSymbol;
import net.morilib.automata.lr.LRAction;
import net.morilib.automata.lr.Nonterminal;
import net.morilib.automata.lr.Terminal;
import net.morilib.deculture.misc.CharacterTerminal;
import net.morilib.deculture.misc.ConstantNonterminal;
import net.morilib.deculture.misc.ConstantTerminal;
import net.morilib.deculture.misc.DecultureException;
import net.morilib.deculture.misc.PrecedenceTerminal;
import net.morilib.deculture.misc.SubautomatonTerminal;
import net.morilib.deculture.parser.DecultureDefinition;
import net.morilib.nina.LRObject;
import net.morilib.nina.Quadro;
import net.morilib.nina.translate.sh.ShNinatBuiltInCommands;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShProcess;

/**
 * 
 * 
 * @author MORIGUCHI, Yuichiro 2006/07/09
 */
public class LRTranslatorJava extends LRTranslator {

	private abstract class Puts implements ShProcess {

		//
		abstract boolean isput(LRObject lr, int st, GrammarSymbol g);

		//
		abstract void put(PrintStream out, LRObject lr, String f,
				int st, GrammarSymbol g);

		//
		abstract void end(PrintStream out, LRObject lr, int st,
				SubautomatonTerminal g);

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream stdin, PrintStream stdout,
				PrintStream stderr, String... args)
				throws IOException {
			Set<Map.Entry<GrammarSymbol, LRAction>> s;
			SubautomatonTerminal g, h;
			LRObject lr = getlr(args);
			GrammarSymbol l;
			String f;

			for(int k = 0; k < lr.getTable().getSizeOfStates(); k++) {
				stdout.printf("\t\t\tcase %d:\n", k);
				s = lr.getTable().actions(k).entrySet();
				f = "";  g = null;  l = null;

				// constants
				for(Map.Entry<GrammarSymbol, LRAction> e : s) {
					if(!isput(lr, k, e.getKey())) {
						// do nothing
					} else if(e.getKey() == ContextFreeGrammar.ENDMARKER) {
						l = e.getKey();
					} else {
						h = _put1(lr, this, k, e.getKey(), f, stdout);
						f = h != null ? f : "} else ";
						g = h != null ? h : g;
					}
				}

				// classes
				for(Map.Entry<GrammarSymbol, LRAction> e : s) {
					if(!isput(lr, k, e.getKey())) {
						// do nothing
					} else if(e.getKey() == ContextFreeGrammar.ENDMARKER) {
						l = e.getKey();
					} else {
						h = _put2(lr, this, k, e.getKey(), f, stdout);
						f = h != null ? f : "} else ";
						g = h != null ? h : g;
					}
				}

				if(l != null) {
					stdout.printf("\t\t\t\t%s{\n", f);
					put(stdout, lr, f, k, l);
					stdout.printf("\t\t\t\t}\n");
				} else {
					if(!f.equals(""))  stdout.printf("\t\t\t\t}\n");
					end(stdout, lr, k, g);
				}
			}
			return 0;
		}

	}

	private class PutShift extends Puts {

		@Override
		boolean isput(LRObject lr, int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isShift();
		}

		@Override
		void put(PrintStream out, LRObject lr, String f, int st,
				GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			if(a.isShift()) {
				out.printf("\t\t\t\t\treturn %d;\n",
						a.getNextStateID());
			}
		}

		@Override
		void end(PrintStream out, LRObject lr, int st,
				SubautomatonTerminal g) {
			LRAction a;

			if(g == null) {
				out.printf("\t\t\t\t\treturn -1;\n");
			} else if((a = lr.getTable().action(st, g)).isShift()) {
				out.printf("\t\t\t\t__stkpush(%d, ENGINE_%s);\n",
						a.getNextStateID(), g.getSubautomaton());
//				out.printf("\t\t\t\tSTATE = 0;\n");
//				out.printf("\t\t\t\treturn NINA_ACCEPT;\n");
				out.printf("\t\t\t\t\treturn %d;\n",
						-(a.getNextStateID() + 2));
			} else {
				out.printf("\t\t\t\t\treturn -1;\n");
			}
		}

	}

	private class PutReduce extends Puts {

		@Override
		boolean isput(LRObject lr, int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isReduce();
		}

		@Override
		void put(PrintStream out, LRObject lr, String f, int st,
				GrammarSymbol g) {
			ContextFreeRule r;
			LRAction a;
			Object o;
			String s;
			int l;

			a = lr.getTable().action(st, (Terminal)g);
			if(!a.isReduce())  return;
			r = a.getReduceRule();
			l = r.getDerivedSymbolLength();
			o = lr.getActionMap().get(a.getReduceRule());

			if(o != null) {
				out.printf("\t\t\t\t\t$$ = null;\n", l);
				s = LRActionConverter.translate(r, lr.getDefinition(),
						o.toString(), LRTranslatorJava.this);
				out.printf("\t\t\t\t\t%s\n", s);
			} else if(l > 0) {
				s = LRActionConverter.translate(r, lr.getDefinition(),
						"$$ = $1;", LRTranslatorJava.this);
				out.printf("\t\t\t\t\t%s\n", s);
			} else {
				out.printf("\t\t\t\t\t$$ = null;\n", l);
			}
			out.printf("\t\t\t\t\t_pop(%d);\n", l);
			out.printf("\t\t\t\t\t__k = getGoto(_refsyn(), %d);\n",
					lr.getNonterminalNumber(r.getLeftSymbol()));
			out.printf("\t\t\t\t\t_push(__k, $$);\n");
			out.printf("\t\t\t\t\treturn __k;\n");
		}

		@Override
		void end(PrintStream out, LRObject lr, int st,
				SubautomatonTerminal g) {
			out.printf("\t\t\t\t\treturn -1;\n");
		}

	}

	private class PutAccept extends Puts {

		@Override
		boolean isput(LRObject lr, int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isAccept();
		}

		@Override
		void put(PrintStream out, LRObject lr, String f, int st,
				GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			if(a.isAccept()) {
				out.printf("\t\t\t\t\treturn true;\n");
			}
		}

		@Override
		void end(PrintStream out, LRObject lr, int st,
				SubautomatonTerminal g) {
			out.printf("\t\t\t\t\treturn false;\n");
		}

	}

	private class PutGoto implements ShProcess {

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream stdin, PrintStream out,
				PrintStream stderr, String... args)
				throws IOException {
			Set<Map.Entry<Nonterminal, Integer>> s;
			LRObject lr = getlr(args);
			String f;
			int n;

			for(int k = 0; k < lr.getTable().getSizeOfStates(); k++) {
				out.printf("\t\t\tcase %d:\n", k);
				s = lr.getTable().goTos(k).entrySet();
				f = "";
				for(Map.Entry<Nonterminal, Integer> e : s) {
					n = lr.getNonterminalNumber(e.getKey());
					out.printf("\t\t\t\t%sif(__n == %d) {\n", f, n);
					out.printf("\t\t\t\t\treturn %d;\n", e.getValue());
					f = "} else ";
				}
				if(!f.equals(""))  out.printf("\t\t\t\t}\n");
				out.printf("\t\t\t\tthrow new RuntimeException();\n");
			}
			return 0;
		}

	}

	private class PutGrammarInitial implements ShProcess {

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream in, PrintStream out, PrintStream err,
				String... args) throws IOException {
			LRObject lr = getlr(args);
			int s = lr.getStartNumber();

			out.printf("\t\t\t\tSTATE = %d << LRSTATESHIFT;\n", s);
			out.printf("\t\t\t\t__synarray[0] = %d << LRSTATESHIFT;\n", s);
			return 0;
		}

	}

	private class CatDefinitionPart implements ShProcess {

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream in, PrintStream out, PrintStream err,
				String... args) throws IOException {
			LRObject lr = getlr(args);

			out.println(lr.getDefinition().getDefinition());
			return 0;
		}

	}

	//
	private static final Pattern E_CONST = Pattern.compile(
			"([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*");
	private static final Pattern E_STR = Pattern.compile("\"(.*)\"");

	//
	LRTranslatorJava(Quadro q) {
		super(q);
	}

	//
	private boolean _put10(LRObject lr, GrammarSymbol g, String f,
			PrintStream out) {
		String x;
		int c;

		if(g instanceof CharacterTerminal) {
			c = ((CharacterTerminal)g).getCharacter();
			if(c == '\'' || c == '\\') {
				out.printf("\t\t\t\t%sif(__eq(__t, '\\%c')) {\n",
						f, (char)c);
				return true;
			} else if(c == '\n') {
				out.printf("\t\t\t\t%sif(__eq(__t, '\\n')) {\n", f);
				return true;
			} else if(c == '\r') {
				out.printf("\t\t\t\t%sif(__eq(__t, '\\r')) {\n", f);
				return true;
			} else if(c == '\t') {
				out.printf("\t\t\t\t%sif(__eq(__t, '\\t')) {\n", f);
				return true;
			} else {
				out.printf("\t\t\t\t%sif(__eq(__t, '%c')) {\n",
						f, (char)c);
				return true;
			}
		} else if(g instanceof ConstantTerminal) {
			x = lr.getDefinition().getType(
					((ConstantTerminal)g).getConstant());
			if(E_CONST.matcher(x).matches()) {
				out.printf("\t\t\t\t%sif(__eq(__t, %s)) {\n",
						f, x);
				return true;
			} else if(E_STR.matcher(x).matches()) {
				out.printf("\t\t\t\t%sif(__eq(__t, %s)) {\n",
						f, x);
				return true;
			}
		} else if(g instanceof PrecedenceTerminal) {
			return _put10(lr, ((PrecedenceTerminal)g).getWrapee(), f,
					out);
		}
		return false;
	}

	//
	private SubautomatonTerminal _put1(LRObject lr, Puts p, int st,
			GrammarSymbol g, String f, PrintStream out) {
		if(g instanceof CharacterTerminal) {
			if(_put10(lr, g, f, out))  p.put(out, lr, f, st, g);
		} else if(g instanceof ConstantTerminal) {
			if(_put10(lr, g, f, out))  p.put(out, lr, f, st, g);
		} else if(g instanceof PrecedenceTerminal) {
			if(_put10(lr, g, f, out))  p.put(out, lr, f, st, g);
//		} else if(g instanceof SubautomatonTerminal) {
//			return (SubautomatonTerminal)g;
		} else if(g == ContextFreeGrammar.ENDMARKER) {
			// do nothing
		} else {
			throw new DecultureException(g.toString());
		}
		return null;
	}

	//
	private boolean _put20(LRObject lr, GrammarSymbol g, String f,
			PrintStream out) {
		String x;

		if(g instanceof ConstantTerminal) {
			x = lr.getDefinition().getType(
					((ConstantTerminal)g).getConstant());
			if(E_CONST.matcher(x).matches()) {
				// do nothing
			} else if(E_STR.matcher(x).matches()) {
				// do nothing
			} else {
				out.printf("\t\t\t\t%sif(__t instanceof %s) {\n",
						f, x);
				return true;
			}
		} else if(g instanceof PrecedenceTerminal) {
			return _put20(lr, ((PrecedenceTerminal)g).getWrapee(), f,
					out);
		}
		return false;
	}

	//
	private SubautomatonTerminal _put2(LRObject lr, Puts p, int st,
			GrammarSymbol g, String f, PrintStream out) {
		if(g instanceof CharacterTerminal) {
			if(_put20(lr, g, f, out))  p.put(out, lr, f, st, g);
		} else if(g instanceof ConstantTerminal) {
			if(_put20(lr, g, f, out))  p.put(out, lr, f, st, g);
		} else if(g instanceof PrecedenceTerminal) {
			if(_put20(lr, g, f, out))  p.put(out, lr, f, st, g);
//		} else if(g instanceof SubautomatonTerminal) {
//			return (SubautomatonTerminal)g;
		} else if(g == ContextFreeGrammar.ENDMARKER) {
			// do nothing
		} else {
			throw new DecultureException(g.toString());
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.LRTranslator#getReturn()
	 */
	@Override
	public String getReturn() {
		return "$$";
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.LRTranslator#translateVariable(java.lang.StringBuffer, net.morilib.deculture.parser.DecultureDefinition, java.lang.Object, int)
	 */
	@Override
	public void translateVariable(StringBuffer buf,
			DecultureDefinition def, Object g, int z, boolean bury) {
		Matcher m;
		String s;

		if(g instanceof CharacterTerminal) {
			buf.append(String.format("((Character)_refv(%d))", z));
		} else if(g instanceof ConstantTerminal) {
			s = ((ConstantTerminal)g).getConstant();
			s = def.getType(s);
			s = s != null ? s : "Object";
			if((m = E_STR.matcher(s)).matches()) {
				buf.append(bury ? m.group(1) : s);
			} else {
				buf.append(String.format("((%s)_refv(%d))", s, z));
			}
		} else if(g instanceof PrecedenceTerminal) {
			translateVariable(buf, def,
					((PrecedenceTerminal)g).getWrapee(), z, bury);
		} else if(g == ContextFreeGrammar.ENDMARKER) {
			buf.append(String.format("(_refv(%d))", z));
		} else if(g instanceof ConstantNonterminal) {
			s = ((ConstantNonterminal)g).getConstant();
			s = def.getNonterminalType(s);
			s = s != null ? s : "Object";
			buf.append(String.format("((%s)_refv(%d))", s, z));
		} else {
			throw new DecultureException(g.toString());
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.LRTranslator#setCommands(net.morilib.nina.translate.sh.ShNinatBuiltInCommands)
	 */
	public void setCommands(ShNinatBuiltInCommands c) {
		c.putCommand("put_shift", new PutShift());
		c.putCommand("put_reduce", new PutReduce());
		c.putCommand("put_accept", new PutAccept());
		c.putCommand("put_goto", new PutGoto());
		c.putCommand("put_grammar_initial", new PutGrammarInitial());
		c.putCommand("cat_definition", new CatDefinitionPart());
		c.putCommand("echo_lexer", new EchoLexer());
	}

}
