/*
 * Copyright (c) 2016, Yuichiro MORIGUCHI
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the Yuichiro MORIGUCHI nor the names of its contributors 
 *   may be used to endorse or promote products derived from this software 
 *   without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Yuichiro MORIGUCHI BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
MioVM = {};
(function(M, V) {
	var fstack2op,
		fstack1op,
		ffunc1op,
		ffunc1pred,
		ffunc1num;
	fstack2op = function(method) {
		return function() {
			return function(vmcode, varspace, stack) {
				var op2, op1, res;
				op2 = stack.pop();
				op1 = stack.pop();
				if(op1[method] === undefined) {
					throw 'Type Mismatch';
				}
				res = op1[method](op2);
				stack.push(res);
			};
		};
	};
	fstack1op = function(method) {
		return function() {
			return function(vmcode, varspace, stack) {
				var op1, res;
				op1 = stack.pop();
				if(op1[method] === undefined) {
					throw 'Type Mismatch';
				}
				res = op1[method]();
				stack.push(res);
			};
		};
	};
	ffunc1op = function(method) {
		return function(op) {
			if(arguments.length !== 1) {
				throw 'Illegal Arity';
			}
			return op[method]();
		};
	};
	ffunc1pred = function(method) {
		return function(op) {
			if(arguments.length !== 1) {
				throw 'Illegal Arity';
			}
			return M.real(op[method]() ? 1 : 0);
		};
	};
	V.codes = {};
	V.codes.add = fstack2op('add');
	V.codes.sub = fstack2op('sub');
	V.codes.mul = fstack2op('mul');
	V.codes.div = fstack2op('div');
	V.codes.pow = fstack2op('pow');
	V.codes.uminus = fstack1op('negate');
	V.codes.toString = fstack1op('toString');
	V.codes.lt = fstack2op('lt');
	V.codes.le = fstack2op('le');
	V.codes.gt = fstack2op('gt');
	V.codes.ge = fstack2op('ge');
	V.codes.eq = fstack2op('veq');
	V.codes.ne = fstack2op('vne');
	V.codes.and = fstack2op('logicalAnd');
	V.codes.or = fstack2op('logicalOr');
	V.codes.not = fstack1op('logicalNot');
	V.codes.jmp = function(addr) {
		return function(vmcode, varspace, stack) {
			return addr;
		};
	};
	V.codes.jz = function(addr) {
		return function(vmcode, varspace, stack) {
			var val;
			val = stack.pop();
			return val.isZero() ? addr : undefined;
		};
	};
	V.codes.jnz = function(addr) {
		return function(vmcode, varspace, stack) {
			var val;
			val = stack.pop();
			return val.isZero() ? undefined : addr;
		};
	};
	V.codes.jmpGoto = function(lineno) {
		return function(vmcode, varspace, stack) {
			return vmcode.linenoToAddress(lineno);
		};
	};
	V.codes.noop = function() {
		return function(vmcode, varspace, stack) {
			throw 'Internal Error';
		};
	};
	V.codes.refer = function(varname) {
		return function(vmcode, varspace, stack) {
			var val;
			val = varspace[varname];
			val = val === undefined ? M.real(0) : val;
			stack.push(val);
		};
	};
	V.codes.assign = function(varname) {
		return function(vmcode, varspace, stack) {
			var val;
			val = stack.pop();
			varspace[varname] = val;
		};
	};
	V.codes.konst = function(val) {
		return function(vmcode, varspace, stack) {
			stack.push(val);
		};
	};
	V.codes.print = function(printFunction) {
		return function(vmcode, varspace, stack) {
			var val = stack.pop();
			if(vmcode.printMode === V.PRINTRATIONAL) {
				if(val.isRational()) {
					printFunction(val.toString());
				} else {
					printFunction(val.toNumber());
				}
			} else if(vmcode.printMode === V.PRINTINEXACT) {
				printFunction(val.toNumber());
			} else {
				printFunction(val.toString());
			}
		};
	};
	V.codes.printString = function(printFunction, str) {
		return function(vmcode, varspace, stack) {
			printFunction(str);
		};
	};
	V.codes.input = function(inputFunction, prompt, vnames) {
		return function(vmcode, varspace, stack) {
			var out, res, i, num, e, ok;
			while(true) {
				out = inputFunction(prompt);
				res = out.split(',');
				if(res.length === vnames.length) {
					ok = true;
					for(i = 0; i < res.length; i++) {
						try {
							num = M.real(M.rational(res[i]));
							varspace[vnames[i]] = num;
						} catch(e) {
							ok = false;
						}
					}
					if(ok) {
						return;
					}
				}
			}
		};
	};
	V.codes.func = function(func, arity) {
		return function(vmcode, varspace, stack) {
			var args = [], i;
			for(i = 0; i < arity; i++) {
				args.unshift(stack.pop());
			}
			stack.push(func.apply(null, args));
		};
	};
	V.funcs = {};
	V.funcs.SQR = ffunc1op('sqrt');
	V.funcs.SIN = ffunc1op('sin');
	V.funcs.COS = ffunc1op('cos');
	V.funcs.TAN = ffunc1op('tan');
	V.funcs.ASIN = ffunc1op('asin');
	V.funcs.ACOS = ffunc1op('acos');
	V.funcs.ATN = ffunc1op('atan');
	V.funcs.EXP = ffunc1op('exp');
	V.funcs.LOG = ffunc1op('log');
	V.funcs.LOG10 = ffunc1op('log10');
	V.funcs.INT = ffunc1op('floor');
	V.funcs.FIX = ffunc1op('integer');
	V.funcs.ISEXACT = ffunc1pred('isExact');
	V.funcs.INEXACT = ffunc1op('toInexact');
    V.PRINTEXACT = 0;
    V.PRINTRATIONAL = 1;
    V.PRINTINEXACT = 2;
	V.vmcode = function() {
		var codelist = [],
			codeindex = {},
			that = {};
		that.currentAddress = function() {
			return codelist.length;
		};
		that.addCode = function(code) {
			codelist.push(code ? code : V.codes.noop());
			return codelist.length - 1;
		};
		that.setCode = function(addr, code) {
			codelist[addr] = code;
		};
		that.relateLine = function(lineno, addr) {
			codeindex[lineno] = addr;
		};
		that.linenoToAddress = function(lineno) {
			var addr;
			addr = codeindex[lineno];
			if(addr === undefined) {
				throw 'Undefined Line Number';
			}
			return addr;
		};
		that.getCode = function(addr) {
			return codelist[addr];
		};
		that.printMode = V.PRINTEXACT;
		return that;
	};
	V.vmexecuter = function(vmcode) {
		var varspace = {},
			stack = [],
			that = {};
		that.execute = function() {
			var code, jmp, pc = 0;
			while(pc < vmcode.currentAddress()) {
				code = vmcode.getCode(pc);
				jmp = code(vmcode, varspace, stack);
				pc = jmp === undefined ? pc + 1 : jmp;
			}
		};
		return that;
	};
})(Mio, MioVM);