/*
 * Copyright (c) 2016, Yuichiro MORIGUCHI
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the Yuichiro MORIGUCHI nor the names of its contributors 
 *   may be used to endorse or promote products derived from this software 
 *   without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Yuichiro MORIGUCHI BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
Mio = {};
(function(M) {
	var piMemo = [],
		piFact5 = null,
		piFact239 = null;
	M.isInteger = function(num) {
		return (typeof num === "number" &&
				isFinite(num) &&
				num > -9007199254740992 &&
				num < 9007199254740992 &&
				Math.floor(num) === num);
	};
	M.isNaturalNumber = function(num) {
		return M.isInteger(num) && num >= 0;
	};
	M.isArray = function(obj) {
		return Object.prototype.toString.call(obj) === '[object Array]';
	};
	M.isObject = function(obj) {
		return typeof obj === 'object' && obj !== null && !M.isArray(obj);
	};
	M.isNaN = function(obj) {
		return typeof obj === "number" && obj !== obj;
	};
	M.naturalNumber = function(arg) {
		var numlist = [],
			i,
			code,
			that = {},
			head = true,
			checkDigit;
		checkDigit = function(obj) {
			if(!M.isInteger(obj) || obj < 0 || obj > 9) {
				throw 'Illegal Argument';
			}
		};
		if(typeof arg === 'string') {
			if(arg.length === 0) {
				throw 'Illegal Argument';
			}
			for(i = 0; i < arg.length; i++) {
				code = arg.charCodeAt(i) - 48;
				if(code < 0 || code > 9) {
					throw 'Illegal Argument';
				} else if(!(head && code === 0)) {
					numlist.unshift(code);
					head = false;
				}
			}
		} else if(M.isNaturalNumber(arg)) {
			for(i = arg; i !== 0; i = Math.floor(i / 10)) {
				numlist.push(i % 10);
			}
		} else if(M.isArray(arg)) {
			for(i = arg.length - 1; i >= 0; i--) {
				checkDigit(arg[i]);
				if(!(head && arg[i] === 0)) {
					numlist.unshift(arg[i]);
					head = false;
				}
			}
		} else {
			throw 'Illegal Argument';
		}
		that.thisIsNaturalNumber = function() {
			return 'thisIsNaturalNumber';
		};
		that.digit = function(num) {
			if(!M.isNaturalNumber(num)) {
				throw 'Illegal Argument';
			} else if(num >= numlist.length) {
				return 0;
			} else {
				return numlist[num];
			}
		};
		that.length = function() {
			return numlist.length;
		};
		that.isZero = function() {
			return numlist.length === 0;
		};
		that.isUnit = function() {
			return numlist.length === 1 && numlist[0] === 1;
		};
		that.isEven = function() {
			return numlist.length === 0 || numlist[0] % 2 === 0;
		};
		that.isOdd = function() {
			return numlist.length !== 0 && numlist[0] % 2 === 1;
		};
		that.toNumber = function() {
			var res = 0,
				i;
			for(i = numlist.length - 1; i >= 0; i--) {
				res *= 10;
				res += numlist[i];
			}
			return res;
		};
		that.add = function(nat) {
			var mlen = Math.max(numlist.length, nat.length()),
				i,
				nnum = [],
				carry = 0,
				sum;
			for(i = 0; i < mlen; i++) {
				if(i >= numlist.length) {
					sum = nat.digit(i) + carry;
				} else if(i >= nat.length()) {
					sum = numlist[i] + carry;
				} else {
					sum = numlist[i] + nat.digit(i) + carry;
				}
				nnum.push(sum % 10);
				carry = sum < 10 ? 0 : 1;
			}
			if(carry > 0) {
				nnum.push(carry);
			}
			return M.naturalNumber(nnum);
		};
		that.compareTo = function(nat) {
			var i;
			if(numlist.length < nat.length()) {
				return -1;
			} else if(numlist.length > nat.length()) {
				return 1;
			} else {
				for(i = numlist.length - 1; i >= 0; i--) {
					if(numlist[i] < nat.digit(i)) {
						return -1;
					} else if(numlist[i] > nat.digit(i)) {
						return 1;
					}
				}
				return 0;
			}
		};
		that.sub = function(nat) {
			var mlen = numlist.length,
				i,
				j,
				nnum = [];
			if(that.compareTo(nat) < 0) {
				throw 'Arithmetic Exception';
			}
			for(i = mlen - 1; i >= 0; i--) {
				if(i >= nat.length()) {
					nnum.unshift(numlist[i]);
				} else if(i >= numlist.length) {
					throw 'Internal Error';
				} else if(numlist[i] >= nat.digit(i)) {
					nnum.unshift(numlist[i] - nat.digit(i));
				} else {
					for(j = 0; true; j++) {
						if(j >= nnum.length) {
							throw 'Internal Error';
						} else if(nnum[j] === 0) {
							nnum[j] = 9;
						} else {
							nnum[j]--;
							break;
						}
					}
					nnum.unshift(10 + numlist[i] - nat.digit(i));
				}
			}
			return M.naturalNumber(nnum);
		};
		that.mul1 = function(num) {
			var nnum = [],
				i,
				res,
				carry = 0;
			checkDigit(num);
			if(num === 0) {
				return M.naturalNumber(0);
			} else {
				for(i = 0; i < numlist.length; i++) {
					res = numlist[i] * num + carry;
					nnum.push(res % 10);
					carry = Math.floor(res / 10);
				}
				if(carry > 0) {
					nnum.push(carry);
				}
			}
			return M.naturalNumber(nnum);
		};
		that.shiftLeft = function(num) {
			var nnum = [],
				i;
			if(M.isArray(num)) {
				for(i = 0; i < num.length; i++) {
					checkDigit(num[i]);
					nnum.push(num[i]);
				}
				nnum = nnum.concat(numlist);
			} else if(!M.isNaturalNumber(num)) {
				throw 'Illegal Argument';
			} else if(numlist.length > 0) {
				for(i = 0; i < num; i++) {
					nnum.push(0);
				}
				nnum = nnum.concat(numlist);
			}
			return M.naturalNumber(nnum);
		};
		that.mul = function(nat) {
			var res = M.naturalNumber(0),
				trs,
				op1,
				op2,
				i;
			if(numlist.length === 0 || nat.isZero()) {
				return res;
			}
			if(that.compareTo(nat) < 0) {
				op1 = nat;
				op2 = that;
			} else {
				op1 = that;
				op2 = nat;
			}
			for(i = 0; i < op2.length(); i++) {
				trs = op1.mul1(op2.digit(i)).shiftLeft(i);
				res = res.add(trs);
			}
			return res;
		};
		that.shiftRight = function(num) {
			if(!M.isNaturalNumber(num)) {
				throw 'Illegal Argument';
			} else if(num >= numlist.length) {
				return M.naturalNumber(0);
			} else {
				return M.naturalNumber(numlist.slice(num, numlist.length));
			}
		};
		that.divAndRemainder = function(nat) {
			var divlist = [],
				rmd,
				sft,
				i,
				j;
			if(that.isZero()) {
				return {
					div: M.naturalNumber(0),
					remainder: M.naturalNumber(0)
				};
			} else if(nat.isZero()) {
				throw 'Arithmetic Exception';
			} else if(that.compareTo(nat) < 0) {
				return {
					div: M.naturalNumber(0),
					remainder: that
				};
			}
			rmd = that;
			for(i = numlist.length - nat.length(); i >= 0; i--) {
				sft = rmd.shiftRight(i);
				for(j = 1; true; j++) {
					if(j === 10) {
						j--;
						break;
					} else if(sft.compareTo(nat.mul1(j)) < 0) {
						j--;
						break;
					}
				}
				rmd = rmd.sub(nat.mul1(j).shiftLeft(i));
				divlist.unshift(j);
			}
			return {
				div: M.naturalNumber(divlist),
				remainder: rmd
			};
		};
		that.div = function(nat) {
			return that.divAndRemainder(nat).div;
		};
		that.remainder = function(nat) {
			return that.divAndRemainder(nat).remainder;
		};
		that.sqrtAndRemainder = function() {
			var i,
				j,
				left = null,
				leftt,
				right = null,
				root = null,
				init,
				initroot;
			if(numlist.length === 0) {
				return {
					sqrt: M.naturalNumber(0),
					remainder: M.naturalNumber(0)
				};
			}
			for(i = Math.floor((numlist.length - 1) / 2) * 2; i >= 0; i -= 2) {
				if(right === null) {
					init = numlist[i];
					if(i + 1 < numlist.length) {
						init += numlist[i + 1] * 10;
					}
					initroot = Math.floor(Math.sqrt(init));
					root = M.naturalNumber(initroot);
					right = M.naturalNumber(init - initroot * initroot);
					left = M.naturalNumber(initroot * 2);
				} else {
					right = right.shiftLeft(numlist.slice(i, i + 2));
					for(j = 1; true; j++) {
						if(j === 10) {
							j--;
							break;
						}
						leftt = left.shiftLeft([j]);
						if(right.compareTo(leftt.mul1(j)) < 0) {
							j--;
							break;
						}
					}
					root = root.shiftLeft([j]);
					leftt = left.shiftLeft([j]);
					right = right.sub(leftt.mul1(j));
					left = leftt.add(M.naturalNumber(j));
				}
			}
			return {
				sqrt: root,
				remainder: right
			};
		};
		that.gcm = function(num) {
			var m,
				n,
				nn,
				cmp;
			if(that.isZero() || num.isZero()) {
				return M.naturalNumber(1);
			} else if(that.isUnit() || num.isUnit()) {
				return M.naturalNumber(1);
			} else if((cmp = that.compareTo(num)) > 0) {
				m = that;
				n = num;
			} else if(cmp < 0) {
				m = num;
				n = that;
			} else {
				return that;
			}
			while(!n.isZero()) {
				nn = m.remainder(n);
				m = n;
				n = nn;
			}
			return m;
		};
		that.pow = function(num) {
			var i,
				maxi,
				res = that;
			maxi = num.toNumber();
			if(maxi < 0) {
				throw 'Illegal argument';
			} else if(!M.isInteger(maxi)) {
				throw 'Argument is too large';
			} else if(maxi === 0) {
				return M.naturalNumber(1);
			} else if(that.isZero()) {
				return M.naturalNumber(0);
			} else if(that.isUnit()) {
				return M.naturalNumber(1);
			}
			for(i = 1; i < maxi; i++) {
				res = res.mul(that);
			}
			return res;
		};
		that.toString = function() {
			var res = '',
				i;
			if(numlist.length === 0) {
				return '0';
			}
			for(i = numlist.length - 1; i >= 0; i--) {
				res += String.fromCharCode(numlist[i] + 48);
			}
			return res;
		};
		return that;
	};
	M.integer = function(arg) {
		var signum = 1,
			natnum,
			that = {};
		if(typeof arg === 'string') {
			if(arg.charAt(0) === '-') {
				natnum = M.naturalNumber(arg.substring(1, arg.length));
				signum = natnum.isZero() ? 0 : -1;
			} else {
				natnum = M.naturalNumber(arg);
				signum = natnum.isZero() ? 0 : 1;
			}
		} else if(M.isInteger(arg)) {
			natnum = M.naturalNumber(Math.abs(arg));
			signum = arg > 0 ? 1 : arg < 0 ? -1 : 0;
		} else if(M.isObject(arg)) {
			if(arg.thisIsNaturalNumber && typeof arg.thisIsNaturalNumber === 'function') {
				natnum = arg;
				signum = natnum.isZero() ? 0 : 1;
			} else if(arg.natnum && M.isInteger(arg.signum)) {
				if(arg.signum === 0) {
					natnum = M.naturalNumber(0);
					signum = 0;
				} else {
					natnum = arg.natnum;
					signum = arg.signum > 0 ? 1 : -1;
				}
			} else {
				throw 'Illegal Argument';
			}
		} else {
			throw 'Illegal Argument';
		}
		that.thisIsInteger = function() {
			return 'thisIsInteger';
		};
		that.isZero = function() {
			return signum === 0;
		};
		that.isUnit = function() {
			return signum > 0 && natnum.isUnit();
		};
		that.isEven = function() {
			return signum === 0 || natnum.isEven();
		};
		that.isOdd = function() {
			return signum !== 0 && natnum.isOdd();
		};
		that.signum = function() {
			return signum;
		};
		that.naturalNumber = function() {
			return natnum;
		};
		that.negate = function() {
			return M.integer({
				natnum: natnum,
				signum: -signum
			});
		};
		that.abs = function() {
			return M.integer({
				natnum: natnum,
				signum: natnum.isZero() ? 0 : 1
			});
		};
		that.toNumber = function() {
			return signum * natnum.toNumber();
		};
		that.compareTo = function(num) {
			if(signum < num.signum()) {
				return -1;
			} else if(signum > num.signum()) {
				return 1;
			} else {
				return natnum.compareTo(num.naturalNumber());
			}
		};
		that.add = function(num) {
			var cmp;
			if(signum === 0) {
				return num;
			} else if(num.isZero()) {
				return that;
			} else if(signum * num.signum() > 0) {
				return M.integer({
					natnum: natnum.add(num.naturalNumber()),
					signum: signum
				});
			} else if((cmp = natnum.compareTo(num.naturalNumber())) > 0) {
				return M.integer({
					natnum: natnum.sub(num.naturalNumber()),
					signum: signum
				});
			} else if(cmp < 0) {
				return M.integer({
					natnum: num.naturalNumber().sub(natnum),
					signum: num.signum()
				});
			} else {
				return M.integer(0);
			}
		};
		that.sub = function(num) {
			return that.add(num.negate());
		};
		that.mul = function(num) {
			return M.integer({
				natnum: natnum.mul(num.naturalNumber()),
				signum: signum * num.signum()
			});
		};
		that.divAndRemainder = function(num) {
			var res = natnum.divAndRemainder(num.naturalNumber());
			return {
				div: M.integer({
					natnum: res.div,
					signum: signum * num.signum()
				}),
				remainder: M.integer({
					natnum: res.remainder,
					signum: res.remainder.isZero() ? 0 : signum
				})
			};
		};
		that.div = function(num) {
			return that.divAndRemainder(num).div;
		};
		that.remainder = function(num) {
			return that.divAndRemainder(num).remainder;
		};
		that.sqrtAndRemainder = function() {
			var res;
			if(signum < 0) {
				throw 'Out of Domain';
			} else {
				res = natnum.sqrtAndRemainder();
				return {
					sqrt: M.integer({
						natnum: res.sqrt,
						signum: res.sqrt.isZero() ? 0 : 1
					}),
					remainder: M.integer({
						natnum: res.remainder,
						signum: res.remainder.isZero() ? 0 : 1
					})
				};
			}
		};
		that.pow = function(num) {
			var resn,
				maxi;
			if(num.signum() === 0) {
				return M.integer(1);
			} else if(num.signum() < 0) {
				throw 'Illegal argument';
			} else if(signum === 0) {
				return M.integer(0);
			}
			maxi = num.toNumber();
			resn = natnum.pow(num.naturalNumber());
			return M.integer({
				natnum: resn,
				signum: signum > 0 ? 1 : (maxi % 2 === 0) ? 1 : -1
			});
		};
		that.toString = function() {
			var res = '';
			if(signum < 0) {
				res = '-';
			} else if(signum === 0) {
				return '0';
			}
			return res + natnum.toString();
		};
		return that;
	};
	M.rational = function(argnumer, argdenom) {
		var numer,
			denom,
			signum,
			gcm,
			that = {},
			reres,
			rerat,
			i,
			zeros;

		if(typeof argnumer === 'string' && argdenom === undefined) {
			reres = /^(-?[0-9]+)(\.([0-9]+))?$/.exec(argnumer);
			rerat = /^(-?[0-9]+)\/([0-9]+)$/.exec(argnumer);
			if(!reres && !rerat) {
				throw 'Illegal Argument';
			} else if(rerat) {
				return M.rational(M.integer(rerat[1]), M.integer(rerat[2]));
			} else if(!reres[2]) {
				return M.rational(M.integer(argnumer), M.integer(1));
			} else {
				zeros = '1';
				for(i = 0; i < reres[3].length; i++) {
					zeros += '0';
				}
				return M.rational(M.integer(reres[1] + reres[3]), M.integer(zeros));
			}
		} else if(M.isInteger(argnumer) && M.isInteger(argdenom)) {
			return M.rational(M.integer(argnumer), M.integer(argdenom));
		} else if(argdenom.isZero()) {
			throw 'Illegal Argument';
		} else if(argnumer.isZero()) {
			numer = M.integer(0);
			denom = M.integer(1);
		} else {
			gcm = argnumer.naturalNumber().gcm(argdenom.naturalNumber());
			signum = argnumer.signum() * argdenom.signum();
			numer = M.integer({
				natnum: argnumer.naturalNumber().div(gcm),
				signum: signum
			});
			denom = M.integer({
				natnum: argdenom.naturalNumber().div(gcm),
				signum: 1
			});
		}
		that.thisIsRational = function() {
			return 'thisIsRational';
		};
		that.numerator = function() {
			return numer;
		};
		that.denominator = function() {
			return denom;
		};
		that.isZero = function() {
			return numer.isZero();
		};
		that.isUnit = function() {
			return numer.isUnit() && denom.isUnit();
		};
		that.isInteger = function() {
			return denom.isUnit();
		};
		that.isEven = function() {
			return denom.isUnit() && numer.isEven();
		};
		that.isOdd = function() {
			return denom.isUnit() && numer.isOdd();
		};
		that.signum = function() {
			return numer.signum();
		};
		that.toNumber = function() {
			return numer.toNumber() / denom.toNumber();
		};
		that.abs = function() {
			return M.rational(numer.abs(), denom);
		};
		that.negate = function() {
			return M.rational(numer.negate(), denom);
		};
		that.invert = function() {
			var signum = numer.signum(),
				nnumer;
			if(numer.isZero()) {
				throw 'Arithmetic Exception';
			}
			nnumer = M.integer({
				natnum: denom.naturalNumber(),
				signum: signum
			});
			return M.rational(nnumer, numer.abs());
		};
		that.compareTo = function(num) {
			var nnumer1,
				nnumer2;
			nnumer1 = numer.mul(num.denominator());
			nnumer2 = num.numerator().mul(denom);
			return nnumer1.compareTo(nnumer2);
		};
		that.add = function(num) {
			var nnumer,
				ndenom;
			ndenom = denom.mul(num.denominator());
			nnumer = numer.mul(num.denominator()).add(num.numerator().mul(denom));
			return M.rational(nnumer, ndenom);
		};
		that.sub = function(num) {
			var nnumer,
				ndenom;
			ndenom = denom.mul(num.denominator());
			nnumer = numer.mul(num.denominator()).sub(num.numerator().mul(denom));
			return M.rational(nnumer, ndenom);
		};
		that.mul = function(num) {
			return M.rational(numer.mul(num.numerator()), denom.mul(num.denominator()));
		};
		that.div = function(num) {
			return M.rational(numer.mul(num.denominator()), denom.mul(num.numerator()));
		};
		that.remainder = function(num) {
			var numer1, nnumer2, ndenom, nnumer;
			ndenom = denom.mul(num.denominator());
			numer1 = numer.mul(num.denominator());
			numer2 = denom.mul(num.numerator());
			nnumer = numer1.remainder(numer2);
			return M.rational(nnumer, ndenom);
		};
		that.modulo = function(num) {
			var res;
			res = that.remainder(num);
			if(res.signum() < 0) {
				res = res.add(num.abs());
			}
			return res;
		};
		that.integer = function() {
			return M.rational(numer.div(denom), M.integer(1));
		};
		that.floor = function() {
			var res;
			res = numer.divAndRemainder(denom);
			if(numer.signum() < 0 && !res.remainder.isZero()) {
				res.div = res.div.add(M.integer(-1));
			}
			return M.rational(res.div, M.integer(1));
		};
		that.ceil = function() {
			var res;
			res = numer.divAndRemainder(denom);
			if(numer.signum() > 0 && !res.remainder.isZero()) {
				res.div = res.div.add(M.integer(1));
			}
			return M.rational(res.div, M.integer(1));
		};
		that.sqrt = function() {
			var nnumer,
				ndenom;
			if(numer.signum() < 0) {
				throw 'Out of domain';
			}
			nnumer = numer.sqrtAndRemainder();
			ndenom = denom.sqrtAndRemainder();
			if(nnumer.remainder.isZero() && ndenom.remainder.isZero()) {
				return M.rational(nnumer.sqrt, ndenom.sqrt);
			} else {
				return Math.sqrt(that.toNumber());
			}
		};
		that.pow = function(num) {
			var nthat,
				nnumer,
				ndenom,
				npow,
				sqrtres;
			if(num.isZero()) {
				return M.rational(1, 1);
			} else if(num.isUnit()) {
				return that;
			} else if(num.denominator().compareTo(M.integer(2)) === 0) {
				sqrtres = that.sqrt();
				if(typeof sqrtres === 'number') {
					return Math.pow(sqrtres, num.numerator().toNumber());
				} else {
					return sqrtres.pow(M.rational(num.numerator(), M.integer(1)));
				}
			} else if(!num.isInteger()) {
				throw 'Illegal argument';
			} else if(that.isZero() && num.signum() > 0) {
				return M.rational(0, 1);
			} else if(num.numerator().signum() < 0) {
				nthat = that.invert();
				npow = num.numerator().abs();
			} else {
				nthat = that;
				npow = num.numerator();
			}
			nnumer = nthat.numerator().pow(npow);
			ndenom = nthat.denominator().pow(npow);
			return M.rational(nnumer, ndenom);
		};
		that.twice = function() {
			return that.mul(M.rational(2, 1));
		};
		that.mul3 = function() {
			return that.mul(M.rational(3, 1));
		};
		that.half = function() {
			return that.div(M.rational(2, 1));
		};
		that.div3 = function() {
			return that.div(M.rational(3, 1));
		};
		that.toString = function() {
			if(numer.signum() === 0) {
				return '0';
			} else if(denom.isUnit()) {
				return numer.toString();
			} else {
				return numer.toString() + '/' + denom.toString();
			}
		};
		return that;
	};
	M.piIterator = function() {
		var termno = 0,
			calcnext,
			calcnextfact,
			that = {};
		calcnextfact = function() {
			var res;
			if(piFact5 === null) {
				piFact5 = M.rational(1, 5);
				piFact239 = M.rational(1, 239);
			} else {
				piFact5 = piFact5.mul(M.rational(1, 25));
				piFact239 = piFact239.mul(M.rational(1, 239 * 239));
			}
		};
		calcnext = function() {
			var lastterm,
				coeff5,
				coeff239,
				tmax,
				tmin;
			if(termno === 0) {
				lastterm = M.rational(0, 1);
			} else {
				lastterm = piMemo[termno - 1].min;
			}
			calcnextfact();
			coeff5 = M.rational(4, 2 * (2 * termno) + 1);
			coeff239 = M.rational(1, 2 * (2 * termno) + 1);
			tmax = lastterm.add(coeff5.mul(piFact5).sub(coeff239.mul(piFact239)));
			calcnextfact();
			coeff5 = M.rational(4, 2 * (2 * termno + 1) + 1);
			coeff239 = M.rational(1, 2 * (2 * termno + 1) + 1);
			tmin = tmax.sub(coeff5.mul(piFact5).sub(coeff239.mul(piFact239)));
			piMemo.push({
				max: tmax,
				min: tmin
			});
		};
		that.next = function() {
			if(termno > piMemo.length) {
				throw 'Internal Error';
			} else if(termno === piMemo.length) {
				calcnext();
			}
			return piMemo[termno++];
		};
		return that;
	};
	M.real = function(_re, _exact) {
		var that = {},
			wrapf,
			re,
			exact;

		wrapf = function(result, errormsg) {
			if(typeof result !== 'number') {
				return M.real(result, true);
			} else if(M.isNaN(result)) {
				throw errormsg || 'Out of Domain';
			} else if(isFinite(result)) {
				return M.real(result, false);
			} else {
				throw 'Too large value';
			}
		};
		if(_exact !== undefined) {
			re = _re;
			exact = _exact;
		} else if(M.isInteger(_re)) {
			re = M.exactNumber(M.rational(_re, 1), false, false, false);
			exact = true;
		} else if(typeof _re === 'number') {
			if(M.isNaN(_re) || !isFinite(_re)) {
				throw 'Illegal Argument';
			}
			re = _re;
			exact = false;
		} else if(typeof (_re.thisIsRational) === 'function') {
			re = M.exactNumber(_re, false, false, false);
			exact = true;
		} else if(typeof (_re.thisIsExactNumber) === 'function') {
			re = _re;
			exact = true;
		}
		that.isExact = function() {
			return !!exact;
		};
		that.value = function() {
			return re;
		};
		that.isZero = function() {
			return (exact && re.isZero()) || (!exact && re === 0.0);
		};
		that.isRational = function() {
			return exact && re.isRational();
		};
		that.isRadian = function() {
			return exact && re.isRadian();
		};
		that.isSqrt = function() {
			return exact && re.isSqrt();
		};
		that.toNumber = function() {
			if(exact) {
				return re.toNumber();
			} else {
				return re;
			}
		};
		that.equalTo = function(op) {
			if(exact && op.isExact()) {
				return re.equalTo(op.value());
			} else {
				return that.toNumber() === op.toNumber();
			}
		};
		that.eq = function(op) {
			if(exact && op.isExact()) {
				return re.equalTo(op.value());
			} else if(!exact && !op.isExact()) {
				return that.toNumber() === op.toNumber();
			} else {
				return false;
			}
		};
		that.negate = function() {
			if(exact) {
				return M.real(re.negate(), true);
			} else {
				return M.real(-re, false);
			}
		};
		that.compareTo = function(op) {
			var v1, v2;
			if(exact && op.isExact()) {
				return re.compareTo(op.value());
			} else {
				v1 = that.toNumber();
				v2 = op.toNumber();
				return v1 < v2 ? -1 : v1 > v2 ? 1 : 0;
			}
		};
		that.lt = function(op) {
			return M.real(that.compareTo(op) < 0 ? 1 : 0);
		};
		that.le = function(op) {
			return M.real(that.compareTo(op) <= 0 ? 1 : 0);
		};
		that.gt = function(op) {
			return M.real(that.compareTo(op) > 0 ? 1 : 0);
		};
		that.ge = function(op) {
			return M.real(that.compareTo(op) >= 0 ? 1 : 0);
		};
		that.veq = function(op) {
			return M.real(that.equalTo(op) ? 1 : 0);
		};
		that.vne = function(op) {
			return M.real(!that.equalTo(op) ? 1 : 0);
		};
		that.logicalAnd = function(op) {
			return M.real(!that.isZero() && !op.isZero() ? 1 : 0);
		};
		that.logicalOr = function(op) {
			return M.real(!that.isZero() || !op.isZero() ? 1 : 0);
		};
		that.logicalNot = function(op) {
			return M.real(that.isZero() ? 1 : 0);
		};
		that.exp = function() {
			return wrapf(Math.exp(that.toNumber()));
		};
		that.log = function() {
			return wrapf(Math.log(that.toNumber()));
		};
		that.log10 = function() {
			return wrapf(Math.log(that.toNumber()) / Math.log(10));
		};
		that.toInexact = function() {
			return wrapf(that.toNumber());
		};
		that.integer = function() {
			if(exact) {
				return wrapf(re.integer());
			} else if(that.toNumber() < 0) {
				return wrapf(Math.ceil(that.toNumber()));
			} else {
				return wrapf(Math.floor(that.toNumber()));
			}
		};
		that.floor = function() {
			if(exact) {
				return wrapf(re.floor());
			} else {
				return wrapf(Math.floor(that.toNumber()));
			}
		};
		that.ceil = function() {
			if(exact) {
				return wrapf(re.ceil());
			} else {
				return wrapf(Math.ceil(that.toNumber()));
			}
		};
		that.add = function(op) {
			if(exact && op.isExact()) {
				return wrapf(re.add(op.value()));
			} else {
				return wrapf(that.toNumber() + op.toNumber());
			}
		};
		that.sub = function(op) {
			if(exact && op.isExact()) {
				return wrapf(re.sub(op.value()));
			} else {
				return wrapf(that.toNumber() - op.toNumber());
			}
		};
		that.mul = function(op) {
			if(exact && op.isExact()) {
				return wrapf(re.mul(op.value()));
			} else {
				return wrapf(that.toNumber() * op.toNumber());
			}
		};
		that.div = function(op) {
			var resi;
			if(exact && op.isExact()) {
				return wrapf(re.div(op.value()));
			} else {
				return wrapf(that.toNumber() / op.toNumber(), 'Arithmetic Exception');
			}
		};
		that.pow = function(op) {
			if(exact && op.isExact()) {
				return wrapf(re.pow(op.value()));
			} else {
				return wrapf(Math.pow(that.toNumber(), op.toNumber()));
			}
		};
		that.sqrt = function() {
			if(exact) {
				return wrapf(re.sqrt());
			} else {
				return wrapf(Math.sqrt(re));
			}
		};
		that.sin = function() {
			if(exact) {
				return wrapf(re.sin());
			} else {
				return wrapf(Math.sin(re));
			}
		};
		that.cos = function() {
			if(exact) {
				return wrapf(re.cos());
			} else {
				return wrapf(Math.cos(re));
			}
		};
		that.tan = function() {
			if(exact) {
				return wrapf(re.tan());
			} else {
				return wrapf(Math.tan(re));
			}
		};
		that.asin = function() {
			if(exact) {
				return wrapf(re.asin());
			} else {
				return wrapf(Math.asin(re));
			}
		};
		that.acos = function() {
			if(exact) {
				return wrapf(re.acos());
			} else {
				return wrapf(Math.acos(re));
			}
		};
		that.atan = function() {
			if(exact) {
				return wrapf(re.atan());
			} else {
				return wrapf(Math.atan(re), false);
			}
		};
		that.toString = function() {
			if(exact) {
				return re.toString();
			} else {
				return '' + re;
			}
		};
		return that;
	}
	M.exactNumber = function(_val, _pi, _sqrt2, _sqrt3) {
		var that = {},
			val = _val.isZero() ? M.rational(0, 1) : _val,
			pi = !val.isZero() && _pi,
			sqrt2 = !val.isZero() && _sqrt2,
			sqrt3 = !val.isZero() && _sqrt3,
			rat = !(pi || sqrt2 || sqrt3),
			sqrt = sqrt2 || sqrt3,
			sameOrder;

		if((pi && sqrt2) || (pi && sqrt3) || (sqrt2 && sqrt3)) {
			throw 'Internal Error';
		}
		sameOrder = function(op) {
			var num1, num2;

			num1 = (pi ? 4 : 0) + (sqrt2 ? 2 : 0) + (sqrt3 ? 1 : 0);
			num2 = (op.isRadian() ? 4 : 0) + (op.isSqrt2() ? 2 : 0) + (op.isSqrt3() ? 1 : 0);
			return num1 === num2;
		};
		that.thisIsExactNumber = function() {
			return 'thisIsExactNumber';
		};
		that.coefficient = function() {
			return val;
		};
		that.isRational = function() {
			return rat;
		};
		that.isRadian = function() {
			return !!pi;
		};
		that.isSqrt = function() {
			return sqrt2 || sqrt3;
		};
		that.isSqrt2 = function() {
			return !!sqrt2;
		};
		that.isSqrt3 = function() {
			return !!sqrt3;
		};
		that.isZero = function() {
			return rat && val.isZero();
		};
		that.isUnit = function() {
			return rat && val.isUnit();
		};
		that.toNumber = function() {
			if(pi) {
				return val.toNumber() * Math.PI;
			} else if(sqrt2) {
				return val.toNumber() * Math.sqrt(2);
			} else if(sqrt3) {
				return val.toNumber() * Math.sqrt(3);
			} else {
				return val.toNumber();
			}
		};
		that.equalTo = function(op) {
			return sameOrder(op) && val.compareTo(op.coefficient()) === 0;
		};
		that.compareTo = function(op) {
			var comparepi,
				normal;
			comparepi = function(normal, sqr) {
				var piseries,
					pirange,
					pmax,
					pmin;
				piseries = M.piIterator();
				while(true) {
					pirange = piseries.next();
					if(sqr) {
						pmax = pirange.max.mul(pirange.max);
						pmin = pirange.min.mul(pirange.min);
					} else {
						pmax = pirange.max;
						pmin = pirange.min;
					}
					if(pmax.compareTo(normal) <= 0) {
						return -1;
					} else if(pmin.compareTo(normal) >= 0) {
						return 1;
					}
				}
			};
			if(val.signum() < op.coefficient().signum()) {
				return -1;
			} else if(val.signum() > op.coefficient().signum()) {
				return 1;
			} else if(val.signum() === 0 && op.coefficient().signum() === 0) {
				return 0;
			} else if(val.signum() < 0 && op.coefficient().signum() < 0) {
				return -that.negate().compareTo(op.negate());
			} else if(sameOrder(op)) {
				return val.compareTo(op.coefficient());
			} else if(rat) {
				if(op.isRadian()) {
					return -op.compareTo(that);
				} else if(op.isSqrt()) {
					return that.mul(that).compareTo(op.mul(op));
				}
			} else if(pi) {
				if(op.isRational()) {
					normal = op.coefficient().div(val).div(M.rational(4, 1));
					return comparepi(normal, false);
				} else if(op.isSqrt()) {
					normal = op.mul(op).coefficient().div(val.mul(val)).div(M.rational(16, 1));
					return comparepi(normal, true);
				}
			} else if(sqrt) {
				if(op.isRadian()) {
					return -op.compareTo(that);
				} else {
					return that.mul(that).compareTo(op.mul(op));
				}
			} else {
				throw 'Internal Error';
			}
		};
		that.integer = function() {
			if(rat) {
				return M.exactNumber(val.integer(), pi, sqrt2, sqrt3);
			} else if(that.toNumber() < 0) {
				return Math.ceil(that.toNumber());
			} else {
				return Math.floor(that.toNumber());
			}
		};
		that.floor = function() {
			if(rat) {
				return M.exactNumber(val.floor(), pi, sqrt2, sqrt3);
			} else {
				return Math.floor(that.toNumber());
			}
		};
		that.ceil = function() {
			if(rat) {
				return M.exactNumber(val.ceil(), pi, sqrt2, sqrt3);
			} else {
				return Math.ceil(that.toNumber());
			}
		};
		that.negate = function() {
			return M.exactNumber(val.negate(), pi, sqrt2, sqrt3);
		};
		that.invert = function() {
			if(that.isZero()) {
				throw 'Arithmetic Exception';
			} else if(rat) {
				return M.exactNumber(val.invert(), pi, sqrt2, sqrt3);
			} else if(pi) {
				return 1 / that.toNumber();
			} else if(sqrt2) {
				return M.exactNumber(val.invert().half(), pi, sqrt2, sqrt3);
			} else if(sqrt3) {
				return M.exactNumber(val.invert().div3(), pi, sqrt2, sqrt3);
			}
		};
		that.add = function(op) {
			if(that.isZero()) {
				return op;
			} else if(op.isZero()) {
				return that;
			} else if(sameOrder(op)) {
				return M.exactNumber(val.add(op.coefficient()), pi, sqrt2, sqrt3);
			} else {
				return that.toNumber() + op.toNumber();
			}
		};
		that.sub = function(op) {
			if(that.isZero()) {
				return op.negate();
			} else if(op.isZero()) {
				return that;
			} else if(sameOrder(op)) {
				return M.exactNumber(val.sub(op.coefficient()), pi, sqrt2, sqrt3);
			} else {
				return that.toNumber() - op.toNumber();
			}
		};
		that.mul = function(op) {
			if(that.isZero() || op.isZero()) {
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(that.isUnit()) {
				return op;
			} else if(op.isUnit()) {
				return that;
			} else if(rat) {
				return M.exactNumber(val.mul(op.coefficient()), op.isRadian(), op.isSqrt2(), op.isSqrt3());
			} else if(op.isRational()) {
				return M.exactNumber(val.mul(op.coefficient()), pi, sqrt2, sqrt3);
			} else if(sqrt2 && op.isSqrt2()) {
				return M.exactNumber(val.mul(op.coefficient()).twice(), false, false, false);
			} else if(sqrt3 && op.isSqrt3()) {
				return M.exactNumber(val.mul(op.coefficient()).mul3(), false, false, false);
			} else {
				return that.toNumber() * op.toNumber();
			}
		};
		that.div = function(op) {
			if(that.isZero()) {
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(op.isZero()) {
				throw 'Arithmetic Exception';
			} else if(that.isUnit()) {
				return op.invert();
			} else if(op.isUnit()) {
				return that;
			} else if(op.isRational()) {
				return M.exactNumber(val.div(op.coefficient()), pi, sqrt2, sqrt3);
			} else if(sameOrder(op)) {
				return M.exactNumber(val.div(op.coefficient()), false, false, false);
			} else if(rat && op.isSqrt2()) {
				return M.exactNumber(val.div(op.coefficient()).half(), false, true, false);
			} else if(rat && op.isSqrt3()) {
				return M.exactNumber(val.div(op.coefficient()).div3(), false, false, true);
			} else {
				return that.toNumber() / op.toNumber();
			}
		};
		that.pow = function(op) {
			var coeff, numer, denom, rres, hfloor, odd;

			coeff = op.coefficient();
			if(op.isZero()) {
				return M.exactNumber(M.rational(1, 1), false, false, false);
			} else if(op.isUnit()) {
				return that;
			} else if(that.isZero()) {
				if(coeff.signum() < 0) {
					throw 'Arithmetic Exception';
				}
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(!op.isRational()) {
				return Math.pow(that.toNumber(), op.toNumber());
			} else if(rat) {
				if(op.isRational()) {
					rres = val.pow(coeff);
					if(typeof rres === 'number') {
						return rres;
					} else {
						return M.exactNumber(rres, false, false, false);
					}
				} else {
					return Math.pow(that.toNumber(), op.toNumber());
				}
			} else if(sqrt) {
				if(coeff.isInteger()) {
					hfloor = coeff.half().floor();
					rres = M.rational(sqrt2 ? 2 : 3, 1).pow(hfloor);
					if(coeff.signum() < 0) {
						rres = rres.invert();
					}
					rres = val.pow(hfloor.twice()).mul(rres);
					odd = op.coefficient().isOdd();
					return M.exactNumber(rres, false, sqrt2 && odd, sqrt3 && odd);
				} else {
					return Math.pow(that.toNumber(), op.toNumber());
				}
			} else {
				return Math.pow(that.toNumber(), op.toNumber());
			}
		};
		that.sqrt = function() {
			var result;

			if(!rat) {
				return Math.sqrt(that.toNumber());
			} else if(typeof (result = val.sqrt()) !== 'number') {
				return M.exactNumber(result, false, false, false);
			} else if(typeof (result = val.half().sqrt()) !== 'number') {
				return M.exactNumber(result, false, true, false);
			} else if(typeof (result = val.div3().sqrt()) !== 'number') {
				return M.exactNumber(result, false, false, true);
			} else {
				return Math.sqrt(that.toNumber());
			}
		};
		that.sin = function() {
			var period1, period2, res;

			period1 = val.modulo(M.rational(1, 1));
			period2 = val.modulo(M.rational(2, 1));
			if(!pi && !that.isZero()) {
				return Math.sin(that.toNumber());
			} else if(period1.isZero()) {
				res = M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(period1.compareTo(M.rational(1, 6)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, false, false);
			} else if(period1.compareTo(M.rational(1, 4)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, true, false);
			} else if(period1.compareTo(M.rational(1, 3)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, false, true);
			} else if(period1.compareTo(M.rational(1, 2)) === 0) {
				res = M.exactNumber(M.rational(1, 1), false, false, false);
			} else if(period1.compareTo(M.rational(2, 3)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, false, true);
			} else if(period1.compareTo(M.rational(3, 4)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, true, false);
			} else if(period1.compareTo(M.rational(5, 6)) === 0) {
				res = M.exactNumber(M.rational(1, 2), false, false, false);
			} else {
				return Math.sin(that.toNumber());
			}
			if(period2.compareTo(M.rational(1, 1)) > 0) {
				res = res.negate();
			}
			return res;
		};
		that.cos = function() {
			var sft;
			sft = that.add(M.exactNumber(M.rational(1, 2), true, false, false));
			if(typeof sft === 'number') {
				return Math.cos(that.toNumber());
			} else {
				return sft.sin();
			}
		};
		that.tan = function() {
			var rsin, rcos, rtan;
			rsin = that.sin();
			rcos = that.cos();
			if(typeof rsin !== 'number' && typeof rcos !== 'number') {
				return rsin.div(rcos);
			} else {
				rtan = Math.tan(that.toNumber());
				if(M.isNaN(rtan)) {
					throw 'Out of Domain';
				} else {
					return rtan;
				}
			}
		};
		that.asin = function() {
			if(that.compareTo(M.exactNumber.ONE) > 0 || that.compareTo(M.exactNumber.MINUS_ONE) < 0) {
				throw 'Out of Domain';
			} else if(val.isUnit() && rat) {
				return M.exactNumber(M.rational(1, 2), true, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(1, 3), true, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && sqrt2) {
				return M.exactNumber(M.rational(1, 4), true, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && rat) {
				return M.exactNumber(M.rational(1, 6), true, false, false);
			} else if(val.isZero() && rat) {
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && rat) {
				return M.exactNumber(M.rational(-1, 6), true, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && sqrt2) {
				return M.exactNumber(M.rational(-1, 4), true, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(-1, 3), true, false, false);
			} else if(val.negate().isUnit() && rat) {
				return M.exactNumber(M.rational(-1, 2), true, false, false);
			} else {
				return Math.asin(that.toNumber());
			}
		};
		that.acos = function() {
			if(that.compareTo(M.exactNumber.ONE) > 0 || that.compareTo(M.exactNumber.MINUS_ONE) < 0) {
				throw 'Out of Domain';
			} else if(val.isUnit() && rat) {
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(1, 6), true, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && sqrt2) {
				return M.exactNumber(M.rational(1, 4), true, false, false);
			} else if(val.compareTo(M.rational(1, 2)) === 0 && rat) {
				return M.exactNumber(M.rational(1, 3), true, false, false);
			} else if(val.isZero() && rat) {
				return M.exactNumber(M.rational(1, 2), true, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && rat) {
				return M.exactNumber(M.rational(2, 3), true, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && sqrt2) {
				return M.exactNumber(M.rational(3, 4), true, false, false);
			} else if(val.compareTo(M.rational(-1, 2)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(5, 6), true, false, false);
			} else if(val.negate().isUnit() && rat) {
				return M.exactNumber(M.rational(1, 1), true, false, false);
			} else {
				return Math.acos(that.toNumber());
			}
		};
		that.atan = function() {
			if(val.isUnit() && sqrt3) {
				return M.exactNumber(M.rational(1, 3), true, false, false);
			} else if(val.isUnit() && rat) {
				return M.exactNumber(M.rational(1, 4), true, false, false);
			} else if(val.compareTo(M.rational(1, 3)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(1, 6), true, false, false);
			} else if(val.isZero() && rat) {
				return M.exactNumber(M.rational(0, 1), false, false, false);
			} else if(val.compareTo(M.rational(-1, 3)) === 0 && sqrt3) {
				return M.exactNumber(M.rational(-1, 6), true, false, false);
			} else if(val.negate().isUnit() && rat) {
				return M.exactNumber(M.rational(-1, 4), true, false, false);
			} else if(val.negate().isUnit() && sqrt3) {
				return M.exactNumber(M.rational(-1, 3), true, false, false);
			} else {
				return Math.atan(that.toNumber());
			}
		};
		that.toString = function() {
			if(rat) {
				return val.toString();
			} else if(pi) {
				return '(' + val.toString() + ')*PI';
			} else if(sqrt2) {
				return '(' + val.toString() + ')*SQR(2)';
			} else if(sqrt3) {
				return '(' + val.toString() + ')*SQR(3)';
			}
		};
		return that;
	};
	M.exactNumber.ZERO = M.exactNumber(M.rational(0, 1), false, false, false);
	M.exactNumber.ONE = M.exactNumber(M.rational(1, 1), false, false, false);
	M.exactNumber.MINUS_ONE = M.exactNumber(M.rational(-1, 1), false, false, false);
	M.exactNumber.TWO = M.exactNumber(M.rational(2, 1), false, false, false);
})(Mio);
