/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.range.Interval;
import net.morilib.range.Range;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/25
 */
public class NFAEpsilonLink<T, B> implements NFA<T, NFAState, B> {

	private NFA<T, NFAState, B> nfa;
	private NFAState end;

	/**
	 * 
	 * @param nfa
	 * @param e
	 */
	public NFAEpsilonLink(NFA<T, NFAState, B> nfa, NFAState e) {
		this.nfa = nfa;
		this.end = e;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isState(NFAState o) {
		return nfa.isState(o) || end.equals(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.lang.Object)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state, T alphabet) {
		return nfa.getStates(state, alphabet);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, net.morilib.range.Range)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		return nfa.getStates(state, rng);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state,
			EnumSet<TextBound> bound) {
		return nfa.getStates(state, bound);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesEpsilon(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		if(nfa.isAccepted(state)) {
			return Collections.singleton(end);
		} else {
			return nfa.getStatesEpsilon(state);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesBound(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		return nfa.getStatesBound(state, bound);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getInitialStates()
	 */
	@Override
	public Set<NFAState> getInitialStates() {
		return nfa.getInitialStates();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isInitialState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isInitialState(NFAState o) {
		return nfa.isInitialState(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinal(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isFinal(NFAState state) {
		return end.equals(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinalAny(java.util.Set)
	 */
	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		return states.contains(end);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getEdges(net.morilib.automata.NFAState)
	 */
	@Override
	public NFAEdges<T> getEdges(NFAState state) {
		return nfa.getEdges(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		return nfa.nextAlphabets(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		return nfa.nextAlphabets(states);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<T> nextDiscreteAlphabets(NFAState state) {
		return nfa.nextDiscreteAlphabets(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<T> nextDiscreteAlphabets(Set<NFAState> states) {
		return nfa.nextDiscreteAlphabets(states);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAcceptedStates()
	 */
	@Override
	public Set<NFAState> getAcceptedStates() {
		return Collections.singleton(end);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTag(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTag(NFAState state) {
		return nfa.getMatchTag(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTagEnd(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTagEnd(NFAState state) {
		return nfa.getMatchTagEnd(state);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAccept(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<NFAState> getAccept(NFAState state) {
		return Collections.singleton(end);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isAccepted(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isAccepted(NFAState state) {
		return end.equals(state);
	}

}
