/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.misc;

import java.io.Closeable;
import java.io.IOException;

/**
 * 行番号を持ち、1文字読み戻し可能なストリームのインターフェースです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public interface PushbackLineNumberReadable extends Closeable {

	/**
	 * 1文字読み戻します。
	 * 
	 * @param c 読み戻す文字
	 * @throws IOException
	 */
	public void unread(int c) throws IOException;

	/**
	 * 行番号を取得します。
	 */
	public int getLineNumber();

	/**
	 * 配列へ文字を読み込みます。
	 * 
	 * @param cbuf 読み込む配列
	 * @param off  読み込みを始める配列のオフセット
	 * @param len  読み込む文字数
	 * @return     実際に読み込まれた文字数、-1でEOF
	 * @throws IOException
	 */
	public int read(char[] cbuf, int off, int len) throws IOException;

	/**
	 * 配列へ文字を読み込みます。
	 * 
	 * @param cbuf 読み込む配列
	 * @return     実際に読み込まれた文字数、-1でEOF
	 * @throws IOException
	 */
	public int read(char[] cbuf) throws IOException;

	/**
	 * 1文字読み込みます。
	 * 
	 * @return 読み込まれた文字、-1のときはEOF
	 * @throws IOException
	 */
	public int read() throws IOException;

	/**
	 * プロンプトをリセットします。
	 * 
	 * @throws IOException 
	 */
	public void resetPrompt() throws IOException;

	/**
	 * 次が改行のときtrueを得ます。
	 * 
	 * @throws IOException 
	 */
	public boolean isNewline() throws IOException;

}
