/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sed;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.Charset;
import java.util.List;

import net.morilib.sh.ShFileSystem;

/**
 * sedのFacadeです。
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class Sed {

	/**
	 * sedコマンドを実行します。
	 * 
	 * @param fs      ファイルシステム
	 * @param b       コマンド集
	 * @param command コマンド
	 * @param reader  入力
	 * @param writer  出力
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 戻り値
	 * @throws IOException
	 */
	public static int execute(ShFileSystem fs,
			SedCommandBundle b,
			Reader command,
			Reader reader,
			PrintStream writer,
			SedPatternEngine engine,
			boolean display) throws IOException {
		List<SedCommandLine> cmds;

		cmds = SedParser.parse(b, command);
		return SedEngine.execute(fs, reader, writer, engine, display,
				cmds.toArray(new SedCommandLine[0]));
	}

	/**
	 * sedコマンドを実行します。
	 * 
	 * @param fs      ファイルシステム
	 * @param b       コマンド集
	 * @param command コマンド
	 * @param reader  入力
	 * @param writer  出力
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 戻り値
	 * @throws IOException
	 */
	public static int execute(ShFileSystem fs,
			SedCommandBundle b, String command,
			Reader reader, PrintStream writer, SedPatternEngine engine,
			boolean display) throws IOException {
		return execute(fs, b, new StringReader(command), reader,
				writer, engine, display);
	}

	/**
	 * デフォルトのコマンド集を用いてsedコマンドを実行します。
	 * 
	 * @param fs      ファイルシステム
	 * @param command コマンド
	 * @param reader  入力
	 * @param writer  出力
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 戻り値
	 * @throws IOException
	 */
	public static int execute(ShFileSystem fs,
			Reader command, Reader reader, PrintStream writer,
			SedPatternEngine engine,
			boolean display) throws IOException {
		return execute(fs, SedCommandBundle.newInstance(), command,
				reader, writer, engine, display);
	}

	/**
	 * デフォルトのコマンド集を用いてsedコマンドを実行します。
	 * 
	 * @param fs      ファイルシステム
	 * @param command コマンド
	 * @param reader  入力
	 * @param writer  出力
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 戻り値
	 * @throws IOException
	 */
	public static int execute(ShFileSystem fs,
			String command, Reader reader, PrintStream writer,
			SedPatternEngine engine,
			boolean display) throws IOException {
		return execute(fs, SedCommandBundle.newInstance(),
				new StringReader(command), reader, writer,
				engine, display);
	}

	/**
	 * 与えられた文字列についてsedコマンドを実行します。
	 * 
	 * @param fs      ファイルシステム
	 * @param b コマンド集
	 * @param command コマンド
	 * @param input 入力行
	 * @param outputEncoding 出力ファイルのエンコード
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 処理した結果
	 * @throws IOException
	 */
	public static String execute(ShFileSystem fs,
			SedCommandBundle b, String command, String input,
			Charset outputEncoding, SedPatternEngine engine,
			boolean display) throws IOException {
		ByteArrayOutputStream ous = new ByteArrayOutputStream();
		StringReader rd = new StringReader(input);
		PrintStream wr = new PrintStream(ous);

		execute(fs, b, command, rd, wr, engine, display);
		return new String(ous.toByteArray(), outputEncoding);
	}

	/**
	 * 与えられた文字列についてsedコマンドを実行します。
	 * 
	 * @param b コマンド集
	 * @param command コマンド
	 * @param input 入力行
	 * @param outputEncoding 出力ファイルのエンコード
	 * @param engine  正規表現エンジン
	 * @param display パターンスペースを表示するときtrue
	 * @return 処理した結果
	 * @throws IOException
	 */
	public static String execute(ShFileSystem fs,
			SedCommandBundle b, String command, String input,
			String outputEncoding, SedPatternEngine engine,
			boolean display) throws IOException {
		return execute(fs, b, command, input,
				Charset.forName(outputEncoding),
				engine, display);
	}

}
