﻿// Copyright (c) 2008 - 2009 rankingloid
//
// under GNU General Public License Version 2.
//
using System;
using System.Collections.Generic;
using System.IO;
using IJLib;

namespace NicoTools
{
    public class NicoNetworkManager
    {
        private NicoNetwork niconico_network_ = new NicoNetwork();
        private MessageOut msgout_;
        private CancelObject cancel_object_;

        public delegate void StringDelegate(string str);

        private StringDelegate SetDownloadInfo = null;

        public NicoNetworkManager(NicoNetwork network, MessageOut msgout, CancelObject cancel_object)
        {
            niconico_network_ = network;
            msgout_ = msgout;
            cancel_object_ = cancel_object;
        }

        public void SetDelegateSetDonwloadInfo(StringDelegate dlg)
        {
            SetDownloadInfo = dlg;
        }

        public void CheckLogin()
        {
            if (niconico_network_.IsLoginNiconico())
            {
                msgout_.Write("ログインされています。\r\n");
            }
            else
            {
                msgout_.Write("ログインされていません。\r\n");
            }
        }

        public string DownloadRanking(DownloadKind download_kind, string rank_dl_dir)
        {
            msgout_.Write("ランキングのDLを開始します。\r\n");
            DateTime dt = DateTime.Now;
            if (rank_dl_dir[rank_dl_dir.Length - 1] != '\\')
            {
                rank_dl_dir += '\\';
            }

            rank_dl_dir += dt.Year.ToString() + dt.Month.ToString("00") + dt.Day.ToString("00")
                + dt.Hour.ToString("00");

            rank_dl_dir = IJFile.GetNoExistDirName(rank_dl_dir);
            Directory.CreateDirectory(rank_dl_dir);
            niconico_network_.DownloadRanking(rank_dl_dir, download_kind, /*hour, */OnDownloadRankingEvent);
            msgout_.Write("すべてのランキングのDLが完了しました。\r\n");
            return rank_dl_dir;
        }

        // ランキングDL中に呼び出される
        public void OnDownloadRankingEvent(string message, int current, int total)
        {
            msgout_.Write("ランキングをDLしました。" + current + "/" + total + "\r\n");
            if (current < total)
            {
                cancel_object_.CheckCancel();
                cancel_object_.Wait(3000, 5000);
            }
        }

        public void MakeListAndWriteBySearchTag(InputOutputOption iooption, SearchingTagOption option,
            RankingMethod ranking_method)
        {
            RankFile rank_file = MakeListBySearchTag(iooption, option, ranking_method.GetFilter());
            rank_file.Sort(ranking_method);
            iooption.OutputRankFile(rank_file, ranking_method);
            msgout_.Write("リストを作成しました。\r\n");
        }

        public RankFile MakeListBySearchTag(InputOutputOption iooption, SearchingTagOption option, IFilterManager filter)
        {
            RankFile rank_file = new RankFile(iooption.GetRankFileCustomFormat());
            int log_number = 0;

            if (!System.IO.Directory.Exists("log"))
            {
                System.IO.Directory.CreateDirectory("log");
            }

            msgout_.Write("タグ検索を開始します。\r\n");

            List<Video> video_list = new List<Video>();
            for (int i = 0; i < option.searching_tag_list.Count; ++i)
            {
                video_list.AddRange(SearchTag(option.searching_tag_list[i], option, ref log_number));
            }
            if (option.is_detail_getting)
            {
                GetDetail(video_list, option.detail_info_lower, filter, option.getting_detail_interval);
            }
            for (int i = 0; i < video_list.Count; ++i)
            {
                rank_file.Add(video_list[i]);
            }
            return rank_file;
        }

        private List<Video> SearchTag(string tag_word, SearchingTagOption option, ref int log_number)
        {
            List<Video> ret_list = new List<Video>();
            msgout_.Write(tag_word + " の検索を開始します。\r\n");
            double tag_search_interval_lower, tag_search_interval_upper;
            if (option.searching_interval != "")
            {
                IJStringUtil.ParseDlInterval(option.searching_interval, out tag_search_interval_lower, out tag_search_interval_upper);
            }
            else
            {
                tag_search_interval_lower = 10.0;
                tag_search_interval_upper = 12.0;
            }
            int start_page = (option.is_page_all ? 1 : option.page_start);
            int end_page = (option.is_page_all ? int.MaxValue : option.page_end);

            for (int page = start_page; page <= end_page; ++page)
            {
                List<Video> current_list = GetPage(tag_word, page, option, ref log_number, option.is_searching_kind_tag);
                cancel_object_.CheckCancel();
                if (current_list.Count == 0)
                {
                    break;
                }
                for (int i = 0; i < current_list.Count; ++i)
                {
                    if (option.IsEndSearch(current_list[i]))
                    {
                        return ret_list;
                    }
                    if (option.IsConditionSatisfy(current_list[i]))
                    {
                        ret_list.Add(current_list[i]);
                    }
                }
                if (page < end_page)
                {
                    cancel_object_.Wait((int)(tag_search_interval_lower * 1000), (int)(tag_search_interval_upper * 1000));
                }
            }
            return ret_list;
        }

        private List<Video> GetPage(string tag_word, int page, SearchingTagOption option, ref int log_number, bool is_searching_kind_tag)
        {
            int error_times = 0;
            string str = "";
            msgout_.Write(page.ToString() + "ページ目を取得中…\r\n");
            while (true)
            {
                try
                {
                    if (is_searching_kind_tag)
                    {
                        str = niconico_network_.GetSearchTag(tag_word, page, option.GetSortMethod(), option.GetSearchOrder());
                    }
                    else
                    {
                        str = niconico_network_.GetSearchKeyword(tag_word, page, option.GetSortMethod(), option.GetSearchOrder());
                    }
                    if (str.IndexOf("ここから先をご利用いただくにはログインしてください") >= 0)
                    {
                        throw new NiconicoLoginException();
                    }
                }
                catch (NiconicoAccessDeniedException)
                {
                    ++error_times;
                    if (error_times >= 5)
                    {
                        msgout_.Write("ニコニコ動画にアクセスを拒否されました。初めからやり直してください。\r\n");
                        throw;
                    }
                    else
                    {
                        msgout_.Write("ニコニコ動画にアクセスを拒否されました。1分待機した後に再試行します。\r\n");
                        cancel_object_.Wait(60000);
                        continue;
                    }
                }
                catch (System.Net.WebException e)
                {
                    if (e.Status == System.Net.WebExceptionStatus.ProtocolError)
                    {
                        if (((System.Net.HttpWebResponse)e.Response).StatusCode == System.Net.HttpStatusCode.ServiceUnavailable)
                        {
                            ++error_times;
                            if (error_times >= 5)
                            {
                                msgout_.Write("ニコニコ動画へのアクセスエラー(503)が起きました。初めからやり直してください。\r\n");
                                throw;
                            }
                            else
                            {
                                msgout_.Write("ニコニコ動画へのアクセスエラー(503)が起きました。20秒後に再試行します。\r\n");
                                cancel_object_.Wait(20000);
                                continue;
                            }
                        }
                    }
                    throw;
                }
                break;
            }
            msgout_.Write(page.ToString() + "ページ目を取得しました。\r\n");
            IJFile.WriteUTF8("log\\search" + log_number.ToString() + ".html", str);
            cancel_object_.CheckCancel();
            ++log_number;
            return ParseSearch(str);
        }

        private void GetDetail(List<Video> video_list, int detail_info_lower, IFilterManager filter, string interval)
        {
            msgout_.Write("詳細情報を取得中…\r\n");
            double interval_lower, interval_upper;
            if (interval != "")
            {
                IJStringUtil.ParseDlInterval(interval, out interval_lower, out interval_upper);
            }
            else
            {
                interval_lower = 0.3;
                interval_upper = 0.5;
            }
            for (int i = 0; i < video_list.Count; ++i)
            {
                if (video_list[i].point.mylist >= detail_info_lower && filter.IsThrough(video_list[i]))
                {
                    Video video = NicoUtil.GetVideo(niconico_network_, video_list[i].video_id, cancel_object_, msgout_);
                    if (video.IsStatusOK())
                    {
                        int dummy;
                        video_list[i] = video;
                        video_list[i].pname = TagSet.GetPname(video_list[i].tag_set, out dummy);
                    }
                    else
                    {
                        msgout_.Write(video.GetErrorMessage(video_list[i].video_id) + "\r\n");
                    }
                    cancel_object_.Wait((int)(interval_lower * 1000), (int)(interval_upper * 1000));
                }
                cancel_object_.CheckCancel();
            }
            msgout_.Write("詳細情報を取得しました。\r\n");
        }

        public void MylistSearch(string mylist_str, InputOutputOption iooption, RankingMethod ranking_method)
        {
            msgout_.Write("マイリスト取得中…\r\n");
            List<string> mylist_number_list = new List<string>();

            string[] line = IJStringUtil.SplitWithCRLF(mylist_str);

            for (int i = 0; i < line.Length; ++i)
            {
                string str = line[i];
                if ('0' <= str[0] && str[0] <= '9')
                {
                    int end = 1;
                    while (end < str.Length && '0' <= str[end] && str[end] <= '9')
                    {
                        ++end;
                    }
                    mylist_number_list.Add(str.Substring(0, end));
                }
                int cp = 0;
                while (cp < str.Length && (cp = str.IndexOf("mylist/", cp)) >= 0)
                {
                    cp += "mylist/".Length;
                    int end = cp;
                    while (end < str.Length && '0' <= str[end] && str[end] <= '9')
                    {
                        ++end;
                    }
                    mylist_number_list.Add(str.Substring(cp, end - cp));
                    cp = end + 1;
                }
            }
            List<Video> video_list = new List<Video>();
            for (int i = 0; i < mylist_number_list.Count; ++i)
            {
                List<Video> temp_list = ParseMylistHtml(niconico_network_.GetMylistHtml(mylist_number_list[i]));
                msgout_.Write("マイリスト" + mylist_number_list[i] + "を取得しました。\r\n");
                video_list.AddRange(temp_list);
                if (i < mylist_number_list.Count - 1)
                {
                    cancel_object_.Wait(100);
                }
            }
            RankFile rank_file = new RankFile(video_list, iooption.GetRankFileCustomFormat());
            rank_file.Sort(ranking_method);
            iooption.OutputRankFile(rank_file, ranking_method);
            msgout_.Write("マイリスト取得が終了しました。\r\n");
        }

        private static List<Video> ParseMylistHtml(string html)
        {
            List<Video> video_list = new List<Video>();
            int index = html.IndexOf("<table id=\"mylists\"");

            while ((index = html.IndexOf("<p class=\"TXT12\" style=\"margin-top:2px;\"", index)) >= 0)
            {
                Video video = new Video();
                video.length = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                string date_str = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.submit_date = DateTime.ParseExact(date_str, "yyyy年MM月dd日 HH:mm:ss", null);
                string view_str = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                string res_str = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                string mylist_str = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.point.view = IJStringUtil.ToIntFromCommaValue(view_str);
                video.point.res = IJStringUtil.ToIntFromCommaValue(res_str);
                video.point.mylist = IJStringUtil.ToIntFromCommaValue(mylist_str);
                int in2 = html.IndexOf("watch/", index) + "watch/".Length;
                int en2 = html.IndexOf('"', in2);
                video.video_id = html.Substring(in2, en2 - in2);
                video.title = IJStringUtil.GetStringBetweenTag(ref index, "a", html);
                int p_start = html.IndexOf("<p", index);
                int td_end = html.IndexOf("</td>", index);
                if (p_start >= 0 && td_end >= 0 && p_start < td_end)
                {
                    video.description = IJStringUtil.GetStringBetweenTag(ref index, "p", html); // description の欄を仮に使う
                }
                video_list.Add(video);
            }
            return video_list;
        }

        public void GetNewArrival(int start, int end, InputOutputOption iooption, RankingMethod ranking_method)
        {
            msgout_.Write("新着投稿取得を開始します…\r\n");
            List<Video> video_list = new List<Video>();

            if (end < start)
            {
                int temp = end; end = start; start = temp;
            }

            for (int i = start; i <= end; ++i)
            {
                video_list.AddRange(ParseSearch(niconico_network_.GetNewArrival(i)));
                msgout_.Write("新着" + i + "ページ目を取得しました。\r\n");
                if (i < end)
                {
                    cancel_object_.Wait(i * 1000);
                }
            }

            RankFile rank_file = new RankFile(video_list, iooption.GetRankFileCustomFormat());
            rank_file.Sort(ranking_method);
            iooption.OutputRankFile(rank_file, ranking_method);
            msgout_.Write("新着投稿取得が終了しました。\r\n");
        }

        public void UpdateDetailInfo(InputOutputOption iooption, UpdateRankKind update_rank_kind, RankingMethod ranking_method, string interval)
        {
            RankFile rank_file = iooption.GetRankFile();
            RankFile new_rank_file = new RankFile(iooption.GetRankFileCustomFormat());
            msgout_.Write("情報の更新を開始します。\r\n");
            double interval_lower, interval_upper;
            if (interval != "")
            {
                IJStringUtil.ParseDlInterval(interval, out interval_lower, out interval_upper);
            }
            else
            {
                interval_lower = 0.3;
                interval_upper = 0.5;
            }

            for (int i = 0; i < rank_file.Count; ++i)
            {
                int dummy;
                Video rank_file_video = rank_file.GetVideo(i);
                Video video = NicoUtil.GetVideo(niconico_network_, rank_file[i], cancel_object_, msgout_);
                if (video.IsStatusOK())
                {
                    if (update_rank_kind == UpdateRankKind.AddingTag)
                    {
                        rank_file_video.tag_set.Add(video.tag_set);
                        rank_file_video.pname = TagSet.GetPname(video.tag_set, out dummy);
                        new_rank_file.Add(rank_file_video);
                    }
                    else
                    {
                        if (update_rank_kind == UpdateRankKind.ExceptPoint || update_rank_kind == UpdateRankKind.All)
                        {
                            rank_file_video.video_id = video.video_id;
                            rank_file_video.title = video.title;
                        }
                        if (update_rank_kind == UpdateRankKind.All)
                        {
                            rank_file_video.point = video.point;
                        }
                        rank_file_video.submit_date = video.submit_date;
                        rank_file_video.tag_set = video.tag_set;
                        rank_file_video.pname = TagSet.GetPname(video.tag_set, out dummy);
                        new_rank_file.Add(rank_file_video);
                    }
                    msgout_.Write(rank_file_video.video_id + " の情報を更新しました。\r\n");
                }
                else
                {
                    msgout_.Write(video.GetErrorMessage(rank_file_video.video_id) + "\r\n");
                }
                cancel_object_.Wait((int)(interval_lower * 1000), (int)(interval_upper * 1000));
            }
            new_rank_file.Sort(ranking_method);
            iooption.OutputRankFile(new_rank_file, ranking_method);

            msgout_.Write("ファイルに書き込みました。\r\n情報の更新を終了します。\r\n");
        }

        public void GetDetailInfo(List<string> video_id_list, InputOutputOption iooption, bool is_reading_input, RankingMethod ranking_method, string interval)
        {
            msgout_.Write("情報の取得を開始します。\r\n");
            double interval_lower, interval_upper;
            if (interval != "")
            {
                IJStringUtil.ParseDlInterval(interval, out interval_lower, out interval_upper);
            }
            else
            {
                interval_lower = 0.3;
                interval_upper = 0.5;
            }

            RankFile rank_file;
            if (is_reading_input)
            {
                rank_file = iooption.GetRankFile();
            }
            else
            {
                rank_file = new RankFile(iooption.GetRankFileCustomFormat());
            }

            for (int i = 0; i < video_id_list.Count; ++i)
            {
                Video video = NicoUtil.GetVideo(niconico_network_, video_id_list[i], cancel_object_, msgout_);
                if (video.IsStatusOK())
                {
                    int dummy;
                    video.pname = TagSet.GetPname(video.tag_set, out dummy);
                    rank_file.Add(video);
                    msgout_.Write(video_id_list[i] + " の情報を取得しました。\r\n");
                }
                else
                {
                    msgout_.Write(video.GetErrorMessage(video_id_list[i]) + "\r\n");
                }
                cancel_object_.Wait((int)(interval_lower * 1000), (int)(interval_upper * 1000));
            }
            rank_file.Sort(ranking_method);
            iooption.OutputRankFile(rank_file, ranking_method);
            msgout_.Write("ファイルに書き込みました。\r\n情報の取得を終了します。\r\n");
        }

        public void DownloadFlv(InputOutputOption iooption, string dl_interval, string flv_save_dir, bool is_fixing_extension)
        {
            double interval_min, interval_max;
            IJStringUtil.ParseDlInterval(dl_interval, out interval_min, out interval_max);
            DownloadFlv(iooption, interval_min, interval_max, flv_save_dir, is_fixing_extension);
        }

        public void DownloadFlv(InputOutputOption iooption, double interval_min, double interval_max, string flv_save_dir, bool is_fixing_extension)
        {
            bool start_flag = true;
            bool error_flag = false;

            RankFile rank_file = iooption.GetRankFile();
            if (!System.IO.Directory.Exists(flv_save_dir))
            {
                System.IO.Directory.CreateDirectory(flv_save_dir);
            }
            for (int i = 0; i < rank_file.Count; ++i)
            {
                if (!File.Exists(flv_save_dir + rank_file[i] + ".flv") && !File.Exists(flv_save_dir + rank_file[i] + ".mp4") &&
                    !File.Exists(flv_save_dir + rank_file[i] + ".swf"))
                {
                    if (start_flag)
                    {
                        start_flag = false;
                    }
                    else
                    {
                        cancel_object_.CheckCancel();
                        msgout_.Write("次の動画DLまで待機中。\r\n");
                        cancel_object_.Wait((int)(interval_min * 1000), (int)(interval_max * 1000));
                    }
                    int try_times = 5;
                    for (int j = 0; j < try_times; ++j)
                    {
                        msgout_.Write("動画 " + rank_file[i] + " をDLしています…\r\n");
                        try
                        {
                            niconico_network_.DownloadAndSaveFlv(rank_file[i], flv_save_dir + rank_file[i] + ".flv", InformDownloading);
                            if (!is_fixing_extension)
                            {
                                RenameFlv(flv_save_dir + rank_file[i] + ".flv");
                            }
                        }
                        catch (MyCancelException)
                        {
                            throw;
                        }
                        catch (Exception e)
                        {
                            msgout_.Write("動画 " + rank_file[i] + " のDLに失敗しました。\r\n");
                            msgout_.Write(e.Message + "\r\n");
                            if (j < try_times - 1)
                            {
                                cancel_object_.CheckCancel();
                                msgout_.Write("再試行します。\r\n");
                                cancel_object_.Wait(5000);
                            }
                            else
                            {
                                error_flag = true;
                            }
                            continue;
                        }
                        msgout_.Write("動画 " + rank_file[i] + " のDLが完了しました。\r\n");
                        if (SetDownloadInfo != null)
                        {
                            SetDownloadInfo("完了");
                        }
                        break;
                    }
                }
                else
                {
                    msgout_.Write(rank_file[i] + " は存在するのでDLしません。\r\n");
                }
            }
            if (error_flag)
            {
                msgout_.Write("動画DLを完了しましたが、一部動画をDLできませんでした。\r\n");
            }
            else
            {
                msgout_.Write("すべての動画DLを完了しました。\r\n");
            }
        }

        public void InformDownloading(ref bool is_cancel, long current_size, long file_size)
        {
            if (is_cancel)
            {
                if (SetDownloadInfo != null)
                {
                    SetDownloadInfo("");
                }
                cancel_object_.CheckCancel();
            }
            else
            {
                is_cancel = cancel_object_.IsCanceling();
                if (SetDownloadInfo != null)
                {
                    SetDownloadInfo((current_size / 1024).ToString() + "KB / " + (file_size / 1024).ToString() + "KB");
                }
            }
        }

        public void RenameFlvInDirectory(string dir_name)
        {
            string[] files = Directory.GetFiles(dir_name);
            RenameFlvAll(new List<string>(files), true);
        }

        private void RenameFlvAll(List<string> filename_list, bool is_show_message)
        {
            if (is_show_message)
            {
                msgout_.Write("ファイル名を変更しています。\r\n");
            }
            for (int i = 0; i < filename_list.Count; ++i)
            {
                RenameFlv(filename_list[i]);
            }
            if (is_show_message)
            {
                msgout_.Write("ファイル名を変更しました。\r\n");
            }
        }

        private void RenameFlv(string filename)
        {
            string extension = "";

            switch (NicoUtil.JudgeFileType(filename))
            {
                case NicoUtil.FileType.Mp4:
                    extension = ".mp4";
                    break;
                case NicoUtil.FileType.Swf:
                    extension = ".swf";
                    break;
            }

            if (extension != "")
            {
                string new_filename = Path.GetDirectoryName(filename) + "\\" +
                    Path.GetFileNameWithoutExtension(filename) + extension;
                if (File.Exists(new_filename))
                {
                    msgout_.Write(Path.GetFileName(new_filename) + " は存在するので名前は変更されませんでした。\r\n");
                }
                else
                {
                    File.Move(filename, new_filename);
                }
            }
        }

        public void DownloadThumbnail(InputOutputOption iooption, string thumbnail_dir)
        {
            const int try_num = 5;
            RankFile rank_file = iooption.GetRankFile();
            List<string> video_list = new List<string>();

            for (int i = rank_file.Count - 1; i >= 0; --i)
            {
                video_list.Add(rank_file[i]);
            }

            if (!System.IO.Directory.Exists(thumbnail_dir))
            {
                System.IO.Directory.CreateDirectory(thumbnail_dir);
            }
            msgout_.Write("サムネイルのダウンロードを開始します。\r\n");
            for (int i = 0; i < try_num; ++i)
            {
                for (int j = video_list.Count - 1; j >= 0; --j)
                {
                    string video_id = video_list[j];
                    if (!System.IO.File.Exists(thumbnail_dir + video_id + ".jpg"))
                    {
                        try
                        {
                            niconico_network_.SaveThumbnailWithVideoId(video_id, thumbnail_dir + video_id + ".jpg");
                            msgout_.Write("サムネイル " + video_id + ".jpg をDLしました。\r\n");
                            video_list.RemoveAt(j);
                        }
                        catch (System.Exception)
                        {
                            msgout_.Write("サムネイル " + video_id + ".jpg のDLに失敗しました。");
                            if (i < try_num - 1)
                            {
                                msgout_.Write("あとでもう一度試行します。\r\n");
                            }
                        }
                        cancel_object_.Wait(1500, 2000);
                    }
                    else
                    {
                        msgout_.Write(video_id + ".jpg は存在します。\r\n");
                        video_list.RemoveAt(j);
                    }
                }
            }
            msgout_.Write("サムネイルのDLを完了しました。\r\n");
        }

        public string DownloadNicoChart(string ranking_dir, DateTime start_date, DateTime end_date)
        {
            msgout_.Write("ランキングのDLを開始します。\r\n");
            start_date = new DateTime(start_date.Year, start_date.Month, start_date.Day);
            end_date = new DateTime(end_date.Year, end_date.Month, end_date.Day);
            DownloadKindNicoChart download_kind = new DownloadKindNicoChart();
            List<string> name_list = new List<string>();
            List<string> filename_list = new List<string>();
            download_kind.GetRankingNameList(ref name_list, ref filename_list);

            if (ranking_dir[ranking_dir.Length - 1] != '\\')
            {
                ranking_dir += '\\';
            }
            ranking_dir += "nicochart\\";

            IJNetwork network = new IJNetwork();

            for (DateTime dt = start_date; dt <= end_date; dt = dt.AddDays(1.0))
            {
                msgout_.Write(dt.ToString("yyyyMMdd") + " のランキングをDLします。\r\n");
                string dir_name = ranking_dir + dt.ToString("yyyyMMdd") + "\\";
                System.IO.Directory.CreateDirectory(dir_name);
                for (int i = 0; i < name_list.Count; ++i)
                {
                    string html = network.GetAndReadFromWebUTF8("http://www.nicochart.jp/ranking/" +
                        dt.ToString("yyyyMMdd") + name_list[i]);
                    IJFile.Write(dir_name + filename_list[i] + ".txt", html);
                    msgout_.Write("ランキングをDLしました。" + (i + 1).ToString() +
                        "/" + name_list.Count.ToString() + "\r\n");
                    cancel_object_.CheckCancel();
                    cancel_object_.Wait(2000);
                }
            }
            msgout_.Write("すべてのランキングのDLが完了しました。\r\n");
            return ranking_dir;
        }

        public static List<string> ParseTag(string html)
        {
            List<string> tag_list = new List<string>();
            int index = html.IndexOf("<meta name=\"keywords");
            if (index >= 0)
            {
                index = html.IndexOf("content", index);
                if (index >= 0)
                {
                    index = html.IndexOf('"', index) + 1;
                    int last = html.IndexOf('"', index);
                    if (index >= 0 && last >= 0)
                    {
                        int end;
                        while (index >= 0 && index < last)
                        {
                            end = html.IndexOf(',', index);
                            if (end < 0 || end >= last)
                            {
                                tag_list.Add(html.Substring(index, last - index));
                                break;
                            }
                            tag_list.Add(html.Substring(index, end - index));
                            index = end + 1;
                        }
                    }
                }
            }
            return tag_list;
        }

        public static List<Video> ParseSearch(string html)
        {
            return ParseSearch(html, -1);
        }

        // 前から start_num - 1 件は捨てる
        // start_num が -1 なら全件登録
        public static List<Video> ParseSearch(string html, int start_num)
        {
            int index = -1;
            List<Video> list = new List<Video>();
            int count = 0;

            while ((index = html.IndexOf("class=\"thumb_frm\"", index + 1)) >= 0)
            {
                Video video = new Video();
                string viewStr = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.point.view = IJStringUtil.ToIntFromCommaValue(viewStr);
                string resStr = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.point.res = IJStringUtil.ToIntFromCommaValue(resStr);
                string mylistStr = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.point.mylist = IJStringUtil.ToIntFromCommaValue(mylistStr);

                int start = html.IndexOf("watch/", index) + 6;
                int end = html.IndexOf('"', start);
                video.video_id = html.Substring(start, end - start);

                video.length = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);

                string dateStr = IJStringUtil.GetStringBetweenTag(ref index, "strong", html);
                video.submit_date = DateTime.ParseExact(dateStr, "yy/MM/dd HH:mm", null);

                video.title = IJStringUtil.UnescapeHtml(IJStringUtil.GetStringBetweenTag(ref index, "span", html));

                ++count;
                if (count >= start_num)
                {
                    list.Add(video);
                }
            }
            return list;
        }

        private class DownloadKindNicoChart : DownloadKind
        {
            public override void GetRankingNameList(ref List<string> name_list, ref List<string> filename_list)
            {
                for (int i = 0; i < kind_name.Length; ++i)
                {
                    for (int m = 1; m <= 3; ++m)
                    {
                        string option = "";
                        if (m >= 2)
                        {
                            option = "page=" + m.ToString();
                        }
                        if (kind_name[i] == "view" || kind_name[i] == "res")
                        {
                            if (option != "")
                            {
                                option += "&";
                            }
                            option += ((kind_name[i] == "view") ? "type=vd" : "type=rd");
                        }
                        if (option != "")
                        {
                            option = "?" + option;
                        }
                        if (name_list != null)
                        {
                            name_list.Add(option);
                        }
                        if (filename_list != null)
                        {
                            filename_list.Add(kind_file[i] + "_" + m.ToString());
                        }
                    }
                }
            }
        }
    }
}
