﻿// Copyright (c) 2008 - 2009 rankingloid
//
// under GNU General Public License Version 2.
//
using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Xml;
using IJLib;
using System.Text.RegularExpressions;

namespace NicoTools
{
    /// <summary>
    /// ニコニコ動画から各種情報を取得するためのクラス。
    /// </summary>
    /// <remarks>
    /// ニコニコ動画から公式ランキングHTML取得、FLV取得などの機能を持つクラス。ログインが必要な操作は事前にログインの必要がある。
    /// </remarks>
    public class NicoNetwork
    {
        /// <summary>
        /// ブラウザからクッキーを自動取得して設定するときにブラウザの種類を指定するための列挙型
        /// </summary>
        public enum CookieKind { None, IE6, IE7, Firefox3, Opera };

        public enum SearchSortMethod { SubmitDate, View, ResNew, Res, Mylist, Time };
        public enum SearchOrder { Asc, Desc };

        public delegate void NetworkWaitDelegate(string message, int current, int total);

        private static Random random_ = new Random(); // for Wait

        private CookieKind cookie_kind_ = CookieKind.IE6; // クッキーを読み込むブラウザ
        private IJNetwork network_ = new IJNetwork(); // ネットワーク通信用
        private bool is_loaded_cookie_ = false; // 通信を一度でもしたことがあるなら true になる
        private int wait_millisecond_; // デフォルトのイベント関数で Sleep するときのミリ秒数を保存
        private bool is_no_cache_ = false; // HTTP通信でキャッシュをしないように HttpWebRequest に強制させるか

        private const string nicovideo_uri_ = "http://www.nicovideo.jp"; // ニコニコ動画URL
        private const string nicovideo_ext_uri_ = "http://ext.nicovideo.jp";

        /// <summary>
        /// HTTP通信でキャッシュをしないように HttpWebRequest に強制させるか
        /// </summary>
        public bool NoCache
        {
            get { return is_no_cache_; }
            set { is_no_cache_ = value; }
        }

        /// <summary>
        /// ニコニコ動画にログインする
        /// </summary>
        /// <param name="username">ユーザー名</param>
        /// <param name="password">パスワード</param>
        /// <returns>ログインに成功したかどうか</returns>
 
        public bool LoginNiconico(string username, string password)
        {
            network_.ClearCookie();

            string str = network_.PostAndReadFromWebUTF8("https://secure.nicovideo.jp/secure/login?site=niconico",
                "mail=" + username + "&password=" + password);
            is_loaded_cookie_ = true;

            return str.IndexOf("ログインエラー") < 0;
        }
        
        /// <summary>
        /// ニコニコ動画にログインされていない場合はログインする
        /// </summary>
        /// <param name="username">ユーザー名</param>
        /// <param name="password">パスワード</param>
        /// <returns>ログインに成功したかどうか。すでにログイン済みの場合は true を返す</returns> 
        public bool CheckAndLoginNiconico(string username, string password)
        {
            if (!IsLoginNiconico())
            {
                return LoginNiconico(username, password);
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// ニコニコ動画にログインされているかをチェックする。
        /// </summary>
        /// <returns>ログインされているかどうか</returns> 
        public bool IsLoginNiconico()
        {
            CheckCookie();

            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_);

            if (str.IndexOf("<p id=\"login_username\">") >= 0)
            {
                is_loaded_cookie_ = true;
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// クッキーを読み込むブラウザの種類を設定する
        /// </summary>
        /// <param name="kind">クッキーを読み込むブラウザの種類</param>
        public void SetCookieKind(CookieKind kind)
        {
            cookie_kind_ = kind;
        }

        /// <summary>
        /// 内部のクッキーを空にする
        /// </summary>
        public void ClearCookie()
        {
            network_.ClearCookie();
            is_loaded_cookie_ = false;
        }

        /// <summary>
        /// 内部のクッキーを空にして、クッキーを読み込むブラウザの種類を設定し、
        /// そのブラウザから実際にクッキーを読み込む
        /// </summary>
        /// <param name="kind">クッキーを読み込むブラウザの種類</param>
        public void ReloadCookie(CookieKind kind)
        {
            ClearCookie();
            SetCookieKind(kind);
            LoadCookies();
        }

        /// <summary>
        /// 現在の user_session を取得
        /// </summary>
        /// <returns></returns>
        public string GetUserSession()
        {
            CookieCollection cookie_collection = network_.GetCookie(nicovideo_uri_);
            if (cookie_collection["user_session"] != null)
            {
                return cookie_collection["user_session"].Value;
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// user_session を設定
        /// </summary>
        /// <param name="user_session">user_session</param>
        public void ResetUserSession(string user_session)
        {
            network_.ClearCookie();
            network_.SetCookie(nicovideo_uri_, "user_session=" + user_session);
            is_loaded_cookie_ = true;
        }

        /// <summary>
        /// ニコニコ動画公式ランキングHTMLまたはRSSをダウンロードする。
        /// </summary>
        /// <param name="saved_dir">ランキングHTMLまたはRSSを保存するディレクトリ</param>
        /// <param name="download_kind">ダウンロードするランキングの種類</param>
        /// <param name="wait_millisecond">1件ごとの待ち時間（ミリ秒）</param>
        public void DownloadRanking(string saved_dir, DownloadKind download_kind, int wait_millisecond)
        {
            wait_millisecond_ = wait_millisecond;
            DownloadRanking(saved_dir, download_kind, OnDefaultWaitEvent);
        }

        /// <summary>
        /// ニコニコ動画公式ランキングHTMLまたはRSSをダウンロードする。
        /// </summary>
        /// <param name="saved_dir">ランキングHTMLまたはRSSを保存するディレクトリ</param>
        /// <param name="download_kind">ダウンロードするランキングの種類</param>
        /// <param name="dlg">1件ダウンロードするごとに呼び出されるイベント関数</param>
        public void DownloadRanking(string saved_dir, DownloadKind download_kind, NetworkWaitDelegate dlg)
        {
            if (!saved_dir.EndsWith("\\") && saved_dir != "")
            {
                saved_dir += "\\";
            }

            if (saved_dir != "")
            {
                Directory.CreateDirectory(saved_dir);
            }

            List<string> name_list = new List<string>();
            List<string> filename_list = new List<string>();

            download_kind.GetRankingNameList(ref name_list, ref filename_list);

            for (int i = 0; i < name_list.Count; ++i)
            {
                DownloadRankingOnePage(name_list[i], saved_dir, filename_list[i], -1, download_kind.IsFormatRss());
                if (dlg != null)
                {
                    dlg(name_list[i], i + 1, name_list.Count);
                }
            }
        }

        /// <summary>
        /// 動画をダウンロードする
        /// </summary>
        /// <param name="video_id">ダウンロードする動画ID</param>
        /// <param name="save_flv_filename">ダウンロードしたファイルを保存するファイル名</param>
        /// <param name="dlg">ダウンロード最中に呼び出されるコールバックメソッド。null でもよい。</param>
        /// <exception cref="System.Exception">erer</exception>
        public void DownloadAndSaveFlv(string video_id, string save_flv_filename, IJNetwork.DownloadingEventDelegate dlg)
        {
            CheckCookie();
            GetVideoPage(video_id); // クッキーのために HTML のページを取得しておく

            VideoInfo video_info = new VideoInfo(GetVideoInfo(video_id));

            if (video_info.url == "")
            {
                throw new NiconicoAccessFailedException();
            }

            network_.SetReferer(nicovideo_uri_ + "/watch/" + video_id);
            network_.SetDownloadingEventDelegate(dlg);

            try
            {
                network_.PostAndSaveToFile(video_info.url, "", save_flv_filename);
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// 動画のHTMLページを取得する
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <returns>取得したHTML</returns>
        public string GetVideoPage(string video_id)
        {
            CheckCookie();
            return network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/watch/" + video_id);
        }

        /// <summary>
        /// getflv API で動画情報を取得
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <returns>getflv API の結果文字列</returns>
        public string GetVideoInfo(string video_id)
        {
            CheckCookie();
            string postfix_and = "";
            string postfix_q = "";
            if (video_id.StartsWith("nm"))
            {
                postfix_and = "&as3=1";
                postfix_q = "?as3=1";
            }
            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/api/getflv/" + video_id + postfix_q);
            if (str.StartsWith("closed=1&done=true"))
            {
                str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/api/getflv?v=" + video_id + postfix_and);
            }
            else if (str.IndexOf("error=invalid_v1") >= 0) // so で始まるIDの動画用
            {
                string thread_id = GetThreadId(video_id);
                if (thread_id != "")
                {
                    return network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/api/getflv?v=" + thread_id + postfix_and);
                }
            }
            return str;
        }

        /// <summary>
        /// 動画IDから thread id を取得
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <returns>thread id（取得できなければ空文字）</returns>
        public string GetThreadId(string video_id)
        {
            string new_uri = network_.GetLocationHeader(nicovideo_uri_ + "/watch/" + video_id);
            int last_separate = new_uri.LastIndexOf('/');
            if (last_separate >= 0)
            {
                return new_uri.Substring(last_separate + 1); // thread id を取得
            }
            else
            {
                return "";
            }
        }

        /// <summary>
        /// getthumbinfo API で動画情報を取得
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <returns>getthumbinfo API の結果文字列</returns>
        public string GetThumbInfo(string video_id)
        {
            CheckCookie();

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                return network_.GetAndReadFromWebUTF8(nicovideo_ext_uri_ + "/api/getthumbinfo/" + video_id);
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// ext.nicovideo.jp/thumb からサムネイルページを取得
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <returns>取得した文字列</returns>
        public string GetExtThumb(string video_id)
        {
            CheckCookie();

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                return network_.GetAndReadFromWebUTF8(nicovideo_ext_uri_ + "/thumb/" + video_id);
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// サムネイルをダウンロードしてファイルに保存
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="filename">ファイル名</param>
        public void SaveThumbnailWithVideoId(string video_id, string filename)
        {
            CheckCookie();
            string url = "";
            if (video_id.StartsWith("so"))
            {
                video_id = GetThreadId(video_id);
                if (video_id == "")
                {
                    throw new FormatException("サムネイルのURLの取得に失敗しました。");
                }
            }
            if ('0' <= video_id[0] && video_id[0] <= '9')
            {
                string html = GetExtThumb(video_id);
                Match m = Regex.Match(html, "<img alt=\"\" src=\"([^\"]*)\" class=\"video_img\"\\>");
                if (m.Success)
                {
                    url = m.Groups[1].Value;
                }
            }
            else
            {
                string xml = GetThumbInfo(video_id);
                Match m = Regex.Match(xml, "<thumbnail_url>([^<]*)</thumbnail_url>");
                if (m.Success)
                {
                    url = m.Groups[1].Value;
                }
            }
            
            if (url != "")
            {
                Thread.Sleep(500);
                SaveThumbnail(url, filename);
            }
            else
            {
                throw new FormatException("サムネイルのURLの取得に失敗しました。");
            }
        }

        /// <summary>
        /// サムネイルをダウンロードしてファイルに保存
        /// </summary>
        /// <param name="video_id">サムネイルURL</param>
        /// <param name="filename">ファイル名</param>
        public void SaveThumbnail(string url, string filename)
        {
            CheckCookie();
            network_.GetAndSaveToFile(url, filename);
        }

        /// <summary>
        /// 指定したワードでキーワード検索を行う
        /// </summary>
        /// <param name="search_word">検索ワード</param>
        /// <param name="page">ページ数（1から始まる）</param>
        /// <param name="sort_method">検索時の並べ方指定</param>
        /// <param name="order">昇順 or 降順</param>
        /// <returns>検索結果のHTML</returns>
        public string GetSearchKeyword(string search_word, int page, SearchSortMethod sort_method, SearchOrder order)
        {
            return GetSearchKeywordOrTag(search_word, page, sort_method, order, false);
        }

        /// <summary>
        /// 指定したワードでタグ検索を行う
        /// </summary>
        /// <param name="search_word">検索ワード</param>
        /// <param name="page">ページ数（1から始まる）</param>
        /// <param name="sort_method">検索時の並べ方指定</param>
        /// <param name="order">昇順 or 降順</param>
        /// <returns>検索結果のHTML</returns>
        public string GetSearchTag(string search_word, int page, SearchSortMethod sort_method, SearchOrder order)
        {
            return GetSearchKeywordOrTag(search_word, page, sort_method, order, true);
        }

        /// <summary>
        /// 新着動画情報を取得する
        /// </summary>
        /// <param name="page">ページ数（1から始まる）</param>
        /// <returns>取得結果のHTML</returns>
        public string GetNewArrival(int page)
        {
            CheckCookie();

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/newarrival" +
                    (page >= 2 ? "?page=" + page : ""));
                CheckDenied(str);
                return str;
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// 指定したマイリストIDのHTMLページを取得する
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <returns>取得結果のHTML</returns>
        public string GetMylistHtml(string mylist_id)
        {
            CheckCookie();

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/mylist/" + mylist_id);
                CheckDenied(str);
                return str;
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// マイページのマイリスト編集画面からマイリストの情報を取得する
        /// </summary>
        /// <returns>マイリスト情報リスト</returns>
        public List<MylistInfo> GetMylistInfoListFromMypage()
        {
            CheckCookie();
            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/mylistgroup_edit");
            CheckDenied(str);

            List<MylistInfo> mylist_info_list = new List<MylistInfo>();

            int index = 0;
            while ((index = str.IndexOf("_row\" valign=\"top\">", index)) >= 0)
            {
                MylistInfo mylist_info = new MylistInfo();
                index = str.IndexOf("mylist/", index) + "mylist/".Length;
                int end = str.IndexOf('"', index);
                mylist_info.mylist_id = str.Substring(index, end - index);
                mylist_info.title = IJStringUtil.GetStringBetweenTag(ref index, "a", str);
                string number_str = IJStringUtil.GetStringBetweenTag(ref index, "strong", str);
                mylist_info.number_of_item = int.Parse(number_str.Remove(number_str.Length - 1));
                string public_str = IJStringUtil.GetStringBetweenTag(ref index, "a", str);
                mylist_info.is_public = (public_str != "非公開");
                mylist_info.description = IJStringUtil.GetStringBetweenTag(ref index, "p", str);
                mylist_info_list.Add(mylist_info);
            }
            return mylist_info_list;
        }

        /// <summary>
        /// マイリストを新規作成する。
        /// </summary>
        /// <param name="title">新規作成するマイリストのタイトル</param>
        /// <returns>サーバからの応答</returns>
        public string MakeNewMylist(string title)
        {
            CheckCookie();
            network_.SetContentType("application/x-www-form-urlencoded; charset=UTF-8");
            network_.AddCustomHeader("x-requested-with: XMLHttpRequest");
            string str;
            try
            {
                str = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/mylistgroup_edit", "action=create&name=" +
                    Uri.EscapeDataString(title) + "&description=&public=0&default_sort=0");
            }
            finally
            {
                network_.Reset();
            }
            return str;
        }

        /// <summary>
        /// 指定した動画をマイリストに加える。csrf_token を取得するため、最初に動画のページを取りに行く。
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="video_id">動画ID</param>
        /// <returns>サーバからの応答</returns>
        public string AddMylist(string mylist_id, string video_id)
        {
            CheckCookie();
            string html = GetVideoPage(video_id);
            string csrf_token = ParseCsrfToken(html);
            if (csrf_token != "")
            {
                Thread.Sleep(500);
                return AddMylist(mylist_id, video_id, csrf_token);
            }
            else
            {
                throw new NiconicoAddingMylistFailedException("マイリストの追加に失敗しました。csrf_token が取得できませんでした。");
            }
        }

        /// <summary>
        /// 指定した動画をマイリストに加える
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="video_id">動画ID</param>
        /// <param name="csrf_token">csrf_token（動画のページのHTMLに書かれている認証キーみたいなもの）</param>
        /// <returns>サーバからの応答</returns>
        public string AddMylist(string mylist_id, string video_id, string csrf_token)
        {
            CheckCookie();
            if (video_id.StartsWith("so"))
            {
                video_id = GetThreadId(video_id);
            }
            string str = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/watch/" + video_id,
                "mylist=add&mylistgroup_name=&csrf_token=" + csrf_token + "&group_id=" + mylist_id + "&mylist_add=%E7%99%BB%E9%8C%B2&ajax=1");
            return str;
        }

        /// <summary>
        /// watch ページから csrf_token を取得する
        /// </summary>
        /// <param name="html">解析する watch ページの html</param>
        /// <returns>csrf_token</returns>
        public string ParseCsrfToken(string html)
        {
            int index = html.IndexOf("<input type=\"hidden\" name=\"csrf_token\"");
            if (index >= 0)
            {
                int right_angle = html.IndexOf('>', index);
                index = html.IndexOf("value=", index);
                if (index >= 0 && index < right_angle)
                {
                    int start = html.IndexOf('"', index) + 1;
                    int end = html.IndexOf('"', start);
                    return html.Substring(start, end - start);
                }
            }
            return "";
        }

        /// <summary>
        /// マイリストの冒頭の説明文を更新
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="title">更新するマイリストのタイトル</param>
        /// <param name="description">更新する説明文</param>
        /// <returns>サーバからの応答</returns>
        public string UpdateMylistHeader(string mylist_id, string title, string description)
        {
            CheckCookie();
            network_.SetContentType("application/x-www-form-urlencoded; charset=UTF-8");
            network_.AddCustomHeader("x-requested-with: XMLHttpRequest");
            network_.SetReferer(nicovideo_uri_ + "/mylistgroup_edit");
            string str;
            try
            {
                str = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/mylistgroup_edit",
                            "action=update&group_id=" + mylist_id + "&name=" + Uri.EscapeDataString(title) +
                            "&description=" + Uri.EscapeDataString(description));
            }
            finally
            {
                network_.Reset();
            }
            return str;
        }

        /// <summary>
        /// マイリストの公開・非公開を変更
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="is_public">公開するかどうか</param>
        /// <returns>サーバからの応答</returns>
        public string ChangeMylistIsPublic(string mylist_id, bool is_public)
        {
            CheckCookie();
            network_.SetContentType("application/x-www-form-urlencoded; charset=UTF-8");
            network_.AddCustomHeader("x-requested-with: XMLHttpRequest");
            network_.SetReferer(nicovideo_uri_ + "/mylistgroup_edit");
            string str;
            try
            {
                str = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/mylistgroup_edit",
                    "action=update&group_id=" + mylist_id + "&public=" + (is_public ? "1" : "0"));
            }
            finally
            {
                network_.Reset();
            }
            return str;
        }

        /// <summary>
        /// マイリストの個々の動画の説明文を更新する。1件だけではなくて、同じマイリスト内の複数の動画に対して説明文を更新できる。
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="video_id_list">動画IDのリスト</param>
        /// <param name="description_list">説明文のリスト。動画IDのリストと対応がとれている必要がある</param>
        /// <param name="wait_millisecond">1件追加した後の待ち時間</param>
        public void UpdateMylistDescription(string mylist_id, List<string> video_id_list,
            List<string> description_list, int wait_millisecond)
        {
            wait_millisecond_ = wait_millisecond;
            UpdateMylistDescription(mylist_id, video_id_list, description_list, OnDefaultWaitEvent);
        }

        /// <summary>
        /// マイリストの個々の動画の説明文を更新する。1件だけではなくて、同じマイリスト内の複数の動画に対して説明文を更新できる。
        /// </summary>
        /// <param name="mylist_id">マイリストID</param>
        /// <param name="video_id_list">動画IDのリスト</param>
        /// <param name="description_list">説明文のリスト。動画IDのリストと対応がとれている必要がある</param>
        /// <param name="wait_millisecond">1件追加した後に呼び出されるイベント関数</param>
        public void UpdateMylistDescription(string mylist_id, List<string> video_id_list,
            List<string> description_list, NetworkWaitDelegate dlg)
        {
            CheckCookie();
            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/mylist_edit/" + mylist_id);
            Dictionary<string, string> id_key_pair = ParseMylistEditAndGetKey(str);

            for (int i = 0; i < video_id_list.Count; ++i)
            {
                string value;
                if (id_key_pair.TryGetValue(video_id_list[i], out value))
                {
                    network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/mylist_edit/" + mylist_id,
                        "action=update&threads%5B%5D=" + value + "&description=" +
                        Uri.EscapeDataString(description_list[i]));
                    if (dlg != null)
                    {
                        dlg(video_id_list[i], i + 1, video_id_list.Count);
                    }
                }
            }
        }

        public void DeleteMylist(string mylist_id, List<string> video_id_list)
        {
            EditMylist("delete", mylist_id, "", video_id_list);
        }

        public void MoveMylist(string mylist_id, string dest_mylist_id, List<string> video_id_list)
        {
            EditMylist("move", mylist_id, dest_mylist_id, video_id_list);
        }

        public void CopyMylist(string mylist_id, string dest_mylist_id, List<string> video_id_list)
        {
            EditMylist("copy", mylist_id, dest_mylist_id, video_id_list);
        }

        public void EditMylist(string command, string mylist_id, string dest_mylist_id, List<string> video_id_list)
        {
            CheckCookie();
            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/mylist_edit/" + mylist_id);
            Dictionary<string, string> id_key_pair = ParseMylistEditAndGetKey(str);

            StringBuilder argument = new StringBuilder();
            argument.Append("action=").Append(command);
            if (dest_mylist_id != "")
            {
                argument.Append("&dest=").Append(dest_mylist_id);
            }
            foreach (string video_id in video_id_list)
            {
                
                string value;
                if (id_key_pair.TryGetValue(video_id, out value))
                {
                    argument.Append("&threads%5B%5D=").Append(value);
                }
            }
            network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/mylist_edit/" + mylist_id,
                argument.ToString());
        }

        /// <summary>
        /// 指定した動画にタグをつける
        /// </summary>
        /// <param name="tag_list">タグのリスト</param>
        /// <param name="is_lock_list">タグをロックするかどうかを表す論理値のリスト。タグのリストと対応が取れている必要がある</param>
        /// <param name="video_id">タグをつける動画ID</param>
        /// <param name="dlg">1件タグを追加した、またはロックした後に呼び出されるイベント関数</param>
        public void AddTag(List<string> tag_list, List<bool> is_lock_list, string video_id, NetworkWaitDelegate dlg)
        {
            CheckCookie();
            network_.AddCustomHeader("X-Requested-With: XMLHttpRequest");
            network_.AddCustomHeader("X-Prototype-Version: 1.5.1.1");
            network_.SetReferer(nicovideo_uri_ + "/watch/" + video_id);
            string s = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id, "");
            Thread.Sleep(2000);
            try
            {
                StringBuilder tag_str = new StringBuilder();
                for (int i = 0; i < tag_list.Count; ++i)
                {
                    tag_str.Append(tag_list[i]);
                    if (i < tag_list.Count - 1)
                    {
                        tag_str.Append(" ");
                    }
                }
                string html = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                        "cmd=add&tag=" + Uri.EscapeUriString(tag_str.ToString()) + "&tag_add=%E8%BF%BD%E5%8A%A0");
                if (dlg != null)
                {
                    dlg("タグを加えました。", 0, 0);
                }

                for (int i = 0; i < tag_list.Count && i < is_lock_list.Count; ++i)
                {
                    if (is_lock_list[i])
                    {
                        string id = GetTagId(html, tag_list[i]);
                        if (id != "")
                        {
                            network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                                "cmd=lock&tag=" + Uri.EscapeUriString(tag_list[i]) + "&id=" + id +
                                "&owner_lock=1&tag_lock=%E3%83%AD%E3%83%83%E3%82%AF%E3%81%99%E3%82%8B%E2%98%85");
                            if (dlg != null)
                            {
                                dlg(tag_list[i] + " をロックしました。", 0, 0);
                            }
                        }
                    }
                }
            }
            finally
            {
                network_.Reset();
            }
        }

        public void UnlockTag(string tag_name, string video_id)
        {
            CheckCookie();
            network_.AddCustomHeader("X-Requested-With: XMLHttpRequest");
            network_.AddCustomHeader("X-Prototype-Version: 1.5.1.1");
            network_.SetReferer(nicovideo_uri_ + "/watch/" + video_id);
            try
            {
                string html = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                            "");
                string id = GetTagId(html, tag_name);
                if (id != "")
                {
                    network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                        "cmd=lock&tag=" + Uri.EscapeUriString(tag_name) + "&id=" + id +
                        "&owner_lock=0&tag_unlock=%E3%83%AD%E3%83%83%E3%82%AF%E8%A7%A3%E9%99%A4");
                }
            }
            finally
            {
                network_.Reset();
            }
        }

        public void RemoveTag(string tag_name, string video_id)
        {
            CheckCookie();
            network_.AddCustomHeader("X-Requested-With: XMLHttpRequest");
            network_.AddCustomHeader("X-Prototype-Version: 1.5.1.1");
            network_.SetReferer(nicovideo_uri_ + "/watch/" + video_id);
            try
            {
                string html = network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                            "");
                string id = GetTagId(html, tag_name);
                if (id != "")
                {
                    network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id,
                        "cmd=remove&tag=" + Uri.EscapeUriString(tag_name) + "&id=" + id +
                        "&tag_remove=%E5%89%8A%E9%99%A4");
                }
            }
            finally
            {
                network_.Reset();
            }
        }

        public string GetTagEditHtml(string video_id)
        {
            CheckCookie();
            network_.AddCustomHeader("X-Requested-With: XMLHttpRequest");
            network_.AddCustomHeader("X-Prototype-Version: 1.5.1.1");
            network_.SetReferer(nicovideo_uri_ + "/watch/" + video_id);
            try
            {
                return network_.PostAndReadFromWebUTF8(nicovideo_uri_ + "/tag_edit/" + video_id, "");
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// wayback を取得
        /// </summary>
        /// <param name="thread_id">thread_id</param>
        /// <returns>wayback</returns>
        public string GetWayback(string thread_id)
        {
            CheckCookie();
            string way = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/api/getwaybackkey?thread=" + thread_id);
            string[] key = way.Split('=');
            if (key.Length < 2 || key[0] != "waybackkey")
            {
                throw new NiconicoAccessFailedException();
            }
            return key[1];
        }

        /// <summary>
        /// コメントを取得する。内部で GetVideoInfo を呼び出す。連続してこの関数を呼び出す場合は
        /// 引数に VideoInfo を指定するバージョンを使う方がよい（その場合は手動で GetVideoInfo を呼び出す）。
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="res_from">取得件数（最新から100件取得する場合は -100 を指定）</param>
        /// <returns>コメントXML</returns>
        public string GetComment(string video_id, int res_from)
        {
            return GetComment(video_id, new DateTime(), res_from);
        }

        /// <summary>
        /// コメントを取得する。内部で GetVideoInfo と GetWayback を呼び出す。
        /// 日時を指定した取得はプレミアム会員専用。連続してこの関数を呼び出す場合は
        /// 引数に VideoInfo を指定するバージョンを使う方がよい（その場合は手動で GetVideoInfo を呼び出す）。
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="datetime">日時指定</param>
        /// <param name="res_from">取得件数（指定した日時から100件取得する場合は -100 を指定）</param>
        /// <returns>コメントXML</returns>
        public string GetComment(string video_id, DateTime datetime, int res_from)
        {
            VideoInfo video_info = new VideoInfo(GetVideoInfo(video_id));
            Thread.Sleep(500);
            string wayback = "";
            if (datetime.Year != 1)
            {
                wayback = GetWayback(video_info.thread_id);
                Thread.Sleep(500);
            }
            return GetComment(video_id, video_info, wayback, datetime, res_from);
        }

        /// <summary>
        /// コメントを取得する。GetVideoInfo を引数で指定する
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="VideoInfo">GetVideoInfo で取得した VideoInfo オブジェクト</param>
        /// <param name="res_from">取得件数（最新から100件取得する場合は -100 を指定）</param>
        /// <returns>コメントXML</returns>
        public string GetComment(string video_id, VideoInfo video_info, int res_from)
        {
            return GetComment(video_id, video_info.thread_id, "", video_info.ms, new DateTime(), res_from, video_info.user_id);
        }

        /// <summary>
        /// コメントを取得する。GetVideoInfo と wayback を引数で指定する。
        /// 日時を指定した取得はプレミアム会員専用。
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="video_info">GetVideoInfo で取得した VideoInfo オブジェクト</param>
        /// <param name="wayback">wayback（GetWayback で取得した値）</param>
        /// <param name="datetime">日時指定</param>
        /// <param name="res_from">取得件数（指定した日時から100件取得する場合は -100 を指定）</param>
        /// <returns>コメントXML</returns>
        public string GetComment(string video_id, VideoInfo video_info, string wayback, DateTime datetime, int res_from)
        {
            return GetComment(video_id, video_info.thread_id, wayback, video_info.ms, datetime, res_from, video_info.user_id);
        }

        /// <summary>
        /// コメントを取得する。
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="thread_id">thread_id</param>
        /// <param name="wayback">wayback（GetWayback で取得した値）</param>
        /// <param name="message_server">メッセージサーバ</param>
        /// <param name="datetime">日時指定</param>
        /// <param name="res_from">取得件数（指定した日時から100件取得する場合は -100 を指定）</param>
        /// <param name="user_id">ユーザID</param>
        /// <returns></returns>
        public string GetComment(string video_id, string thread_id, string wayback, string message_server, DateTime datetime, int res_from, string user_id)
        {
            CheckCookie();
            string post_str = "<thread ";
            if (datetime.Year != 1) // datetime が空でないなら
            {
                post_str += "user_id=\"" + user_id + "\" "+ 
                    "when=\"" + NicoUserSession.DateToUnix(datetime) + "\" "+ 
                    "waybackkey=\"" + wayback + "\" ";
            }
            post_str += "res_from=\"" +
                res_from.ToString() + "\" version=\"20061206\" thread=\"" +
                thread_id + "\" />";
            return network_.PostAndReadFromWebUTF8(message_server, post_str);
        }

        /// <summary>
        /// コメントを投稿する
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="comment">コメント</param>
        /// <param name="vpos">コメントの時間（位置）。1/100秒単位で指定</param>
        /// <returns>サーバからの応答</returns>
        public string PostComment(string video_id, string comment, int vpos)
        {
            CheckCookie();
            network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/watch/" + video_id); // 必要かどうか不明。一応呼ぶ
            Thread.Sleep(1000);
            VideoInfo video_info = new VideoInfo(GetVideoInfo(video_id));
            Thread.Sleep(1000);
            string comment_xml = GetComment(video_id, video_info, -100);
            Thread.Sleep(1000);
            return PostComment(video_id, video_info, GetTicket(comment_xml), comment, GetLastResNo(comment_xml) / 1000, vpos);
        }

        /// <summary>
        /// コメントを投稿する
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="video_info">VideoInfo オブジェクト</param>
        /// <param name="ticket">ticket</param>
        /// <param name="comment">コメント</param>
        /// <param name="block_no">block_no（最後のコメントNo ÷ 100 らしい）</param>
        /// <param name="vpos">コメントの時間（位置）。1/100秒単位で指定</param>
        /// <returns>サーバからの応答</returns>
        public string PostComment(string video_id, VideoInfo video_info, string ticket, string comment, int block_no, int vpos)
        {
            string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/api/getpostkey?thread=" +
                video_info.thread_id + "&block_no=" + block_no);
            if (str.StartsWith("postkey="))
            {
                Thread.Sleep(1000);
                return PostComment(video_id, video_info, ticket, str.Substring("postkey=".Length), comment, vpos);
            }
            else
            {
                throw new NiconicoAccessFailedException();
            }
        }

        /// <summary>
        /// コメントを投稿する
        /// </summary>
        /// <param name="video_id">動画ID</param>
        /// <param name="video_info">VideoInfo オブジェクト</param>
        /// <param name="ticket">ticket</param>
        /// <param name="postkey">postkey</param>
        /// <param name="comment">コメント</param>
        /// <param name="vpos">コメントの時間（位置）。1/100秒単位で指定</param>
        /// <returns>サーバからの応答</returns>
        public string PostComment(string video_id, VideoInfo video_info, string ticket, string postkey, string comment, int vpos)
        {
            string post_data = "<chat premium=\"";
            post_data += (video_info.is_premium ? "1" : "0");
            post_data += "\" postkey=\"" + postkey + "\" user_id=\"" + video_info.user_id + "\" ticket=\"" +
                ticket + "\" mail=\" 184\" vpos=\"" + vpos.ToString() + "\" thread=\"" +
                video_info.thread_id + "\">" + comment + "</chat>";

            network_.SetReferer(nicovideo_uri_ + "/swf/nicoplayer.swf?ts=" + video_info.thread_id +
                "&is_video_owner=1&wv_id=" + video_id +
                "&deleted=8&open_src=true&movie_type=flv&button_threshold=0&player_version_xml=1218448842");
            byte[] post_data_byte = Encoding.UTF8.GetBytes(post_data);
            try
            {
                return network_.PostAndReadFromWebUTF8(video_info.ms, post_data_byte);
            }
            finally
            {
                network_.SetReferer("");
            }
        }

        //-------------------------------------------- private method -----------------------------------------------

        private void LoadCookies()
        {
            string user_session = "";
            switch (cookie_kind_)
            {
                case CookieKind.IE6:
                    user_session = NicoUserSession.GetUserSessionFromIE6(nicovideo_uri_);
                    break;
                case CookieKind.Firefox3:
                    user_session = NicoUserSession.GetUserSessionFromFilefox3();
                    break;
                case CookieKind.IE7:
                    user_session = NicoUserSession.GetUserSessionFromIE7();
                    break;
            }
            if (user_session != "")
            {
                ResetUserSession(user_session);
            }
            is_loaded_cookie_ = true;
        }

        private void CheckCookie()
        {
            if (!is_loaded_cookie_)
            {
                LoadCookies();
            }
        }

        // 連続アクセスで拒否されていないか調べる
        private void CheckDenied(string html)
        {
            if (html.IndexOf("連続アクセスはご遠慮ください") >= 0)
            {
                throw new NiconicoAccessDeniedException();
            }
        }

        private void OnDefaultWaitEvent(string message, int current, int total)
        {
            if (current < total)
            {
                Thread.Sleep(wait_millisecond_);
            }
        }

        private void DownloadRankingOnePage(string url, string dir_name, string filename, int hour, bool is_xml)
        {
            DateTime current_datetime = DateTime.Now;

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                string html = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/ranking/" + url);
                string save_filename = dir_name + filename + current_datetime.ToString("yyyyMMddHHmm");
                save_filename += (is_xml ? ".xml" : ".html");

                File.WriteAllText(save_filename, html, Encoding.UTF8);
            }
            finally
            {
                network_.Reset();
            }
        }

        private string GetSearchKeywordOrTag(string word, int page, SearchSortMethod sort_method, SearchOrder order, bool is_tag)
        {
            CheckCookie();

            if (is_no_cache_)
            {
                network_.SetMaxAgeZero();
            }
            try
            {
                string str = network_.GetAndReadFromWebUTF8(nicovideo_uri_ + "/" + (is_tag ? "tag" : "search") +
                    "/" + Uri.EscapeDataString(word) + GetOption(page, sort_method, order, is_tag));
                CheckDenied(str);
                return str;
            }
            finally
            {
                network_.Reset();
            }
        }

        /// <summary>
        /// タグ検索、キーワード検索のURLオプションを取得
        /// </summary>
        /// <param name="page">ページ数</param>
        /// <param name="sort_method">検索時の並べ方指定</param>
        /// <param name="order">昇順 or 降順</param>
        /// <param name="is_tag">タグ検索の場合 true、キーワード検索の場合 false</param>
        /// <returns></returns>
        private static string GetOption(int page, SearchSortMethod sort_method, SearchOrder order, bool is_tag)
        {
            string option = "";
            if (page >= 2)
            {
                option += "?page=" + page.ToString();
            }
            switch (sort_method)
            {
                case SearchSortMethod.SubmitDate:
                    if (is_tag) // キーワード検索のときはオプションをつけない
                    {
                        option += ((option != "" ? "&" : "?") + "sort=f");
                    }
                    break;
                case SearchSortMethod.View:
                    option += ((option != "" ? "&" : "?") + "sort=v");
                    break;
                case SearchSortMethod.Res:
                    option += ((option != "" ? "&" : "?") + "sort=r");
                    break;
                case SearchSortMethod.ResNew:
                    if (!is_tag) // タグ検索のときはオプションをつけない
                    {
                        option += ((option != "" ? "&" : "?") + "sort=n");
                    }
                    break;
                case SearchSortMethod.Mylist:
                    option += ((option != "" ? "&" : "?") + "sort=m");
                    break;
                case SearchSortMethod.Time:
                    option += ((option != "" ? "&" : "?") + "sort=l");
                    break;
            }
            if (order == SearchOrder.Asc)
            {
                option += ((option != "" ? "&" : "?") + "order=a");
            }
            return option;
        }

        private static Dictionary<string, string> ParseMylistEditAndGetKey(string html)
        {
            int index = -1;
            Dictionary<string, string> dic = new Dictionary<string, string>();

            while ((index = html.IndexOf("<p class=\"TXT12\" style=\"margin-top:2px", index + 1)) >= 0)
            {
                int start = html.LastIndexOf("value", index);
                start = html.IndexOf('"', start) + 1;
                int end = html.IndexOf('"', start);
                string value = html.Substring(start, end - start);
                start = html.IndexOf("watch/", end) + "watch/".Length;
                end = html.IndexOf('"', start);
                string video_id = html.Substring(start, end - start);
                dic.Add(video_id, value);
            }
            return dic;
        }

        // コメントXMLを解析して ticket を取得
        private static string GetTicket(string xml)
        {
            int start = xml.IndexOf("ticket=\"") + "ticket=\"".Length;
            if (start < 0)
            {
                throw new NiconicoAccessFailedException();
            }
            int end = xml.IndexOf('"', start);
            return xml.Substring(start, end - start);
        }

        // コメントXMLを解析して最後のレス番号を取得
        private static int GetLastResNo(string xml)
        {
            int start = xml.LastIndexOf("no=\"") + "no=\"".Length;
            if (start < 0)
            {
                throw new NiconicoAccessFailedException();
            }
            int end = xml.IndexOf('"', start);
            return int.Parse(xml.Substring(start, end - start));
        }

        // タグHTMLを解析して指定したタグのIDを取得
        private static string GetTagId(string html, string tag_name)
        {
            int start = html.IndexOf("name=\"tag\" value=\"" + tag_name);
            if (start < 0)
            {
                return "";
            }
            start = html.IndexOf("name=\"id\" value=\"", start);
            if (start < 0)
            {
                return "";
            }
            start += "name=\"id\" value=\"".Length;
            int end = html.IndexOf('"', start);
            return html.Substring(start, end - start);
        }
    }

    /// <summary>
    /// ニコニコAPI の getflv で取得できる情報を格納するクラス
    /// </summary>
    public class VideoInfo
    {
        public string thread_id = "";
        public string l = "";
        public string url = "";
        public string link = "";
        public string ms = "";
        public string user_id = "";
        public bool is_premium = false;
        public string nickname = "";
        public string time = "";
        public bool done = false;

        public VideoInfo()
        {
            // 何もしない
        }

        public VideoInfo(string xml)
        {
            Parse(xml);
        }

        public void Parse(string xml)
        {
            string[] option_array = xml.Split('&');
            for (int i = 0; i < option_array.Length; ++i)
            {
                string[] name_value_pair = option_array[i].Split('=');
                Set(Uri.UnescapeDataString(name_value_pair[0]), Uri.UnescapeDataString(name_value_pair[1]));
            }
        }

        public void Set(string name, string value)
        {
            switch (name)
            {
                case "thread_id":
                    thread_id = value;
                    break;
                case "l":
                    l = value;
                    break;
                case "url":
                    url = value;
                    break;
                case "link":
                    link = value;
                    break;
                case "ms":
                    ms = value;
                    break;
                case "user_id":
                    user_id = value;
                    break;
                case "is_premium":
                    is_premium = value.Equals("1");
                    break;
                case "nickname":
                    nickname = value;
                    break;
                case "time":
                    time = value;
                    break;
                case "done":
                    done = value.Equals("true");
                    break;
            }
        }
    }

    /// <summary>
    /// マイリストの情報を表す構造体
    /// </summary>
    public struct MylistInfo
    {
        public string mylist_id;
        public string title;
        public string description;
        public int number_of_item;
        public bool is_public;
    }

    /// <summary>
    /// ニコニコ動画公式ランキングをダウンロードする時に、種類を指定するためのクラス
    /// </summary>
    public class DownloadKind
    {
        public enum FormatKind { Html, Rss };

        protected static string[] kind_name = { "view", "res", "mylist" };
        protected static string[] kind_file = { "vie", "res", "myl" };
        protected static string[] duration_name = { "total", "monthly", "weekly", "daily", "hour" };
        protected static string[] duration_file = { "tot", "mon", "wek", "day", "hou" };
        protected static int[] number_of_all_page = { 10, 10, 10, 3, 1 };
        protected static int[] number_of_category_page = { 3, 1, 1, 1, 0 };
        protected static string[] category_name = { "all", "music", "ent", "anime", "game", "radio", "sport", "science",
                                "cooking", "politics", "animal", "history", "nature", "lecture",
                                "play", "sing", "dance", "owner", "diary", "que", "chat", "test", "commons",
                                "other", "r18"};
        protected static string[] category_file = { "all", "mus", "ent", "ame", "gam", "rad", "spt", "sci",
                                     "cok", "pol", "anl", "his", "nat", "lec", "ply", "sng", "dnc",
                                     "own", "dia", "que", "cht", "tst", "com", "oth", "r18"};

        protected bool[] duration_ = new bool[duration_name.Length];
        protected bool[] category_ = new bool[category_name.Length];

        protected FormatKind format_kind_ = FormatKind.Html;

        public DownloadKind()
        {
            for (int i = 0; i < duration_.Length; ++i)
            {
                duration_[i] = false;
            }
            for (int i = 0; i < category_.Length; ++i)
            {
                category_[i] = false;
            }
        }

        /// <summary>
        /// ダウンロードするランキングの期間を設定。true を指定するとその期間のランキングをDLする
        /// </summary>
        /// <param name="total">総合</param>
        /// <param name="month">月間</param>
        /// <param name="week">週間</param>
        /// <param name="day">デイリー</param>
        /// <param name="hour">毎時</param>
        public void SetDuration(bool total, bool month, bool week, bool day, bool hour)
        {
            duration_[0] = total;
            duration_[1] = month;
            duration_[2] = week;
            duration_[3] = day;
            duration_[4] = hour;
        }

        /// <summary>
        /// ダウンロードするランキングのカテゴリを設定。おおざっぱな設定しかできない
        /// </summary>
        /// <param name="all">総合</param>
        /// <param name="music">音楽、歌、演奏</param>
        /// <param name="dance">踊り、講座</param>
        /// <param name="other">それ以外</param>
        public void SetCategory(bool all, bool music, bool dance, bool other)
        {
            for (int i = 0; i < category_.Length; ++i)
            {
                category_[i] = other;
            }
            category_[0] = all;
            category_[1] = category_[14] = category_[15] = music;
            category_[13] = category_[16] = dance;
        }

        /// <summary>
        /// ダウンロードする形式の指定
        /// </summary>
        /// <param name="format_kind">HTML か RSS か</param>
        public void SetFormat(FormatKind format_kind)
        {
            format_kind_ = format_kind;
        }

        /// <summary>
        /// ダウンロードする形式を RSS に指定する
        /// </summary>
        public void SetFormatRss()
        {
            format_kind_ = FormatKind.Rss;
        }

        /// <summary>
        /// 設定されたダウンロード形式が RSS かどうかを取得
        /// </summary>
        /// <returns>true なら RSS、false なら HTML</returns>
        public bool IsFormatRss()
        {
            return format_kind_ == FormatKind.Rss;
        }

        /// <summary>
        /// ダウンロードするランキングのURLをリストで取得
        /// </summary>
        /// <param name="name_list">URL のリスト</param>
        /// <param name="filename_list">ある規則に従ったファイル名のリスト</param>
        public virtual void GetRankingNameList(ref List<string> name_list, ref List<string> filename_list)
        {
            for (int i = 0; i < kind_name.Length; ++i)
            {
                for (int j = 0; j < duration_name.Length; ++j)
                {
                    if (!duration_[j])
                    {
                        continue;
                    }
                    for (int k = 0; k < category_name.Length; ++k)
                    {
                        if (!category_[k])
                        {
                            continue;
                        }
                        int end = (k == 0) ? number_of_all_page[j] : number_of_category_page[j];
                        for (int m = 1; m <= end; ++m)
                        {
                            string option = "";
                            if (m >= 2)
                            {
                                option = "page=" + m.ToString();
                            }
                            if (format_kind_ == FormatKind.Rss)
                            {
                                if (option != "")
                                {
                                    option += "&";
                                }
                                option += "rss=2.0";
                            }
                            if (option != "")
                            {
                                option = "?" + option;
                            }
                            if (name_list != null)
                            {
                                name_list.Add(kind_name[i] + "/" + duration_name[j] + "/" +
                                        category_name[k] + option);
                            }
                            if (filename_list != null)
                            {
                                filename_list.Add(duration_file[j] + "_" + category_file[k] + "_" +
                                    kind_file[i] + "_" + m.ToString() + "_");
                            }
                        }
                    }
                }
            }
        }
    }

    /// <summary>
    /// ブラウザからクッキー内の user_session を取得するクラス。かなりいいかげんな実装
    /// </summary>
    static class NicoUserSession
    {
        [DllImport("wininet.dll")]
        private extern static bool InternetGetCookie(string url, string name, StringBuilder data, ref uint size);

        /// <summary>
        /// IE6 からクッキーを取得
        /// </summary>
        /// <param name="url">取得するクッキーに関連づけられたURL</param>
        /// <returns>クッキー文字列</returns>
        public static string GetCookieFromIE6(string url)
        {
            uint size = 4096;
            StringBuilder buff = new StringBuilder(new String(' ', (int)size), (int)size);
            InternetGetCookie(url, null, buff, ref size);
            return buff.ToString().Replace(';', ',');
        }

        /// <summary>
        /// IE6 から user_session を取得
        /// </summary>
        /// <param name="url">サイト（ニコニコ動画）のURL</param>
        /// <returns>user_session</returns>
        public static string GetUserSessionFromIE6(string url)
        {
            return CutUserSession(GetCookieFromIE6(url));
        }

        /// <summary>
        /// Firefox3 から user_session を取得。エラーが起こった場合、例外を投げずに空文字を返す
        /// </summary>
        /// <returns>user_session</returns>
        public static string GetUserSessionFromFilefox3()
        {
            string user_session = "";
            try
            {
                string app_dir = System.Environment.GetEnvironmentVariable("APPDATA") + "\\Mozilla\\Firefox\\Profiles\\";

                string sqlist_filename = SearchFile("cookies.sqlite", app_dir);
                if (sqlist_filename == "" || !File.Exists(sqlist_filename))
                {
                    return "";
                }
                FileStream fs = new FileStream(sqlist_filename, FileMode.Open, FileAccess.Read, FileShare.ReadWrite);
                byte[] data = new byte[fs.Length];
                fs.Read(data, 0, data.Length);
                fs.Close();
                const string user_session_prefix = "user_session_";
                
                for (int i = 0; i < data.Length; ++i)
                {
                    int j;
                    for (j = 0; j < user_session_prefix.Length; ++j)
                    {
                        if (data[i + j] != (byte)user_session_prefix[j])
                        {
                            break;
                        }
                    }
                    if (j >= user_session_prefix.Length)
                    {
                        while ((byte)'0' <= data[i + j] && data[i + j] <= (byte)'9' || data[i + j] == (byte)'_')
                        {
                            ++j;
                        }
                        for (int k = i; k < i + j; ++k)
                        {
                            user_session += (char)data[k];
                        }
                        break;
                    }
                }
            }
            catch (Exception) { }
            return user_session;
        }

        /// <summary>
        /// IE7 から user_session を取得。エラーが起こった場合、例外を投げずに空文字を返す
        /// </summary>
        /// <returns>user_session</returns>
        public static string GetUserSessionFromIE7()
        {
            string user_session = "";

            string profile_dir = System.Environment.GetEnvironmentVariable("USERPROFILE");
            user_session = GetUserSessionFromDirectory(profile_dir + "\\AppData\\Roaming\\Microsoft\\Windows\\Cookies\\Low\\");
            if (user_session == "")
            {
                user_session = GetUserSessionFromDirectory(profile_dir + "\\AppData\\Roaming\\Microsoft\\Windows\\Cookies\\");
            }
            if (user_session == "")
            {
                user_session = GetUserSessionFromDirectory(profile_dir + "\\Cookies\\");
            }
            return user_session;
        }

        private static string GetUserSessionFromDirectory(string dir_name)
        {
            string user_session = "";
            if (Directory.Exists(dir_name))
            {
                try
                {
                    string[] files = Directory.GetFiles(dir_name);

                    for (int i = 0; i < files.Length; ++i)
                    {
                        string filename = Path.GetFileName(files[i]);
                        if (filename.IndexOf("nicovideo") >= 0 && filename.IndexOf("www") < 0)
                        {
                            user_session = CutUserSession(File.ReadAllText(files[i], Encoding.GetEncoding(932)));
                        }
                    }
                }
                catch (Exception) { }
            }
            return user_session;
        }

        /// <summary>
        /// 文字列から user_session_ で始まる文字列を切り出して返す。数字とアンダーバー以外の文字で切れる。
        /// </summary>
        /// <param name="str">切り出す対象文字列</param>
        /// <returns>user_session 文字列。見つからなければ空文字を返す</returns>
        private static string CutUserSession(string str)
        {
            int start = str.IndexOf("user_session_");
            if (start >= 0)
            {
                int index = start + "user_session_".Length;
                while (index < str.Length && ('0' <= str[index] && str[index] <= '9' || str[index] == '_'))
                {
                    ++index;
                }
                return str.Substring(start, index - start);
            }
            return "";
        }

        /// <summary>
        /// DateTime を Unix 時間に変換
        /// </summary>
        /// <param name="datetime">変換する DateTime</param>
        /// <returns>Unix 時間の文字列</returns>
        public static string DateToUnix(DateTime datetime)
        {
            DateTime origin = new DateTime(1970, 1, 1, 9, 0, 0);
            TimeSpan span = datetime - origin;
            return ((int)span.TotalSeconds).ToString();
        }

        // dir_name 内を再帰的に調べて filename を含む最初のファイル名を返す
        private static string SearchFile(string filename, string dir_name)
        {
            if (!dir_name.EndsWith("\\"))
            {
                dir_name += "\\";
            }
            string[] files = Directory.GetFiles(dir_name);
            for (int i = 0; i < files.Length; ++i)
            {
                if (Path.GetFileName(files[i]).IndexOf(filename) >= 0)
                {
                    return files[i];
                }
            }
            string[] dirs = Directory.GetDirectories(dir_name);
            for (int i = 0; i < dirs.Length; ++i)
            {
                string ret = SearchFile(filename, dirs[i]);
                if (ret != "")
                {
                    return ret;
                }
            }
            return "";
        }
    }

    public class NiconicoLoginException : Exception
    {
        public NiconicoLoginException()
            : base("ログインされていません。")
        {

        }
    }

    public class NiconicoAccessFailedException : Exception
    {
        public NiconicoAccessFailedException()
            : base("ファイル情報の取得に失敗しました。ログインされていないかもしれません。")
        {

        }
    }

    public class NiconicoAccessDeniedException : Exception
    {

    }

    public class NiconicoAddingMylistFailedException : Exception
    {
        public NiconicoAddingMylistFailedException() { }

        public NiconicoAddingMylistFailedException(string message)
            : base(message)
        {

        }
    }
}
