﻿// Copyright (c) 2008 - 2009 rankingloid
//
// under GNU General Public License Version 2.
//
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text.RegularExpressions;
using IJLib;
using IJLib.Vfw;
using NicoTools;

namespace nicorank
{
    /// <summary>
    /// レイアウトを表すクラス
    /// </summary>
    public class Layout
    {
        List<string[]> layout_ = new List<string[]>();
        int line_number_ = 1; // デフォルト値（以下同様）
        int img_width_ = 512;
        int img_height_ = 384;
        int range_start_ = 1;
        int range_end_ = int.MaxValue;

        string base_dir_ = "";
        string filename_ = "%d.png";
        System.Drawing.Imaging.ImageFormat image_format_ = System.Drawing.Imaging.ImageFormat.Png;
        Color background_color_ = Color.White;
        bool is_back_transparent_ = false;

        int vfw_image_count_;
        bool has_video_ = false;

        LayoutVideoManager video_manager_ = new LayoutVideoManager();

        public Layout(string layout_text)
        {
            Parse(layout_text);
        }

        public bool HasVideo()
        {
            return has_video_;
        }
        
        public int GetWidth()
        {
            return img_width_;
        }

        public int GetHeight()
        {
            return img_height_;
        }

        public void Parse(string layout_text)
        {
            // 行番号が必要なので StringSplitOptions.None を指定している
            string[] lines = layout_text.Split(new string[] { "\r\n" }, StringSplitOptions.None);
            string base_dir = ""; // for VFW動画

            for (int i = 0; i < lines.Length; ++i)
            {
                if (lines[i] == "" || lines[i].StartsWith("#") || lines[i].StartsWith("//"))
                {
                    continue;
                }
                else if (lines[i].StartsWith("LineNumber", StringComparison.InvariantCultureIgnoreCase))
                {
                    int dummy = 0;
                    ReadValue(lines[i].Split('\t'), i, "LineNumber", ref line_number_, ref dummy);
                }
                else if (lines[i].StartsWith("OutputSize", StringComparison.InvariantCultureIgnoreCase))
                {
                    ReadValue(lines[i].Split('\t'), i, "OutputSize", ref img_width_, ref img_height_);
                }
                else if (lines[i].StartsWith("ReadRankRange", StringComparison.InvariantCultureIgnoreCase))
                {
                    ReadValue(lines[i].Split('\t'), i, "ReadRankRange", ref range_start_, ref range_end_);
                }
                else if (lines[i].StartsWith("FileName", StringComparison.InvariantCultureIgnoreCase))
                {
                    string[] sa = lines[i].Split('\t');
                    if (sa.Length >= 2 && sa[1] != "")
                    {
                        filename_ = sa[1];
                    }
                    else
                    {
                        throw new FormatException(GetErrorMessage(i, "FileName の後ろにファイル名を指定する必要があります。"));
                    }
                }
                else if (lines[i].StartsWith("FileType", StringComparison.InvariantCultureIgnoreCase))
                {
                    string[] sa = lines[i].Split('\t');
                    if (sa.Length >= 2 && sa[1] != "")
                    {
                        switch (sa[1])
                        {
                            case "jpg":
                            case "jpeg":
                                image_format_ = System.Drawing.Imaging.ImageFormat.Jpeg;
                                break;
                            case "png":
                                image_format_ = System.Drawing.Imaging.ImageFormat.Png;
                                break;
                            case "bitmap":
                            case "bmp":
                                image_format_ = System.Drawing.Imaging.ImageFormat.Bmp;
                                break;
                            case "gif":
                                image_format_ = System.Drawing.Imaging.ImageFormat.Gif;
                                break;
                            case "tiff":
                                image_format_ = System.Drawing.Imaging.ImageFormat.Tiff;
                                break;
                            default:
                                throw new FormatException(GetErrorMessage(i, "ファイルの種類 '" + sa[1] + "' には対応していません"));
                        }
                        filename_ = Path.ChangeExtension(filename_, sa[1]);
                    }
                    else
                    {
                        throw new FormatException(GetErrorMessage(i, "FileType の後ろにファイル種類を指定する必要があります。"));
                    }
                }
                else if (lines[i].StartsWith("BackgroundColor", StringComparison.InvariantCultureIgnoreCase))
                {
                    string[] sa = lines[i].Split('\t');
                    if (sa.Length >= 2 && sa[1].Equals("transparent", StringComparison.InvariantCultureIgnoreCase))
                    {
                        is_back_transparent_ = true;
                    }
                    else if (sa.Length >= 2 && sa[1] != "")
                    {
                        background_color_ = ColorTranslator.FromHtml(sa[1]);
                    }
                    else
                    {
                        throw new FormatException(GetErrorMessage(i, "BackgroundColor の後ろに色を指定する必要があります。"));
                    }
                }
                else if (lines[i].StartsWith("BaseDir", StringComparison.InvariantCultureIgnoreCase))
                {
                    string[] sa = lines[i].Split('\t');
                    if (sa.Length >= 2)
                    {
                        base_dir = sa[1];
                        if (base_dir != "" && !base_dir.EndsWith("\\"))
                        {
                            base_dir += "\\";
                        }
                    }
                    else
                    {
                        base_dir = "";
                    }
                    // コピペ
                    System.Diagnostics.Debug.Assert(sa.Length < 20); // 20を超えることはないと想定
                    string[] elem = new string[20];
                    for (int j = 0; j < elem.Length; ++j)
                    {
                        elem[j] = (j < sa.Length ? sa[j] : "");
                    }
                    if (elem[0].Equals("VFW動画", StringComparison.InvariantCultureIgnoreCase))
                    {
                        video_manager_.Add(elem, base_dir);
                        has_video_ = true;
                    }
                    layout_.Add(elem);
                }
                else
                {
                    string[] sa = lines[i].Split('\t');
                    System.Diagnostics.Debug.Assert(sa.Length < 20); // 20を超えることはないと想定
                    string[] elem = new string[20];
                    for (int j = 0; j < elem.Length; ++j)
                    {
                        elem[j] = (j < sa.Length ? sa[j] : "");
                    }
                    if (elem[0].Equals("VFW動画", StringComparison.InvariantCultureIgnoreCase))
                    {
                        video_manager_.Add(elem, base_dir);
                        has_video_ = true;
                    }
                    layout_.Add(elem);
                }
            }
        }

        /// <summary>
        /// レイアウトを元に、動画または画像を出力
        /// </summary>
        /// <param name="tsv_data">ランクファイルの中身（単なるタブ区切りテキストでよい）</param>
        /// <param name="frame_dir">出力フォルダ</param>
        /// <param name="msgout">情報出力オブジェクト</param>
        public void SaveAllToFile(string tsv_data, string frame_dir, MessageOut msgout)
        {
            if (!frame_dir.EndsWith("\\"))
            {
                frame_dir += "\\";
            }
            string[] ar = IJStringUtil.SplitWithCRLF(tsv_data);
            List<string[]> splitted_data = new List<string[]>();
            for (int i = 0; i < ar.Length; ++i)
            {
                splitted_data.Add(ar[i].Split('\t'));
            }
            int count = 1;
            for (int current_pos = range_start_ - 1; current_pos < range_end_ && current_pos < splitted_data.Count;
                current_pos += line_number_)
            {
                List<string[]> drawing_data = new List<string[]>();
                for (int j = current_pos; j < current_pos + line_number_ && j < range_end_ && j < splitted_data.Count; ++j)
                {
                    drawing_data.Add(splitted_data[j]);
                }
                string output_filename = frame_dir + NicoUtil.GetReplacedString(filename_, drawing_data); //.Replace("%d", count.ToString())
                // %d を count に置き換える。 %04d などの書式（桁揃え0埋め）も認める
                Regex regex = new Regex("%(0[1-9][0-9]*)?d");
                Match match = regex.Match(output_filename);
                if (match.Success)
                {
                    string num = match.Groups[1].Value;
                    if (num != "")
                    {
                        output_filename = regex.Replace(output_filename, count.ToString(new string('0', int.Parse(num))));
                    }
                    else
                    {
                        output_filename = regex.Replace(output_filename, count.ToString());
                    }
                }
                ++count;
                if (has_video_)
                {
                    SaveVideo(output_filename, drawing_data, msgout);
                }
                else
                {
                    SavePicture(output_filename, drawing_data);
                }
            }
        }

        public void SaveVideo(string output_filename, List<string[]> drawing_data, MessageOut msgout)
        {
            if (Path.GetExtension(output_filename) != ".avi")
            {
                output_filename = Path.ChangeExtension(output_filename, "avi");
            }
            video_manager_.OpenAll(drawing_data, msgout);

            if (!video_manager_.IsOpenAtLeastOne())
            {
                msgout.WriteLine("AVIファイルが正しくオープンできませんでした。");
                msgout.WriteLine(output_filename + " は出力されません。");
                return;
            }

            VfwAviWriter avi_writer = new VfwAviWriter(output_filename, video_manager_.Rate, video_manager_.Scale);
            try
            {
                for (int i = 0; i < video_manager_.Length; ++i)
                {
                    using (Bitmap bitmap = new Bitmap(img_width_, img_height_, System.Drawing.Imaging.PixelFormat.Format24bppRgb))
                    {
                        using (Graphics graphics = Graphics.FromImage(bitmap))
                        {
                            if (!is_back_transparent_)
                            {
                                using (Brush brush = new SolidBrush(background_color_))
                                {
                                    graphics.FillRectangle(brush, 0, 0, img_width_, img_height_);
                                }
                            }
                            vfw_image_count_ = 0;
                            DrawPicture(graphics, drawing_data, video_manager_.GetImageList(i));
                        }
                        avi_writer.AddFrame(bitmap);
                    }
                }
                
            }
            finally
            {
                avi_writer.Close();
            }
        }

        /// <summary>
        /// レイアウトを元に、画像を1枚出力
        /// </summary>
        /// <param name="filename">出力画像ファイル名</param>
        /// <param name="tsv_data">ランクファイルの中身（単なるタブ区切りテキストでよい）</param>
        public void SavePicture(string filename, string tsv_data)
        {
            string[] ar = IJStringUtil.SplitWithCRLF(tsv_data);
            List<string[]> splitted_data = new List<string[]>();
            for (int i = 0; i < ar.Length; ++i)
            {
                splitted_data.Add(ar[i].Split('\t'));
            }
            SavePicture(filename, splitted_data);
        }

        /// <summary>
        /// レイアウトを元に、画像を1枚出力
        /// </summary>
        /// <param name="filename">出力画像ファイル名</param>
        /// <param name="splitted_data">ランクファイルの中身を2次元リストとして区切ったもの</param>
        public void SavePicture(string filename, List<string[]> splitted_data)
        {
            using (Bitmap bitmap = new Bitmap(img_width_, img_height_))
            {
                using (Graphics graphics = Graphics.FromImage(bitmap))
                {
                    if (!is_back_transparent_)
                    {
                        using (Brush brush = new SolidBrush(background_color_))
                        {
                            graphics.FillRectangle(brush, 0, 0, img_width_, img_height_);
                        }
                    }
                    DrawPicture(graphics, splitted_data);
                    bitmap.Save(filename,
                        System.Drawing.Imaging.ImageFormat.Png);
                }
            }
        }

        public void DrawPicture(Graphics graphics, string tsv_data, bool is_replacing_video_to_picture)
        {
            if (is_replacing_video_to_picture)
            {
                Bitmap[] replacing_video_to_picture_list = video_manager_.GetReplacingVideoToPictureImageList();
                DrawPicture(graphics, tsv_data, video_manager_.GetReplacingVideoToPictureImageList());
                foreach (Bitmap bmp in replacing_video_to_picture_list)
                {
                    bmp.Dispose();
                }
            }
            else
            {
                DrawPicture(graphics, tsv_data, null);
            }
        }

        /// <summary>
        /// レイアウトを元に、画像を Graphics オブジェクトに描画
        /// </summary>
        /// <param name="graphics">Graphics オブジェクト</param>
        /// <param name="tsv_data">ランクファイルの中身（単なるタブ区切りテキストでよい）</param>
        public void DrawPicture(Graphics graphics, string tsv_data, Bitmap[] video_image_list)
        {
            string[] ar = IJStringUtil.SplitWithCRLF(tsv_data);
            List<string[]> splitted_data = new List<string[]>();
            for (int i = 0; i < ar.Length; ++i)
            {
                splitted_data.Add(ar[i].Split('\t'));
            }
            DrawPicture(graphics, splitted_data, video_image_list);
        }

        public void DrawPicture(Graphics graphics, List<string[]> splitted_data)
        {
            DrawPicture(graphics, splitted_data, null);
        }

        /// <summary>
        /// レイアウトを元に、画像を Graphics オブジェクトに描画
        /// </summary>
        /// <param name="graphics">Graphics オブジェクト</param>
        /// <param name="splitted_data">ランクファイルの中身を2次元リストとして区切ったもの</param>
        private void DrawPicture(Graphics graphics, List<string[]> splitted_data, Bitmap[] video_image_list)
        {
            base_dir_ = "";
            if (!is_back_transparent_)
            {
                using (Brush brush = new SolidBrush(background_color_))
                {
                    graphics.FillRectangle(brush, 0, 0, img_width_, img_height_);
                }
            }
            for (int i = 0; i < layout_.Count; ++i)
            {
                DrawFromLayoutRow(graphics, splitted_data, layout_[i], video_image_list);
            }
        }

        // レイアウトの一行を処理
        private void DrawFromLayoutRow(Graphics graphics, List<string[]> splitted_data, string[] layout, Bitmap[] video_image_list)
        {
            switch (layout[0].ToLower())
            {
                case "antialias":
                    if (layout[1].ToLower() == "on")
                    {
                        graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.AntiAlias;
                    }
                    else
                    {
                        graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.SystemDefault;
                    }
                    break;
                case "basedir":
                    base_dir_ = layout[1];
                    if (base_dir_ != "" && !base_dir_.EndsWith("\\"))
                    {
                        base_dir_ += "\\";
                    }
                    break;
                case "テキスト":
                    DrawText(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data).Replace("\\n", "\r\n"), layout);
                    break;
                case "テキスト四角形":
                    DrawTextRect(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data).Replace("\\n", "\r\n"), layout);
                    break;
                case "サイズ可変テキスト":
                    DrawTextSizeFree(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data).Replace("\\n", "\r\n"), layout);
                    break;
                case "幅固定テキスト":
                    DrawTextFixedWidth(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data).Replace("\\n", "\r\n"), layout);
                    break;
                case "幅固定横倍率可変テキスト":
                    DrawTextFixedWidthFlexible(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data).Replace("\\n", "\r\n"), layout);
                    break;
                case "画像テキスト":
                    DrawImageText(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data), layout, base_dir_);
                    break;
                case "画像テキスト四角形":
                    DrawImageTextRect(graphics, NicoUtil.GetReplacedString(layout[1], splitted_data), layout, base_dir_);
                    break;
                case "画像":
                    DrawImage(graphics, base_dir_ + NicoUtil.GetReplacedString(layout[1], splitted_data), layout);
                    break;
                case "vfw動画":
                    DrawVfwFrame(graphics, base_dir_ + NicoUtil.GetReplacedString(layout[1], splitted_data), layout, (video_image_list != null) ? video_image_list[vfw_image_count_] : null);
                    ++vfw_image_count_;
                    break;
            }
        }

        // line_num は0始まりの数字
        private string GetErrorMessage(int line_num, string message)
        {
            return "行番号 " + (line_num + 1).ToString() + ": " + message;
        }

        private void ReadValue(string[] sa, int line_num, string element_name, ref int value1, ref int value2)
        {
            if (sa.Length >= 2)
            {
                try
                {
                    value1 = int.Parse(sa[1]);
                    if (sa.Length >= 3 && sa[2] != "")
                    {
                        value2 = int.Parse(sa[2]);
                    }
                }
                catch (FormatException)
                {
                    throw new FormatException(GetErrorMessage(line_num, element_name + " には数字を指定する必要があります。"));
                }
            }
            else
            {
                throw new FormatException(GetErrorMessage(line_num, element_name + " には数字を指定する必要があります。"));
            }
        }

        private static void DrawText(Graphics graphics, string text, string[] layout_column)
        {
            FontStyle style = GetFontStyle(layout_column[4]);
            Font font = new Font(layout_column[2], int.Parse(layout_column[3]), style);
            Color color = ColorTranslator.FromHtml(layout_column[5]);
            SolidBrush brush = new SolidBrush(color);
            switch (layout_column[8].ToLower())
            {
                case "right":
                    DrawStringRight(graphics, text, font, brush, int.Parse(layout_column[6]), int.Parse(layout_column[7]));
                    break;
                default: // "left" とその他
                    graphics.DrawString(text, font, brush, int.Parse(layout_column[6]), int.Parse(layout_column[7]));
                    break;
            }
            brush.Dispose();
            font.Dispose();
        }

        private static void DrawTextRect(Graphics graphics, string text, string[] layout_column)
        {
            FontStyle style = GetFontStyle(layout_column[4]);
            Font font = new Font(layout_column[2], int.Parse(layout_column[3]), style);
            Color color = ColorTranslator.FromHtml(layout_column[5]);
            SolidBrush brush = new SolidBrush(color);
            StringFormat format = new StringFormat();
            switch (layout_column[10].ToLower())
            {
                case "left":
                    format.Alignment = StringAlignment.Near;
                    break;
                case "center":
                    format.Alignment = StringAlignment.Center;
                    break;
                case "right":
                    format.Alignment = StringAlignment.Far;
                    break;
            }
            switch (layout_column[11].ToLower())
            {
                case "top":
                    format.LineAlignment = StringAlignment.Near;
                    break;
                case "center":
                    format.LineAlignment = StringAlignment.Center;
                    break;
                case "bottom":
                    format.LineAlignment = StringAlignment.Far;
                    break;
            }
            RectangleF rect = new RectangleF(float.Parse(layout_column[6]), float.Parse(layout_column[7]),
                float.Parse(layout_column[8]), float.Parse(layout_column[9]));
            graphics.DrawString(text, font, brush, rect, format);
            brush.Dispose();
            font.Dispose();
        }

        private static void DrawTextSizeFree(Graphics graphics, string text, string[] layout_column)
        {
            int width = int.Parse(layout_column[8]);
            int font_size = int.Parse(layout_column[3]);
            int lower_size = ((layout_column.Length >= 13 && layout_column[12] != "") ? int.Parse(layout_column[12]) : 10);
            int step = ((layout_column.Length >= 14 && layout_column[13] != "") ? int.Parse(layout_column[13]) : 2);

            FontStyle style = GetFontStyle(layout_column[4]);
            for (; font_size >= lower_size; font_size -= step)
            {
                using (Font font = new Font(layout_column[2], font_size, style))
                {
                    int w = (int)graphics.MeasureString(text, font).Width;
                    if (w < width)
                    {
                        break;
                    }
                }
            }
            string temp = layout_column[3];
            layout_column[3] = font_size.ToString(); // フォントサイズを書き換える
            DrawTextRect(graphics, text, layout_column);
            layout_column[3] = temp; // 念のために元に戻す
        }

        private static void DrawTextFixedWidth(Graphics graphics, string text, string[] layout_column)
        {
            int font_size = int.Parse(layout_column[3]);
            float y = (float)int.Parse(layout_column[7]);

            FontStyle style = GetFontStyle(layout_column[4]);

            using (Font font = new Font(layout_column[2], font_size, style))
            {
                Color color = ColorTranslator.FromHtml(layout_column[5]);
                using (SolidBrush brush = new SolidBrush(color))
                {
                    float total_width = 0.0F;
                    float fixed_width = (float)int.Parse(layout_column[8]);
                    StringFormat format = new StringFormat();
                    CharacterRange[] cr = { new CharacterRange(0, 1) };
                    format.SetMeasurableCharacterRanges(cr);
                    RectangleF layout_rect = new RectangleF(0, 0, 200, 200);
                    for (int i = 0; i < text.Length; ++i)
                    {
                        string t = text.Substring(i, 1);
                        Region[] regions = graphics.MeasureCharacterRanges(t, font, layout_rect, format);
                        float w = regions[0].GetBounds(graphics).Width;
                        total_width += w;
                    }
                    float ratio = (total_width >= fixed_width ? fixed_width / total_width : 1.0F);
                    float x = (float)int.Parse(layout_column[6]);
                    switch (layout_column[9].ToLower())
                    {
                        case "center":
                            x -= ((total_width >= fixed_width ? fixed_width : total_width) / 2);
                            break;
                        case "right":
                            x -= (total_width >= fixed_width ? fixed_width : total_width);
                            break;
                    }
                    for (int i = 0; i < text.Length; ++i)
                    {
                        string t = text.Substring(i, 1);
                        Region[] regions = graphics.MeasureCharacterRanges(t, font, layout_rect, format);
                        float w = regions[0].GetBounds(graphics).Width;
                        graphics.DrawString(t, font, brush, x, y);
                        x += w * ratio;
                    }
                }
            }
        }

        private static void DrawTextFixedWidthFlexible(Graphics graphics, string text, string[] layout_column)
        {
            int font_size = int.Parse(layout_column[3]);
            float y = (float)int.Parse(layout_column[7]);

            FontStyle style = GetFontStyle(layout_column[4]);

            using (Font font = new Font(layout_column[2], font_size, style))
            {
                Color color = ColorTranslator.FromHtml(layout_column[5]);
                using (SolidBrush brush = new SolidBrush(color))
                {
                    float fixed_width = (float)int.Parse(layout_column[8]);

                    float string_width = graphics.MeasureString(text, font).Width;
                    float x = (float)int.Parse(layout_column[6]);

                    switch (layout_column[9].ToLower())
                    {
                        case "center":
                            x -= ((string_width >= fixed_width ? fixed_width : string_width) / 2);
                            break;
                        case "right":
                            x -= (string_width >= fixed_width ? fixed_width : string_width);
                            break;
                    }

                    if (string_width > fixed_width)
                    {
                        graphics.ScaleTransform(fixed_width / string_width, 1.0F);
                        x *= string_width / fixed_width;
                    }

                    graphics.DrawString(text, font, brush, x, y);

                    if (string_width > fixed_width)
                    {
                        graphics.ResetTransform();
                    }
                }
            }
        }

        private static void DrawImageText(Graphics graphics, string text, string[] layout_column, string base_dir)
        {
            Bitmap img = GetImageText(text, base_dir + layout_column[2]);
            double ratio = double.Parse(layout_column[3]);
            Rectangle rect;
            if (layout_column[6].ToLower() == "right")
            {
                rect = new Rectangle(int.Parse(layout_column[4]) - img.Width, int.Parse(layout_column[5]),
                    (int)(img.Width * ratio), (int)(img.Height * ratio));
            }
            else
            {
                rect = new Rectangle(int.Parse(layout_column[4]), int.Parse(layout_column[5]),
                    (int)(img.Width * ratio), (int)(img.Height * ratio));
            }
            graphics.DrawImage(img, rect);
            img.Dispose();
        }

        private static void DrawImageTextRect(Graphics graphics, string text, string[] layout_column, string base_dir)
        {
            Bitmap img = GetImageText(text, base_dir + layout_column[2]);
            double ratio = double.Parse(layout_column[3]);
            Rectangle rect = new Rectangle(0, 0, (int)(img.Width * ratio), (int)(img.Height * ratio));
            if (layout_column[8].ToLower() == "right")
            {
                rect.X = int.Parse(layout_column[4]) + int.Parse(layout_column[6]) - rect.Width;
            }
            else if (layout_column[8].ToLower() == "center")
            {
                rect.X = int.Parse(layout_column[4]) + (int.Parse(layout_column[6]) - rect.Width) / 2;
            }
            else
            {
                rect.X = int.Parse(layout_column[4]);
            }
            if (layout_column[9].ToLower() == "bottom")
            {
                rect.Y = int.Parse(layout_column[5]) + int.Parse(layout_column[7]) - rect.Height;
            }
            else if (layout_column[9].ToLower() == "center")
            {
                rect.Y = int.Parse(layout_column[5]) + (int.Parse(layout_column[7]) - rect.Height) / 2;
            }
            else
            {
                rect.Y = int.Parse(layout_column[5]);
            }
            graphics.DrawImage(img, rect);
            img.Dispose();
        }

        private static void DrawImage(Graphics graphics, string filename, string[] layout_column)
        {
            if (File.Exists(filename))
            {
                using (Bitmap img = new Bitmap(filename))
                {
                    int width, height;

                    if (layout_column.Length < 5 || layout_column[4] == "" || layout_column[4] == "0")
                    {
                        width = img.Width;
                    }
                    else
                    {
                        width = int.Parse(layout_column[4]);
                    }
                    if (layout_column.Length < 6 || layout_column[5] == "" || layout_column[5] == "0")
                    {
                        height = img.Height;
                    }
                    else
                    {
                        height = int.Parse(layout_column[5]);
                    }
                    graphics.DrawImage(img, int.Parse(layout_column[2]), int.Parse(layout_column[3]), width, height);
                }
            }
            else
            {
                IJLog.Write(filename + " は見つかりませんでした。\r\n");
            }
        }

        private static void DrawVfwFrame(Graphics graphics, string filename, string[] layout_column, Bitmap bitmap)
        {
            int width = int.Parse(layout_column[4]);
            int height = int.Parse(layout_column[5]);

            if (bitmap != null)
            {
                graphics.DrawImage(bitmap, int.Parse(layout_column[2]), int.Parse(layout_column[3]), width, height);
            }
        }

        private static int SearchString(string[][] info_matrix, string str)
        {
            for (int i = 0; i < info_matrix.Length; ++i)
            {
                if (info_matrix[i][0].Equals(str))
                {
                    return i;
                }
            }
            return -1;
        }

        // 画像テキストを返す。返り値は Dispose する必要がある
        public static Bitmap GetImageText(string text, string font_filename)
        {
            string info_text = IJFile.Read(font_filename + ".txt");
            Bitmap img = new Bitmap(font_filename + ".png");
            string[] line = IJStringUtil.SplitWithCRLF(info_text);
            string[][] info_matrix = new string[line.Length][];
            for (int i = 0; i < line.Length; ++i)
            {
                info_matrix[i] = line[i].Split('\t');
            }
            int ret_width = 0;
            int ret_height = 0;
            for (int i = 0; i < text.Length; ++i) // 画像の幅、高さを計算
            {
                string s = "";
                s += text[i];
                int index = SearchString(info_matrix, s);
                if (index >= 0)
                {
                    int width = int.Parse(info_matrix[index][3]);
                    int height = int.Parse(info_matrix[index][4]);
                    ret_width += width;
                    if (ret_height < height)
                    {
                        ret_height = height;
                    }
                }
            }
            Bitmap ret_image = new Bitmap(ret_width, ret_height);
            Graphics graphics = Graphics.FromImage(ret_image);
            int x = 0;
            for (int i = 0; i < text.Length; ++i)
            {
                string s = "";
                s += text[i];
                int index = SearchString(info_matrix, s);
                if (index >= 0)
                {
                    int sx = int.Parse(info_matrix[index][1]);
                    int sy = int.Parse(info_matrix[index][2]);
                    int width = int.Parse(info_matrix[index][3]);
                    int height = int.Parse(info_matrix[index][4]);
                    Rectangle src_rect = new Rectangle(sx, sy, width, height);
                    Rectangle dst_rect = new Rectangle(x, 0, width, height);
                    graphics.DrawImage(img, dst_rect, src_rect, GraphicsUnit.Pixel);
                    x += width;
                }
            }
            img.Dispose();
            return ret_image;
        }

        private static void DrawStringRight(Graphics g, string str, Font font, Brush brush, int right, int y)
        {
            SizeF size = g.MeasureString(str, font);
            g.DrawString(str, font, brush, right - size.Width, y);
        }

        private static FontStyle GetFontStyle(string name)
        {
            switch (name.ToLower())
            {
                case "bold":
                    return FontStyle.Bold;
                case "italic":
                    return FontStyle.Italic;
                case "underline":
                    return FontStyle.Underline;
                case "strikeout":
                    return FontStyle.Strikeout;
                default:
                    return FontStyle.Regular;
            }
        }
    }

    class LayoutVideoManager
    {
        List<LayoutVideo> video_list_ = new List<LayoutVideo>();

        public bool HasVideo()
        {
            return video_list_.Count > 0;
        }

        public int Length
        {
            get
            {
                for (int i = 0; i < video_list_.Count; ++i)
                {
                    if (video_list_[i].IsOpen)
                    {
                        return video_list_[i].GetLength();
                    }
                }
                return 0;
            }
        }

        public int Rate
        {
            get
            {
                for (int i = 0; i < video_list_.Count; ++i)
                {
                    if (video_list_[i].IsOpen)
                    {
                        return video_list_[i].GetRate();
                    }
                }
                return 0;
            }
        }

        public int Scale
        {
            get
            {
                for (int i = 0; i < video_list_.Count; ++i)
                {
                    if (video_list_[i].IsOpen)
                    {
                        return video_list_[i].GetScale();
                    }
                }
                return 0;
            }
        }

        // 少なくとも1つの動画がオープンされているか
        public bool IsOpenAtLeastOne()
        {
            for (int i = 0; i < video_list_.Count; ++i)
            {
                if (video_list_[i].IsOpen)
                {
                    return true;
                }
            }
            return false;
        }

        public void Add(string[] splitted_text, string base_dir)
        {
            video_list_.Add(new LayoutVideo(splitted_text, base_dir));
        }

        public void OpenAll(List<string[]> rank_data, MessageOut msgout)
        {
            for (int i = 0 ; i < video_list_.Count; ++i)
            {
                try
                {
                    video_list_[i].OpenVideo(rank_data);
                }
                catch (Exception e)
                {
                    msgout.WriteLine("ファイル " + video_list_[i].FileName + " が開けませんでした。");
                    msgout.WriteLine(e.Message);
                    msgout.WriteLine("このファイルは無視されます。");
                }
            }
        }

        public void CloseAll()
        {
            for (int i = 0; i < video_list_.Count; ++i)
            {
                video_list_[i].Close();
            }
        }

        public Bitmap[] GetImageList(int position)
        {
            Bitmap[] image_list = new Bitmap[video_list_.Count];
            for (int i = 0; i < video_list_.Count; ++i)
            {
                image_list[i] = video_list_[i].GetBitmap(position);
            }
            return image_list;
        }

        public Bitmap[] GetReplacingVideoToPictureImageList()
        {
            Bitmap[] image_list = new Bitmap[video_list_.Count];
            for (int i = 0; i < video_list_.Count; ++i)
            {
                image_list[i] = GetDummyVideoPicture(video_list_[i].GetWidth(), video_list_[i].GetHeight());
            }
            return image_list;
        }

        private static Bitmap GetDummyVideoPicture(int width, int height)
        {
            Bitmap bitmap = new Bitmap(width, height);
            using (Graphics graphics = Graphics.FromImage(bitmap))
            {
                graphics.FillRectangle(Brushes.Gray, 0, 0, width, height);
                using (Font font = new Font("ＭＳゴシック", 12))
                {
                    StringFormat format = new StringFormat();
                    format.Alignment = StringAlignment.Center;
                    format.LineAlignment = StringAlignment.Center;
                    graphics.DrawString("動画", font, Brushes.Black, new Rectangle(0, 0, width, height), format);
                }
            }
            return bitmap;
        }
    }

    class LayoutVideo
    {
        private VfwAviReader avi_reader_ = null;

        private string filename_ = "";
        private int video_x_ = 0;
        private int video_y_ = 0;
        private int video_width_ = 512;
        private int video_height_ = 384;

        private bool is_compress_ = false;

        public string FileName
        {
            get { return filename_; }
        }

        public bool IsOpen
        {
            get { return avi_reader_ != null; }
        }

        public LayoutVideo(string str, string base_dir)
        {
            Parse(str.Split('\t'), base_dir);
        }

        public LayoutVideo(string[] splitted_text, string base_dir)
        {
            Parse(splitted_text, base_dir);
        }

        public void Parse(string[] splitted_text, string base_dir)
        {
            filename_ = base_dir + splitted_text[1];
            is_compress_ = (splitted_text[8].ToLower() == "compress");
            video_x_ = int.Parse(splitted_text[2]);
            video_y_ = int.Parse(splitted_text[3]);
            video_width_ = int.Parse(splitted_text[4]);
            video_height_ = int.Parse(splitted_text[5]);
        }

        public int GetX()
        {
            return video_x_;
        }

        public int GetY()
        {
            return video_y_;
        }

        public int GetWidth()
        {
            return video_width_;
        }

        public int GetHeight()
        {
            return video_height_;
        }

        public void OpenVideo(List<string[]> rank_data)
        {
            bool is_success = false;
            try
            {
                if (avi_reader_ == null)
                {
                    string fname = NicoUtil.GetReplacedString(filename_, rank_data);
                    avi_reader_ = new VfwAviReader(fname);
                    is_success = true;
                }
            }
            finally
            {
                if (!is_success)
                {
                    if (avi_reader_ != null)
                    {
                        avi_reader_.Close();
                        avi_reader_ = null;
                    }
                }
            }
        }

        public int GetRate()
        {
            return avi_reader_.Rate;
        }

        public int GetScale()
        {
            return avi_reader_.Scale;
        }

        public int GetLength()
        {
            return avi_reader_.FrameLength;
        }

        public Bitmap GetBitmap(int position)
        {
            if (avi_reader_ != null)
            {
                if (position < avi_reader_.FrameLength)
                {
                    return avi_reader_.GetBitmap(position);
                }
                else
                {
                    return null;
                }
            }
            else
            {
                return null;
            }
        }

        public void Close()
        {
            if (avi_reader_ != null)
            {
                avi_reader_.Close();
                avi_reader_ = null;
            }
        }
    }
}