/* 
 * xfce4-newsreader-plugin
 * A lightweight RSS/Atom news feed reader on xfce4-panel
 * 
 * Copyright 2005 mueki <mueki@users.sourceforge.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 * Functions are described using gtk-doc style comments, even though no actual
 * documentation is extracted. The format should be self-explanatory.
 *
 * The 'xfce_control_class_init' function is the only direct access to the module
 * from the panel. It is probably easiest to start reading from there. It's
 * the last function definition in this file.
 **/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <libxml/uri.h>
#include <pthread.h>
#include <ne_uri.h>

#include "xmlparse.h"
#include "httpfetch.h"
#include "config_gui.h"
#include "newsreader.h"
#include "config_gui_callbacks.h"

/* $B%j%9%HJ]B8$K$O(BXBEL$B$r;HMQ(B */
#undef USE_OPML

#ifdef DEBUG
#define DEBUG_PRINT(...) printf("%s(%u): ",__func__,__LINE__); \
                         printf(__VA_ARGS__);printf("\n");
//#define DEBUG_PRINT(...)
#else
#define DEBUG_PRINT(...)
#endif

enum{
	DROP_TEXT_URI_LIST = 0,
	DROP_TEXT_X_MOZ_URL,
	DROP_NETSCAPE_URL,
	DROP_TEXT_HTML,
	DROP_TEXT_PLAIN,
};

static GtkTargetEntry drop_types[] = {
  /*
  { "STRING", GTK_TARGET_SAME_APP, 0 },
  { "BINARY", GTK_TARGET_SAME_WIDGET, 0 },
  */
  { "text/uri-list",  0, DROP_TEXT_URI_LIST },
  { "text/x-moz-url", 0, DROP_TEXT_X_MOZ_URL },
  { "_NETSCAPE_URL", 0, DROP_NETSCAPE_URL },
  { "text/html",      0, DROP_TEXT_HTML },
  { "text/plain",     0, DROP_TEXT_PLAIN }
};

static gint n_drop_types = sizeof(drop_types) / sizeof(drop_types[0]);
static const int TOOLTIPS_LEN = 256;

#define BORDER 6

/* newsreader$B%G!<%?$N<hF@(B,$B99?7$N$?$a$N%m%C%/(B */
/* static pthread_mutex_t newsreader_mutex = PTHREAD_MUTEX_INITIALIZER;*/

/* GetFeedList$B$X$N:FF~$rKI$0(B */
static pthread_mutex_t getfeedlist_mutex = PTHREAD_MUTEX_INITIALIZER;

/* $B%M%C%H%o!<%/$+$i$N%U%#!<%I%G!<%?<hF@$N$?$a$N%m%C%/(B*/
static pthread_mutex_t getfeed_mutex = PTHREAD_MUTEX_INITIALIZER;

/* $B%a%C%;!<%8%@%$%"%m%0I=<(MQ$N%m%C%/(B */
static pthread_mutex_t msgdisp_mutex = PTHREAD_MUTEX_INITIALIZER;

//#define NEWSREADER_LOCK()	pthread_mutex_lock(&newsreader_mutex)
//#define NEWSREADER_UNLOCK()	pthread_mutex_unlock(&newsreader_mutex)

#ifdef DEBUG
#define NEWSREADER_LOCK() printf("lock %s: %u\n",__func__,__LINE__);XFCE_PANEL_LOCK();
#define NEWSREADER_UNLOCK() printf("unlock %s: %u\n",__func__,__LINE__);XFCE_PANEL_UNLOCK();
#else
#define NEWSREADER_LOCK() XFCE_PANEL_LOCK();
#define NEWSREADER_UNLOCK() XFCE_PANEL_UNLOCK();
#endif

const gint INIT_LABEL_WIDTH = 300;
const gint INIT_UPDATE_MIN = 60;
const gint INIT_UPDATE_SEC = 10;

enum{
	ORDER_DESC = -1,
    ORDER_NONE,
	ORDER_ASC,
};

static feeddata_t*
create_new_feeddata()
{
	feeddata_t* pFeedData;
	pFeedData = g_new0(feeddata_t, 1);
	return pFeedData;
}

static feeddata_t*
copy_feeddata(feeddata_t* pFeedData)
{
	feeddata_t* pNewFeedData;

	if(!pFeedData)
		return NULL;

	DEBUG_PRINT("copy_feeddata start");

	pNewFeedData = create_new_feeddata();
	if(pFeedData->pchURI)
		pNewFeedData->pchURI = g_strdup(pFeedData->pchURI);

	pNewFeedData->pFeed = copy_feed(pFeedData->pFeed);
	pNewFeedData->tmLastModified = pFeedData->tmLastModified;
	pNewFeedData->pIcon = pFeedData->pIcon;
	pNewFeedData->iValidHours = pFeedData->iValidHours;
	DEBUG_PRINT("copy_feeddata end");

	return pNewFeedData;
}

void
free_feeddata(feeddata_t* pFeedData)
{
	if(!pFeedData)
		return;

	if(pFeedData->pFeed){
		free_feed(pFeedData->pFeed);
	}
	
	if(pFeedData->pchURI){
		g_free(pFeedData->pchURI);
	}
	
	if(pFeedData->pIcon){
		g_object_unref(pFeedData->pIcon);
	}
	
	g_free(pFeedData);

	return;
}

/* $B?75,$K%j%9%H$r:n@.$7(Bnewsreader->poOuglineList$B$r%3%T!<(B */
static GList*
copy_outlineList(newsreader_t* newsreader)
{
	GList* pList = NULL;
	GList* pOutlineList = NULL;
	outline_t* pOutlineSrc;
	outline_t* pOutlineDst;

	DEBUG_PRINT("start");

	for(pList = newsreader->poOutlineList; pList; pList = pList->next){
		if(!(pOutlineSrc = (outline_t*)pList->data)){
			DEBUG_PRINT("copy_outlinelist fatal error");
			return NULL;
		}
		pOutlineDst = g_new0(outline_t,1);
		if(pOutlineSrc->strTitle)
			pOutlineDst->strTitle = g_string_new(pOutlineSrc->strTitle->str);
		if(pOutlineSrc->strURL)
			pOutlineDst->strURL = g_string_new(pOutlineSrc->strURL->str);
		pOutlineDst->bAvailable = pOutlineSrc->bAvailable;
		pOutlineDst->iValidHours = pOutlineSrc->iValidHours;
		pOutlineList = g_list_append(pOutlineList, pOutlineDst);
	}	

	DEBUG_PRINT("end");
	
	return pOutlineList;
}

static void
free_outlineList(GList* pOutlineList)
{
	GList* pList;
	
	DEBUG_PRINT("free_outlineList start");

	for(pList = pOutlineList; pList; pList = pList->next){
		free_outline((outline_t*)pList->data);
	}

	g_list_free(pOutlineList);

	DEBUG_PRINT("free_outlineList end");

	return;
}

/* $B%"%$%F%`%j%9%H$N%=!<%H$K;HMQ(B */
/* use for item list sort */
static gint 
item_list_compare(item_t* item_a, item_t* item_b, gpointer user_data)
{
	return (gint)(GPOINTER_TO_INT(user_data) * (item_a->tmModified - item_b->tmModified));
}

static void
StoreFeedList2File(newsreader_t* newsreader)
{
	gchar* filename;

#ifdef USE_OPML
	/* OPML$B$GJ]B8(B */
	filename = xfce_resource_save_location (XFCE_RESOURCE_CONFIG,
											"xfce4" G_DIR_SEPARATOR_S "newsreader.xml",
											TRUE);

	storeOutlineList(newsreader->poOutlineList, filename);
	g_free(filename); 
#else

	/* XBEL$B$GJ]B8(B */
	filename = xfce_resource_save_location (XFCE_RESOURCE_CONFIG,
											"xfce4" G_DIR_SEPARATOR_S "newsreader.xml",
											TRUE);
	storeFeedList_XBEL(newsreader->poOutlineList, filename);
	g_free(filename);
#endif

	return;
}

static void
write_cache_file(const gchar* xmlBuff, int bufflen, gchar* pchURI, GHashTable* pTable)
{
	ne_uri uri = {0};
	GString* strFile;
	gchar* pchFileName;
	gchar* pchPath;
	gchar* pPrefix;
	int fd;

	if(!pchURI){
		g_warning("URI is NULL!!!");
		return;
	}
	
	NEWSREADER_LOCK();

	if(!(pchFileName = (gchar*)g_hash_table_lookup(pTable, pchURI))){
		if(ne_uri_parse(pchURI, &uri)
		   || uri.host == NULL){
			g_warning("ne_uri_parse faild");
			NEWSREADER_UNLOCK();
			return;
		}
		
		pPrefix = g_strdup(uri.host);
		pPrefix = g_strdelimit(pPrefix, ".", '_');
		strFile = g_string_new(pPrefix);
		g_string_append_printf(strFile, "%08X", (unsigned int)random());
		pchFileName = g_string_free(strFile, FALSE);
		g_hash_table_insert(pTable, g_strdup(pchURI), pchFileName);
	}

	strFile = g_string_new("xfce4" G_DIR_SEPARATOR_S "newsreader" G_DIR_SEPARATOR_S);
	g_string_append(strFile, pchFileName);
	pchPath = xfce_resource_save_location(XFCE_RESOURCE_CACHE, strFile->str, TRUE);
	g_string_free(strFile, TRUE);

	DEBUG_PRINT("write cahce file: %s", pchPath);

	if((fd = open(pchPath, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR)) < 0){
		g_warning("open faild");
		g_free(pchPath);
		NEWSREADER_UNLOCK();
		return;
	}
	
	if(write(fd, xmlBuff, bufflen - 1) < 0){
		g_warning("write faild");
		g_free(pchPath);
		close(fd);
		NEWSREADER_UNLOCK();
		return;
	}

	g_free(pchPath);
	close(fd);
	NEWSREADER_UNLOCK();

	return;
}

static gchar*
get_cache_file(gchar* pchURI, int* plen, GHashTable* pTable)
{
	gchar* pchFile;
	GString* strFile;
	gchar* pchPath;
	GArray* pData;
	gchar buff[1024 * 10];
	int bufflen;
	int fd;
	
	DEBUG_PRINT("get_cache_file start");

	if(!pchURI)
		return NULL;

	if(!(pchFile = g_hash_table_lookup(pTable, pchURI))){
		g_warning("not found cache file: %s", pchURI);
		return NULL;
	}
	
	strFile = g_string_new("xfce4" G_DIR_SEPARATOR_S "newsreader" G_DIR_SEPARATOR_S);	
	g_string_append(strFile, pchFile);
	pchPath = xfce_resource_lookup(XFCE_RESOURCE_CACHE, strFile->str);
	g_string_free(strFile, TRUE);
	if(!pchPath)
		return NULL;

	DEBUG_PRINT("Get Cache file: %s", pchPath);

	pData = g_array_new(TRUE, FALSE, sizeof(gchar));
	if((fd = open(pchPath, O_RDONLY)) < 0){
		g_warning("open faild");
		g_free(pchPath);
		return NULL;
	}

	while((bufflen = read(fd, buff, 1024*10)) > 0){
		pData = g_array_append_vals(pData, buff, bufflen);
	}

	close(fd);

	*plen = pData->len;

	DEBUG_PRINT("get_cache_file end");

	return g_array_free(pData, FALSE);
}

/* based on gaim_program_is_valid */
static gboolean
program_is_valid(const char *program)
{
	GError *error = NULL;
	char **argv;
	gchar *progname;
	gboolean is_valid = FALSE;

	g_return_val_if_fail(program != NULL,  FALSE);
	g_return_val_if_fail(*program != '\0', FALSE);

	if (!g_shell_parse_argv(program, NULL, &argv, &error)) {
		g_warning("Could not parse program '%s': %s\n",
				  program, error->message);
		g_error_free(error);
		return FALSE;
	}

	if (argv == NULL) {
		return FALSE;
	}

	progname = g_find_program_in_path(argv[0]);
	is_valid = (progname != NULL);

	g_strfreev(argv);
	g_free(progname);

	return is_valid;
}

/* $B%V%i%&%6$N5/F0(B */
static void
notify_uri(const char* web_command, const char* uri)
{
	gchar **split;
	gchar *command;
	GError *error = NULL;

	if (web_command == NULL || *web_command == '\0'){
		g_warning(_("Culdn't execute browser"));
		return;
	}

	if (uri == NULL || *uri == '\0'){
		g_warning(_("Culdn't execute browser"));
		return;
	}

	if (strstr(web_command, "%s")){
		split = g_strsplit(web_command, "%s", 0);
		command = g_strjoinv(uri, split);
		g_strfreev(split);
	}
	else{
		/*
		 * There is no "%s" in the browser command.  Assume the user
		 * wanted the URL tacked on to the end of the command.
		 */
		command = g_strdup_printf("%s %s", web_command, uri);
	}

	DEBUG_PRINT(command);

	if (!program_is_valid(command)){
		gchar *tmp;
		tmp = g_strdup_printf(_("The browser command \"%s\" is invalid."),
							  command);

		g_warning(_("Unable to open URL"));
		g_free(tmp);
	}
	else if (!g_spawn_command_line_async(command, &error)){
		gchar *tmp = g_strdup_printf(_("Error launching \"%s\": %s"),
									 command, error->message);

		g_warning(_("Unable to open URL"));

		g_free(tmp);
		g_error_free(error);
	}
	
	g_free(command);
	return;
}

/* $B%a%C%;!<%8I=<(4X?t(B (gtk$B$N%a%$%s%k!<%W$+$i8F$P$l$k(B) */
static gboolean
msg_display_func(newsreader_t* newsreader)
{
	gchar* pchMsg = NULL;
	GList* pList;
	msgdata_t* pMsgdata;
	GtkWidget *dialog;

	pthread_mutex_lock(&msgdisp_mutex);
		
	while((pList = newsreader->pMsgList)){
		if((pMsgdata = (msgdata_t*)pList->data)){
			if(pMsgdata->pchMsg){
				pchMsg = g_strdup(pMsgdata->pchMsg);
				g_free(pMsgdata->pchMsg);
			}
			else{
				DEBUG_PRINT("pMsgdata->pchMsg is null");
			}
			newsreader->pMsgList = g_list_remove(newsreader->pMsgList, pMsgdata);
			g_free(pMsgdata);
		}
		else{
			DEBUG_PRINT("pMsgdata is null");
		}
		
		DEBUG_PRINT(pchMsg);
		
		dialog = gtk_message_dialog_new (NULL,
										 0, //GTK_DIALOG_DESTROY_WITH_PARENT,
										 GTK_MESSAGE_WARNING,
										 GTK_BUTTONS_CLOSE,
										 (const gchar*)pchMsg);

		/* Destroy the dialog when the user responds to it (e.g. clicks a button) */
		g_signal_connect_swapped (dialog, "response",
								  G_CALLBACK (gtk_widget_destroy),
								  dialog);

		gtk_widget_show_all (dialog);
		g_free(pchMsg);		
	}
	
	pthread_mutex_unlock(&msgdisp_mutex);

	/* FALSE$B$rJV$7$F=*N;$HF1;~$K%$%Y%s%H%k!<%W$h$j:o=|$7$F$b$i$&(B */
	return FALSE;
}

/* $B%$%Y%s%H%k!<%W$K%a%C%;!<%8%@%$%"%m%0I=<(4X?t$r%"%$%I%k%=!<%9$H$7$FEPO?(B */
static void
message_register(newsreader_t* newsreader, const gchar* pchMsg)
{
	msgdata_t *pMsgdata;

	pMsgdata = g_new0(msgdata_t, 1);
	pMsgdata->pchMsg = g_strdup(pchMsg);
	pthread_mutex_lock(&msgdisp_mutex);
	newsreader->pMsgList = g_list_append(newsreader->pMsgList, pMsgdata);
	pthread_mutex_unlock(&msgdisp_mutex);

	/* msg_display_func$B$O(B1$B2s$7$+<B9T$5$l$J$$$+$i(B
	   $B9b$a$NM%@hEY$G$bLdBj$J$$(B */
	g_idle_add_full(G_PRIORITY_HIGH_IDLE, (GSourceFunc)msg_display_func, newsreader, NULL);

	return;
}

/* $B%]%C%W%"%C%W%a%K%e!<MQ%j%9%H$N%/%j%"(B */
static void
clear_feed_list_menu(newsreader_t* newsreader)
{
	GtkWidget *menuFeedList;

	if(!(menuFeedList = gtk_menu_item_get_submenu(newsreader->wItmFeedList))){
		return;
	}
	
	/* $B3+J|$O$3$l$G$h$$(B? */
	gtk_widget_destroy(menuFeedList);

	g_list_foreach(newsreader->poMenuItemList, (GFunc)g_free, NULL);
	g_list_free(newsreader->poMenuItemList);
	newsreader->poMenuItemList = NULL;

	return;
}

/* $B%K%e!<%90lMw%a%K%e!<%/%j%C%/;~$N=hM}(B */
static void
on_itemmenu_activate_cb(GtkMenuItem *item, gpointer data)
{
	newsreader_t* newsreader = (newsreader_t*)data;
	GList* poList;
	itemuri_t* pItem;
	
	NEWSREADER_LOCK();

	for(poList = newsreader->poMenuItemList; poList; poList = poList->next){
		if((pItem = (itemuri_t*)poList->data) &&
		   pItem->wItem == item)
			break;
	}
	
	if(poList)
		notify_uri(newsreader->pchBrowserCmd, pItem->pchURI);	
		
	NEWSREADER_UNLOCK();

	return;

}

/* $B%"%$%3%s%j%9%H$+$i3:Ev$9$k(BURL$B$N(Bfavicon$B$r<hF@(B*/
static GdkPixbuf*
getFaviconPixbuf(newsreader_t* newsreader, const gchar* pchUri)
{
	GList* pIconList = newsreader->poIconList;

	if(!pchUri){
		DEBUG_PRINT("uri is null");
		return NULL;
	}
	
	for(;pIconList; pIconList = pIconList->next){
		favicon_t* pIcon = (favicon_t*)pIconList->data;
		if(!pIcon)
			continue;
		if(strcmp(pIcon->uri, pchUri)==0){
			if(pIcon->pIconPixbuf){
				return pIcon->pIconPixbuf;
				/*				gtk_image_set_from_pixbuf(wImage, gdk_pixbuf_scale_simple(pIcon->pIconPixbuf, 16, 16, GDK_INTERP_BILINEAR));
								break;*/
			}
		}
	}

	DEBUG_PRINT("not found icon");
	return NULL;
}

/* $B%]%C%W%"%C%W%a%K%e!<$N99?7(B */
static gboolean
update_feed_list_menu(newsreader_t* newsreader)
{
	GList *pFeedList, *pItemList;
	feeddata_t* pFeedData;
	item_t* pItem;
	GString* strLabel;
	struct tm *t_st;
	itemuri_t *pUri;
	gchar* pchUri = NULL;

	GtkWidget* menuFeedList;
	GtkWidget* menuItemList;
	GtkWidget* itmFeed;
	GtkWidget* itmItem;
	int iFeedNo, iItemNo;

	XFCE_PANEL_LOCK();

	clear_feed_list_menu(newsreader);

	menuFeedList = gtk_menu_new();

	iFeedNo = 0;

	DEBUG_PRINT("update_feed_list_menu Start");

	if(newsreader->g_pFeedList){
		for(pFeedList = newsreader->g_pFeedList->next; pFeedList; pFeedList = pFeedList->next){

			pFeedData = (feeddata_t*)pFeedList->data;
		
			if(!pFeedData)continue;
			
			if(!(pFeedData->pFeed)){
				DEBUG_PRINT("feed is null");
				continue;
			}
			
			if(!(pFeedData->pFeed->title)){
				DEBUG_PRINT("title is null");
				continue;
			}
			
			itmFeed = gtk_image_menu_item_new_with_label(pFeedData->pFeed->title->str);

			if(!(pFeedData->pFeed->url)){
				DEBUG_PRINT("uri is null");
				pchUri = NULL;
			}
			else{
				pchUri = pFeedData->pFeed->url->str;
			}
			
			GdkPixbuf* pIconPixbuf = getFaviconPixbuf(newsreader, pchUri);
			GtkImage* pIconImage = GTK_IMAGE(gtk_image_new());
			if(pIconPixbuf){
				gtk_image_set_from_pixbuf(pIconImage, pIconPixbuf);
			}
			else{
				gtk_image_set_from_stock(pIconImage, "gtk-dialog-info", GTK_ICON_SIZE_MENU);
			}
			
			gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(itmFeed), GTK_WIDGET(pIconImage));

			gtk_widget_show(itmFeed);
			gtk_menu_append(menuFeedList, itmFeed);		
			menuItemList = gtk_menu_new();

			iItemNo = 0;

			for(pItemList = pFeedData->pFeed->items; pItemList; pItemList = pItemList->next){
				pItem = (item_t*)pItemList->data;

				if(!pItem){
					DEBUG_PRINT("item is null");
					continue;
				}
				
				if(!pItem->title){
					DEBUG_PRINT("title is null");
					continue;
				}
				
				strLabel = g_string_new(pItem->title->str);
				if(pItem->tmModified >= 0){
					t_st = localtime(&pItem->tmModified);
					g_string_append_printf(strLabel, " - %04d/%02d/%02d %02d:%02d", 
										   t_st->tm_year+1900, t_st->tm_mon+1, t_st->tm_mday,
										   t_st->tm_hour, t_st->tm_min);
				}
			
				itmItem = gtk_menu_item_new_with_label(strLabel->str);
				gtk_widget_show(itmItem);
				gtk_menu_append(menuItemList, itmItem);

				pUri = g_new0(itemuri_t, 1);
				pUri->wItem = GTK_MENU_ITEM(itmItem);
				pUri->pchURI = pItem->url->str;
				newsreader->poMenuItemList = g_list_append(newsreader->poMenuItemList, pUri);

				g_signal_connect (G_OBJECT(itmItem), "activate",
								  G_CALLBACK(on_itemmenu_activate_cb), newsreader);
			
				iItemNo++;
			}
			
			if(!iItemNo){
				/* Item$B$,0l$D$b$J$$>l9g$N=hM}(B*/
				DEBUG_PRINT("Feed have no item");
				itmItem = gtk_menu_item_new_with_label(_("(no Item)"));
				gtk_widget_show(itmItem);
				gtk_menu_append(menuItemList, itmItem);
				gtk_widget_set_sensitive(itmItem, FALSE);
			}

			/* Item$B%j%9%H$r(BFeed$B%a%K%e!<$KDI2C(B */
			gtk_menu_item_set_submenu(GTK_MENU_ITEM(itmFeed), menuItemList);
			iFeedNo++;
		}
	}
	
	if(!iFeedNo){
		/* Feed$B$,0l$D$b$J$$>l9g$N=hM}(B*/
		DEBUG_PRINT("Feedlist have no feed");
		itmFeed = gtk_menu_item_new_with_label(_("(no Feed)"));
		gtk_widget_show(itmFeed);
		gtk_menu_append(menuFeedList, itmFeed);
		gtk_widget_set_sensitive(itmFeed, FALSE);
	}
	
	gtk_menu_item_set_submenu(newsreader->wItmFeedList, menuFeedList);

	DEBUG_PRINT("update_feed_list_menu End");

	//	return iFeedNo;

	XFCE_PANEL_UNLOCK();

	return FALSE;
}

/* $B%]%C%W%"%C%W%a%K%e!<$r(Bgtk main loop$B$K<B9T$7$F$b$&(B */
static void
update_popupmenu_register(newsreader_t* newsreader)
{
	g_idle_add_full(G_PRIORITY_HIGH_IDLE, (GSourceFunc)update_feed_list_menu, newsreader, NULL);

	return;
}

/* Feed$B%j%9%H$N@hF,$KI,$:2C$($k(BFeed */
static feeddata_t*  create_static_feed(void)
{

	feeddata_t* pFeedData = NULL;
	feed_t* pFeed = NULL;
	item_t* pItem;
 
	pFeedData = create_new_feeddata();
	pFeedData->pchURI = g_strdup("");

	pFeed = create_new_feed();
	pFeed->title = g_string_new("Newsreader");
	pFeed->url = g_string_new("");
  
	pItem = create_new_item();
	pItem->title = g_string_new("Newsreader ");
	g_string_append(pItem->title, VERSION);
	pItem->url = g_string_new("");

	pFeed->items = g_list_append(pFeed->items, pItem);

	pFeedData->pFeed = pFeed;
 
	return pFeedData;
}

static GdkPixbuf*
httpfetch_favicon(const char* uri)
{
	gchar* buff;
	int bufflen;
	GdkPixbufLoader* loader;
	GdkPixbuf* pixbuf = NULL;

	if((bufflen = httpfetch(uri, &buff, -1, NULL)) < 0){
		g_warning("favicon fetch failed");
		return NULL;
	}

	loader = gdk_pixbuf_loader_new();
	gdk_pixbuf_loader_write(loader, (const guchar*)buff, bufflen, NULL);

	pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
	gdk_pixbuf_loader_close(loader, NULL);

	g_free(buff);

	return pixbuf;
}

/* 3$BDL$j$NJ}K!$G(Bfavicon $B$r<hF@(B $B$@$a$J$i(BNULL$B$rJV$9(B */
GdkPixbuf*
get_favicon(feed_t* pfeed)
{
	xmlDocPtr doc;
	gchar* path;
	gchar* base;
	xmlChar* uri;
	GdkPixbuf* pixbuf;
	xmlURIPtr poURI;

	if(!pfeed){
		g_warning("feed is null");
		return NULL;
	}
	
	if(!pfeed->url){
		g_warning("url is null");
		return NULL;
	}

	/* $B%j%s%/@h$N%Z!<%8$K%"%$%3%s$N%Q%9$,4^$^$l$F$$$k>l9g(B */
	if((doc = htmlReadFile(pfeed->url->str, NULL, HTML_PARSE_NOERROR))){
		/* $B%Z!<%8$+$i%"%$%3%s$N%Q%9$r<hF@(B */
		if((path = get_favicon_path(doc))){
			uri = (xmlChar*)xmlBuildURI(path, pfeed->url->str);
			if((pixbuf = httpfetch_favicon(uri))){
				DEBUG_PRINT("get favicon method1");
				xmlFreeDoc(doc);
				g_free(path);
				xmlFree(uri);
				return pixbuf;
			}
			g_warning("icon path failed");
			g_free(path);
			xmlFree(uri);
		}
		xmlFreeDoc(doc);
	}

	/* feed$B%U%!%$%k$N$"$k%Q%9(B + favicon.ico */
	path = g_strdup("favicon.ico");
	uri = (xmlChar*)xmlBuildURI(path, pfeed->url->str);
	if((pixbuf = httpfetch_favicon(uri))){
		DEBUG_PRINT("get favicon method2");
		g_free(path);
		xmlFree(uri);
		return pixbuf;
	}

	xmlFree(uri);
	
	/* get http://$B%U%#!<%IG[I[%5!<%P%"%I%l%9(B/favicon.ico */
	if((poURI = xmlParseURI(pfeed->url->str))){
		base = g_strdup_printf("%s://%s", poURI->scheme, poURI->server);
		uri = (xmlChar*)xmlBuildURI(path, base);
		if((pixbuf = httpfetch_favicon(uri))){
			DEBUG_PRINT("get favicon method3");
			g_free(path);
			g_free(base);
			xmlFree(uri);
			return pixbuf;
		}
		g_free(path);
		g_free(base);
		xmlFree(uri);		
	}

	DEBUG_PRINT(" pixbuf is null");

	return NULL;
	/* free pixbuf? */	
}

/* append favicon to icon list*/
static void
AppendIconList(newsreader_t* newsreader, feeddata_t* pFeedData)
{
	GList* poList = NULL;
	GdkPixbuf* pPixbuf;

	NEWSREADER_LOCK();
	poList = newsreader->poIconList;
	favicon_t* favicon = NULL;
	/* List have target icon? */
	for(; poList; poList = poList->next){
		favicon_t* icon = (favicon_t*)poList->data;
		if(!icon)
			continue;
		if(strcmp(icon->uri, pFeedData->pFeed->url->str)==0){
			break;
		}
	}
	NEWSREADER_UNLOCK();

	if(!poList){
		favicon = g_new0(favicon_t, 1);
		favicon->uri = g_strdup(pFeedData->pFeed->url->str);
		pPixbuf = get_favicon(pFeedData->pFeed);
		if(pPixbuf){
			favicon->pIconPixbuf = gdk_pixbuf_scale_simple(pPixbuf, 16, 16, GDK_INTERP_BILINEAR);
			g_free(pPixbuf);
		}
		NEWSREADER_LOCK();
		newsreader->poIconList = g_list_append(newsreader->poIconList, favicon);
		NEWSREADER_UNLOCK();
	}

	return;
}

static feed_t* 
GetFeed(char* buff, int len)
{  
	xmlDoc *doc = NULL;
	xmlNode *root = NULL;
	feed_t* pfeed;
  
	if(!(doc = xmlParseMemory(buff, len))){
		g_warning("!!! parse error");
		return NULL;
	}
  
	if(!(root = xmlDocGetRootElement(doc))){
		g_warning("!!! get root element");
		xmlFreeDoc(doc);
		return NULL;
	}
  
	if(!(pfeed = get_feed(doc, root))){
		g_warning("!!! get feed");
	}

	if(pfeed && pfeed->items){
		pfeed->items = g_list_sort_with_data(pfeed->items, (GCompareDataFunc)item_list_compare, 
								   GINT_TO_POINTER(ORDER_DESC));
	}
	
	xmlUnlinkNode(root);
	xmlFreeNode(root);
	xmlFreeDoc(doc);

	return pfeed;
}

/* tmOldModified$B$h$j?7$7$$5-;v$N(BbIsNew$B$r(BTRUE$B$K(B */
static void
set_new_flag(feeddata_t* pFeedData, time_t tmOldModified)
{
	GList *pList;
	item_t *pItem;

	if(!pFeedData || !pFeedData->pFeed)
		return;

	if(tmOldModified < 0)
		return;

	for(pList = pFeedData->pFeed->items; pList; pList = pList->next){
		if(!(pItem = (item_t*)pList->data))
			continue;
		if( pItem->tmModified > tmOldModified)
			pItem->bIsNew = TRUE;
	}
	
	return;
}

static void
set_new_flag_all(feeddata_t* pFeedData, gboolean bIsNew)
{
	GList *pList;
	item_t *pItem;

	if(!pFeedData || !pFeedData->pFeed)
		return;

	for(pList = pFeedData->pFeed->items; pList; pList = pList->next){
		if(!(pItem = (item_t*)pList->data))
			continue;
		pItem->bIsNew = bIsNew;
	}
	
	return;
}

/* $B%U%#!<%I%j%9%H$h$j%U%#!<%I%G!<%?$r<hF@$9$k(B */
static gpointer
get_feed_list(newsreader_t* newsreader)
{
	char* buff;
	int feedNo = 0;
	int bufflen;
	feed_t* pfeed;
	feeddata_t* pFeedData;
	feeddata_t* pNewFeedData;

	time_t tmRetLastModified;

	GList* plist;
	GList* poFeedList = NULL;
	GList* poNewList = NULL;
	GList* poOldList = NULL;
	GList* pOutlineList = NULL;

	if(pthread_mutex_trylock(&getfeedlist_mutex)){
		/* $B$9$G$K(BGetFeedList$B$,<B9TCf(B */
		DEBUG_PRINT("GetFeedList have started");
		return NULL;
	}
	
	DEBUG_PRINT("GetFeedList Start");

	pthread_mutex_lock(&getfeed_mutex);

	NEWSREADER_LOCK();
	pOutlineList = copy_outlineList(newsreader);
	NEWSREADER_UNLOCK();

	poNewList = g_list_append(poNewList, create_static_feed());

	feedNo++;
	outline_t* poOutline;

	for(plist = pOutlineList; plist; plist = plist->next){
		pFeedData = NULL;
		poOutline = (outline_t*)plist->data;
		if(!poOutline || !poOutline->bAvailable)
			continue;

		NEWSREADER_LOCK();

		/* $BA02s<hF@$N(BFeedDataList$B$+$iL\E*$N(BFeed$B$r8!:w(B */
		for(poFeedList = newsreader->g_pFeedList; poFeedList; poFeedList = poFeedList->next){
			if(strcmp(((feeddata_t*)(poFeedList->data))->pchURI, poOutline->strURL->str) == 0){
				break;
			}
		}
		
		if(poFeedList)
			pFeedData = copy_feeddata((feeddata_t*)poFeedList->data);

		NEWSREADER_UNLOCK();
		
		DEBUG_PRINT("search feedlist end");

		/* $BA02s<hF@$N(BFeedData$B$,$"$C$?(B */
		if(poFeedList){
			DEBUG_PRINT("FeedData found");
			if((bufflen = httpfetch(pFeedData->pchURI, &buff, pFeedData->tmLastModified, &tmRetLastModified)) < 0){
				DEBUG_PRINT("httpfetch faild");
				if(HF_NOT_MODIFIED == bufflen ){
					DEBUG_PRINT("feed not modified");
					pFeedData->iValidHours = poOutline->iValidHours;
					set_new_flag_all(pFeedData, FALSE);
					poNewList = g_list_append(poNewList, pFeedData);
					feedNo++;
					continue;
				}

				if(HF_ERROR == bufflen){
					DEBUG_PRINT("HF_ERROR");
					if(!(buff = get_cache_file(pFeedData->pchURI, &bufflen, newsreader->pCacheTable))){
						free_feeddata(pFeedData);
						g_warning("httpfetch error for exits feed");
						continue;
					}
				}
			}
			else{
				/* store buff to cache file */
				write_cache_file(buff, bufflen, poOutline->strURL->str, newsreader->pCacheTable);
			}
			
		}
		else{
			DEBUG_PRINT("FeedData not found");
			if((bufflen = httpfetch(poOutline->strURL->str, &buff, -1, &tmRetLastModified)) < 0){
				DEBUG_PRINT("httpfetch faild");
				if(!(buff = get_cache_file(poOutline->strURL->str, &bufflen, newsreader->pCacheTable))){
					DEBUG_PRINT("get_cache_file faild");
					continue;
				}
			}
			else{
				DEBUG_PRINT("write_cache_file");
				write_cache_file(buff, bufflen, poOutline->strURL->str, newsreader->pCacheTable);
			}
			
		}

		if(!(pfeed = GetFeed(buff, bufflen))){
			free_feeddata(pFeedData);
			g_warning("GetFeed error");
			continue;
		}

		if(buff)
			g_free(buff);
		
		pNewFeedData = create_new_feeddata();
		pNewFeedData->pchURI = g_strdup(poOutline->strURL->str);
		pNewFeedData->pFeed = pfeed;
		pNewFeedData->tmLastModified = tmRetLastModified;
		pNewFeedData->iValidHours = poOutline->iValidHours;

		if(pFeedData){
			/* $B8=>u(Bfeed_t$B$N(BtmModified$B%U%#!<%k%I$O<hF@$7$F$$$J$$$N$G(B
			   feeddata_t$B$N(BtmLastModified$B$r;HMQ$7$F$*$/(B*/
			set_new_flag(pNewFeedData, pFeedData->tmLastModified);
		}
		else{
			set_new_flag_all(pNewFeedData, FALSE);
		}

		free_feeddata(pFeedData);

		DEBUG_PRINT("newsreader get URI: %s", pNewFeedData->pchURI);
		DEBUG_PRINT("newsreader - modified %d", (int)tmRetLastModified);

		AppendIconList(newsreader, pNewFeedData);
		
		poNewList = g_list_append(poNewList, pNewFeedData);
		feedNo++;
	}

	free_outlineList(pOutlineList);

	NEWSREADER_LOCK();

	newsreader->g_pCurFeed = NULL;
	newsreader->g_pCurItem = NULL;
	
	poOldList = newsreader->g_pFeedList;
	newsreader->g_pFeedList = poNewList;

	for(poFeedList = poOldList; poFeedList; poFeedList = poFeedList->next){
		free_feeddata((feeddata_t*)poFeedList->data);
	}
	g_list_free(poOldList);

	//	update_feed_list_menu(newsreader);
	update_popupmenu_register(newsreader);

	NEWSREADER_UNLOCK();

	pthread_mutex_unlock(&getfeed_mutex);
	pthread_mutex_unlock(&getfeedlist_mutex);

	return GINT_TO_POINTER(feedNo);

}

static gboolean SetUpdateFeedTimer(void *p_newsreader)
{
  struct newsreader_t *newsreader = (newsreader_t*) p_newsreader;
  pthread_t thread;

  NEWSREADER_LOCK();

  DEBUG_PRINT("SetUpdateFeedTimer start");

  if (newsreader->iFeedTimerID) {
	g_source_remove (newsreader->iFeedTimerID);
	newsreader->iFeedTimerID = 0;
  }

  /*g_thread_create((GThreadFunc)GetFeedList, newsreader, FALSE, NULL);*/
  pthread_create(&thread, NULL, (void*)get_feed_list, (void*)newsreader);
  pthread_detach(thread);

  newsreader->iFeedTimerID = g_timeout_add (newsreader->feedUpdateMin * 60 * 1000,
												  (GtkFunction) SetUpdateFeedTimer, newsreader);

  NEWSREADER_UNLOCK();

  DEBUG_PRINT("SetUpdateFeedTimer end");

  return (1);

}

/* $BI=<(%"%$%F%`$N%]%$%s%?$r0l$D8e$m$K$9$k(B */
/* current display item pointer to next */
static void
newsreader_item_next(newsreader_t* newsreader, time_t tmCur)
{

	gint iValidHours;

	for(;;){
		/* get next feed item to display */
		if(!newsreader->g_pCurFeed || !newsreader->g_pCurItem){
			newsreader->g_pCurFeed = newsreader->g_pFeedList;
			newsreader->g_pCurItem = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items;
		}else{
			if(!(newsreader->g_pCurItem = newsreader->g_pCurItem->next)){
				while((newsreader->g_pCurFeed = newsreader->g_pCurFeed->next)){
					if((newsreader->g_pCurItem = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items))
						break;
				}
				if(!newsreader->g_pCurFeed){
					newsreader->g_pCurFeed = newsreader->g_pFeedList;
					newsreader->g_pCurItem = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items;
				} 
			}
		}
		
		/* tmModified $B$,(B-1$B$J$iL5>r7o$KI=<((B */
		if(((item_t*)(newsreader->g_pCurItem->data))->tmModified < 0)break;

		if((iValidHours = ((feeddata_t*)(newsreader->g_pCurFeed->data))->iValidHours) > 0){
			if(((item_t*)(newsreader->g_pCurItem->data))->tmModified > tmCur - iValidHours * 3600)break;
		}
		else{
			break;
		}
	}
	
	return;
}

/* $BI=<(%"%$%F%`$N%]%$%s%?$r0l$DA0$K$9$k(B */
/* current display item pointer to prev */
static void
newsreader_item_prev(newsreader_t* newsreader, time_t tmCur)
{	
	gint iValidHours;
	GList* pList;

	for(;;){
		
		if(!newsreader->g_pCurFeed || !newsreader->g_pCurItem){
			newsreader->g_pCurFeed = newsreader->g_pFeedList;
			newsreader->g_pCurItem = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items;
		}else{
			if(!(newsreader->g_pCurItem = newsreader->g_pCurItem->prev)){
				while((newsreader->g_pCurFeed = newsreader->g_pCurFeed->prev)){
					for(pList = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items; 
						pList->next; pList = pList->next);

					if((newsreader->g_pCurItem = pList))
						break;
				}
				if(!newsreader->g_pCurFeed){
					for(pList = newsreader->g_pFeedList; pList->next; pList = pList->next);
					newsreader->g_pCurFeed = pList;

					for(pList = ((feeddata_t*)newsreader->g_pCurFeed->data)->pFeed->items; 
						pList->next; pList = pList->next);

					newsreader->g_pCurItem = pList;
				} 
			}
		}

		/* tmModified $B$,(B-1$B$J$iL5>r7o$KI=<((B */
		if(((item_t*)(newsreader->g_pCurItem->data))->tmModified < 0)break;

		if((iValidHours = ((feeddata_t*)(newsreader->g_pCurFeed->data))->iValidHours) > 0){
			if(((item_t*)(newsreader->g_pCurItem->data))->tmModified > tmCur - iValidHours * 3600)break;
		}
		else{
			break;
		}		
	}
	
	return;
}

/* $B8=:_$N%"%$%F%`$rI=<((B */
static void
display_current_item(newsreader_t* newsreader)
{	
	GString* strLabel = g_string_new("");
	GString* strDate = g_string_new("");
	GString* strTooltips = g_string_new("");
	item_t* pItem;
	struct tm *t_st;

	pItem = (item_t*)newsreader->g_pCurItem->data;
	g_string_printf(strLabel, "%s", pItem->title->str);

	if(pItem->tmModified >= 0){
		t_st = localtime(&pItem->tmModified);
		g_string_printf(strDate, "%04d/%02d/%02d %02d:%02d", 
						t_st->tm_year+1900, t_st->tm_mon+1, t_st->tm_mday,
						t_st->tm_hour, t_st->tm_min);
		g_string_append (strLabel, " - ");
		g_string_append (strLabel, strDate->str);
	}

	/* set icon */
	feed_t* pFeed = ((feeddata_t*)(newsreader->g_pCurFeed->data))->pFeed;
	gchar* szUrl = NULL;
	GdkPixbuf* pIconPixbuf = NULL;

	if(pFeed && (pFeed->url)){
		szUrl = pFeed->url->str;
	}

	if((pIconPixbuf = getFaviconPixbuf(newsreader, szUrl))){
		gtk_image_set_from_pixbuf(newsreader->wIcon, pIconPixbuf);
	}
	else{
		gtk_image_set_from_stock(newsreader->wIcon, "gtk-dialog-info", GTK_ICON_SIZE_MENU);	
	}	

	strTooltips = g_string_new("");

	if(pFeed->title){
		g_string_append(strTooltips, pFeed->title->str);
		g_string_append( strTooltips, "\n");
	}

	g_string_append (strTooltips, strLabel->str);
	
	/* set tooltips */
	gtk_tooltips_set_tip(newsreader->wTooltips, GTK_WIDGET(newsreader->wEvItem), strTooltips->str, "");

	gtk_label_set_text(newsreader->wLblItem, strLabel->str);
	
	g_string_free(strLabel, TRUE);
	g_string_free(strDate, TRUE);
	g_string_free(strTooltips, TRUE);

	return;
}

/*-- $BI=<(99?7MQ%?%$%^!<(B --*/
static gboolean SetTimer (void *p_newsreader)
	/* Recurrently update the panel-docked monitor through a timer */
{
    newsreader_t *newsreader = (newsreader_t*) p_newsreader;
	time_t tmCur;

	//    struct param_t *poConf = &(poPlugin->oConf.oParam);

	NEWSREADER_LOCK();

    if (newsreader->iTimerID) {
		g_source_remove (newsreader->iTimerID);
		newsreader->iTimerID = 0;
    }

	tmCur = time(NULL);
	
	newsreader_item_next(newsreader, tmCur);

	display_current_item(newsreader);

    newsreader->iTimerID = g_timeout_add (newsreader->itemUpdateSec * 1000,
										  (GtkFunction) SetTimer, newsreader);

	NEWSREADER_UNLOCK();

    return (1);
}				/* SetTimer() */

/**
 * newsreader_read_config
 *
 * @control : the #Control to read configuration for
 * @node    : an #xmlNodePtr (part of the panel config file) containing the
 *            configuration.
 **/
static void
newsreader_read_config (Control * control, xmlNodePtr node)
{
    xmlChar *value;
    int n;
    newsreader_t *newsreader;
	PangoFontDescription* pFontDesc = NULL;

	DEBUG_PRINT("newsreader_read_config start");

	newsreader = (newsreader_t *) control->data;

    if (!node){
		DEBUG_PRINT("node is null");

		/* $B%i%Y%k%5%$%:$ND4@0(B */
		//		gtk_widget_set_size_request(GTK_WIDGET(newsreader->wLblItem), newsreader->labelWidth, -1);
		//		gtk_widget_set_size_request(newsreader->widget, newsreader->labelWidth, -1);
		gtk_widget_set_size_request(newsreader->control->base, newsreader->labelWidth, -1);

		/* feed $BFI$_9~$_$N%?%$%^!<:F@_Dj(B */
		SetUpdateFeedTimer( newsreader );

		return;
	}

    /* xml properties ... */

    value = xmlGetProp (node, (const xmlChar *) "newsreader_labelWidth");

    if (value){
		n = (int) strtol (value, NULL, 0);

		if (n > 0)
			newsreader->labelWidth = n;

		g_free (value);
	}

	value = xmlGetProp (node, (const xmlChar *) "newsreader_feedUpdateMin");

    if (value){
		n = (int) strtol (value, NULL, 0);

		if (n > 0)
			newsreader->feedUpdateMin = n;

		g_free (value);
	}

	value = xmlGetProp (node, (const xmlChar *) "newsreader_itemUpdateSec");

    if (value){
		n = (int) strtol (value, NULL, 0);

		if (n > 0)
			newsreader->itemUpdateSec = n;

		g_free (value);
	}

	value = xmlGetProp (node, (const xmlChar *) "newsreader_itemFontName");
    if (value){
		if(newsreader->strItemFontName)
			g_string_free(newsreader->strItemFontName, TRUE);
		newsreader->strItemFontName = g_string_new(value);
		g_free (value);
	}

	value = xmlGetProp (node, (const xmlChar *) "newsreader_BrowserCommand");
    if (value){
		if(newsreader->pchBrowserCmd)
			g_free(newsreader->pchBrowserCmd);
		newsreader->pchBrowserCmd = g_strdup(value);
		g_free (value);
	}

	newsreader->bUseTooltips = FALSE;

	value = xmlGetProp (node, (const xmlChar *) "newsreader_Tooltips");
    if (value){
		if(strcmp(value, "true") == 0){
			newsreader->bUseTooltips = TRUE;
		} 
		g_free (value);
	}

	/* $B%i%Y%k%5%$%:$ND4@0(B */
	//gtk_widget_set_size_request(GTK_WIDGET(newsreader->wLblItem), newsreader->labelWidth, -1);
    //gtk_widget_set_size_request(newsreader->widget, newsreader->labelWidth, -1);
    gtk_widget_set_size_request(newsreader->control->base, newsreader->labelWidth, -1);

	/* $B%U%)%s%H@_Dj(B */
	if(newsreader->strItemFontName){
		pFontDesc = pango_font_description_from_string(newsreader->strItemFontName->str);
		gtk_widget_modify_font(GTK_WIDGET(newsreader->wLblItem), pFontDesc);
		pango_font_description_free(pFontDesc);
	}
	
	/* feed $BFI$_9~$_$N%?%$%^!<:F@_Dj(B */
	SetUpdateFeedTimer( newsreader );

	DEBUG_PRINT("newsreader_read_config end");

	return;
    /* do stuff to the widget or otherwise apply new settings */
}

/**
 * newsreader_write_config
 *
 * @control : the #Control to write configuration for
 * @node    : an #xmlNodePtr (part of the panel config file) containing the
 *            configuration.
 **/
static void
newsreader_write_config (Control * control, xmlNodePtr parent)
{
    char value[5];
    newsreader_t *newsreader;

    newsreader = (newsreader_t *) control->data;

    g_snprintf (value, 4, "%d", newsreader->labelWidth);
    xmlSetProp (parent, "newsreader_labelWidth", value);

	g_snprintf (value, 4, "%d", newsreader->feedUpdateMin);
    xmlSetProp (parent, "newsreader_feedUpdateMin", value);

	g_snprintf (value, 4, "%d", newsreader->itemUpdateSec);
    xmlSetProp (parent, "newsreader_itemUpdateSec", value);

	if(newsreader->strItemFontName)
		xmlSetProp (parent, "newsreader_itemFontName", newsreader->strItemFontName->str);

	if(newsreader->pchBrowserCmd)
		xmlSetProp (parent, "newsreader_BrowserCommand", newsreader->pchBrowserCmd);

	if(newsreader->bUseTooltips)
		xmlSetProp (parent, "newsreader_Tooltips", "true");

	NEWSREADER_LOCK();
	StoreFeedList2File(newsreader);
	NEWSREADER_UNLOCK();

	return;
}

/**
 * newsreader_attach_callback
 *
 * @control  : the #Control to attach callbacks to
 * @signal   : the signal name
 * @callback : callback function
 * @data     : user data
 *
 * The plugin is expected to run g_signal_connect() on all widgets that
 * receive events, at least one. This is used, for example to connect the
 * right-click menu.
 **/
static void
newsreader_attach_callback (Control * control, const char *signal,
			GCallback callback, gpointer data)
{
    newsreader_t *newsreader = control->data;

    g_signal_connect (newsreader->widget, signal, callback, data);
    g_signal_connect (newsreader->wBtnMenu, signal, callback, data);
}

static void
on_itemupdate_activate_cb(GtkMenuItem *item, gpointer data)
{
	newsreader_t *newsreader;
	newsreader = (newsreader_t*) data;

	SetUpdateFeedTimer( newsreader );
	return;
}

static void
on_menubutton_press_cb (newsreader_t *newsreader, GdkEvent *event)
{
	GtkMenu *menu;
	GdkEventButton *event_button;

	//	g_return_val_if_fail (widget != NULL, FALSE);
	//	g_return_val_if_fail (GTK_IS_MENU (widget), FALSE);
	g_return_if_fail (event != NULL);

	NEWSREADER_LOCK();

	DEBUG_PRINT("menu popup start");
	/* The "widget" is the menu that was supplied when 
	 * g_signal_connect_swapped() was called.
	 */
	menu = GTK_MENU (newsreader->wMenu);

	if (event->type == GDK_BUTTON_PRESS){
		event_button = (GdkEventButton *) event;
		if (event_button->button == 1){
			gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
							event_button->button, event_button->time);
		}
	}

	DEBUG_PRINT("menu popup end");

	NEWSREADER_UNLOCK();
	return;
}

/* $B%\%?%s%j%j!<%9$G8F$P$l$F%V%i%&%6$r5/F0$9$k(B */
static gboolean
on_button_release_cb (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	newsreader_t * newsreader = (newsreader_t*)data;

	if(!newsreader->g_pCurItem){
		g_warning("no item");
		return TRUE;
	}
			
	NEWSREADER_LOCK();
	if(!((item_t*)newsreader->g_pCurItem->data)->url->str ||
	   !(*((item_t*)newsreader->g_pCurItem->data)->url->str)){
		NEWSREADER_UNLOCK();
		return TRUE;
	}

	notify_uri(newsreader->pchBrowserCmd, ((item_t*)newsreader->g_pCurItem->data)->url->str);

	NEWSREADER_UNLOCK();

	return TRUE;
  
}

/* $B%U%)!<%+%9$,Ev$C$?$H$-$K8F$P$l$F%?%$%^!<$r;_$a$k(B */
static gboolean
on_enter_notify_cb (GtkEventBox *widget, GdkEventCrossing *event, newsreader_t* newsreader)
{

	NEWSREADER_LOCK();

	/* $B%^%&%9%+!<%=%k$NJQ99(B */
	gdk_window_set_cursor ( GTK_WIDGET(newsreader->wEvItem)->window, gdk_cursor_new ( GDK_HAND2));

	/* stop timer */
	if(newsreader->iTimerID) {
		g_source_remove (newsreader->iTimerID);
		newsreader->iTimerID = 0;
	}

	/* display tooltips */
	if(newsreader->bUseTooltips){
		gtk_tooltips_enable(newsreader->wTooltips);
	}

	NEWSREADER_UNLOCK();

	return TRUE;  
}

/* $B%U%)!<%+%9$,30$l$?$H$-$K8F$P$l$F%?%$%^!<$r(B
   $B:F%;%C%H$9$k(B */
static gboolean
on_leave_notify_cb (GtkEventBox *widget, GdkEventCrossing *event, newsreader_t* newsreader)
{
  if (newsreader->iTimerID) {
	g_source_remove (newsreader->iTimerID);
	newsreader->iTimerID = 0;
  }

  /* call SetTimer after 2sec */
  newsreader->iTimerID = g_timeout_add (2000,
										(GtkFunction) SetTimer, newsreader);

  gtk_tooltips_disable(newsreader->wTooltips);

  return TRUE;
}

static gboolean
on_button_press_cb(GtkEventBox *widget, GdkEventKey *event, newsreader_t* newsreader)
{

	GdkEventScroll* evScroll;
	time_t tmCur;

	if(GDK_SCROLL != event->type)
		return FALSE;

	evScroll = (GdkEventScroll*)event;

	tmCur = time(NULL);
		
	if(GDK_SCROLL_UP == evScroll->direction){
		NEWSREADER_LOCK();
		newsreader_item_prev(newsreader, tmCur);
		display_current_item(newsreader);
		NEWSREADER_UNLOCK();
	}
	else if(GDK_SCROLL_DOWN == evScroll->direction){
		NEWSREADER_LOCK();
		newsreader_item_next(newsreader, tmCur);
		display_current_item(newsreader);
		NEWSREADER_UNLOCK();
	}
	
	return FALSE;
}

static void
clear_cache_dir()
{
	
	gchar** ppchFileMatch;
	gchar** ppTemp;
	gchar* pchFileName;

	ppchFileMatch = xfce_resource_match(XFCE_RESOURCE_CACHE, 
								   "xfce4" G_DIR_SEPARATOR_S "newsreader" G_DIR_SEPARATOR_S "*",
								   FALSE);

	for(ppTemp = ppchFileMatch; *ppTemp; *ppTemp++){
		pchFileName = xfce_resource_lookup(XFCE_RESOURCE_CACHE, *ppTemp);
		DEBUG_PRINT("unlink: %s", pchFileName);
		if(unlink(pchFileName) < 0){
			g_warning(strerror(errno));	
		}
		
		g_free(pchFileName);
	}
	
	g_strfreev(ppchFileMatch);

	return;
}

/**
 * newsreader_new
 *
 * Returns a newly allocated and initialized #newsreader_t
 **/
static newsreader_t *
newsreader_new (void)
{
    newsreader_t *newsreader;
    GtkWidget *label;
	GtkWidget *hbox1;
	GtkWidget *alignment1;
	GtkWidget *button1;
	GtkWidget *image1;
	GtkWidget *eventbox1;

	srandom((int)time(NULL));

    newsreader = g_new0 (newsreader_t, 1);

	clear_cache_dir();

	/*	pthread_create(&newsreader->msgdisp_tid, NULL, (void*)msg_display_func, (void *) newsreader);*/

    /* initialize default settings */
    newsreader->labelWidth = INIT_LABEL_WIDTH;
	newsreader->feedUpdateMin = INIT_UPDATE_MIN;
	newsreader->itemUpdateSec = INIT_UPDATE_SEC;

	newsreader->pchBrowserCmd = g_strdup("firefox %s");

	newsreader->pCacheTable = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);

    /* this is our main widget that catches all events */
	hbox1 = gtk_hbox_new(FALSE, 0);
    newsreader->widget = hbox1;
    gtk_widget_show (newsreader->widget);

	alignment1 = gtk_alignment_new (0.5, 0.5, 1, 1);
	gtk_widget_show (alignment1);
	gtk_box_pack_start (GTK_BOX (hbox1), alignment1, FALSE, FALSE, 0);

	button1 = gtk_button_new ();
	gtk_widget_show (button1);
	gtk_container_add (GTK_CONTAINER (alignment1), button1);
	gtk_button_set_relief (GTK_BUTTON (button1), GTK_RELIEF_NONE);
	newsreader->wBtnMenu = GTK_BUTTON (button1);

	image1 = gtk_image_new_from_stock ("gtk-dialog-info", GTK_ICON_SIZE_MENU);
	gtk_widget_show (image1);
	gtk_container_add (GTK_CONTAINER (button1), image1);
	newsreader->wIcon = GTK_IMAGE(image1);

	eventbox1 = gtk_event_box_new ();
	gtk_widget_show (eventbox1);
	gtk_box_pack_start (GTK_BOX (hbox1), eventbox1, TRUE, TRUE, 5);
	newsreader->wEvItem = GTK_EVENT_BOX(eventbox1);

    /* give the user something to look at */
    label = gtk_label_new ("Newsreader");
    gtk_container_add (GTK_CONTAINER (eventbox1), label);
	gtk_misc_set_alignment (GTK_MISC (label), 0.01, 0.5);
    gtk_widget_show (label);
    newsreader->wLblItem = GTK_LABEL(label);

	/* create tooltip */
	newsreader->wTooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(newsreader->wTooltips, GTK_WIDGET(eventbox1), "newsreader", "");
	newsreader->bUseTooltips = TRUE;
 
	/* create model data*/
	newsreader->oGUI.wModel =GTK_TREE_MODEL( gtk_list_store_new (NUM_COLS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT));	

	/* get feed-list from file */
	gchar* filename;
	xmlDocPtr doc = NULL;
	filename = xfce_resource_lookup (XFCE_RESOURCE_CONFIG,
                                     "xfce4" G_DIR_SEPARATOR_S "newsreader.xml");	

    if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
		/* parse the file */
		doc = xmlParseFile(filename);
		if (doc) {
			DEBUG_PRINT("get feedlist from file: %s", filename);
			NEWSREADER_LOCK();
#ifdef USE_OPML
			newsreader->poOutlineList =  getOutlineList(doc, newsreader->poOutlineList);
#else
			newsreader->poOutlineList =  getFeedList_XBEL(doc, newsreader->poOutlineList);
#endif
			NEWSREADER_UNLOCK();
		}
		else{
			g_warning("parse error: %s", filename);
		}	
	}
	else{
		g_warning("file not found: %s", filename);
	}
	
	g_free(filename);

	return newsreader;
}

/**
 * newsreader_free
 *
 * free the memory allocated for a newsreader #Control
 *
 * @control : the #Control to free memory for.
 **/
static void
newsreader_free (Control * control)
{
    newsreader_t *newsreader = (newsreader_t *) control->data;

	if(!newsreader)
		return;

    /* free resources, stop timeouts, etc. */

	if(newsreader->iTimerID)
		g_source_remove(newsreader->iTimerID);

	if(newsreader->iFeedTimerID)
		g_source_remove(newsreader->iFeedTimerID);

	newsreader->g_pCurFeed = NULL;
	newsreader->g_pCurItem = NULL;

	GList* plist;
	for(plist = newsreader->g_pFeedList; plist; plist = plist->next)
		free_feeddata((feeddata_t*)plist->data);
	g_list_free(newsreader->g_pFeedList);

	newsreader->g_pFeedList = NULL;
	
	g_object_unref(newsreader->oGUI.wModel);

	g_hash_table_destroy(newsreader->pCacheTable);

	/* don't forget to free the memory for the newsreader_t struct itself */
    g_free (newsreader);
}

/**
 * newsreader_set_theme
 *
 * @control : #Control to set theme for
 * @theme   : theme name
 *
 * If your plugin uses icons, you should consider if you want to 
 * support icon themes. You can uses functions in panel/icons.h for this.
 **/
static void
newsreader_set_theme (Control * control, const char *theme)
{
    newsreader_t *newsreader;

    newsreader = (newsreader_t *) control->data;

    /* update the icons */
}

/**
 * free_newsreader_dialog
 *
 * @sd : the #NewsreaderDialog to free resources of
 *
 * Free memory allocated for the options dialog.
 **/
#if 0
static void
free_newsreader_dialog (NewsreaderDialog * rd)
{
    /* gtk widgets are handle by gtk */

    /* we have to free the NewsreaderDialog struct */
    g_free (rd);
}
#endif

/**
 * newsreader_apply_options
 *
 * @sd : #NewsreaderDialog to apply options for.
 *
 * This is our simple all-in-one callback for when one of the options changes.
 * In more complex situation you will probably want to have a separate
 * callback for each option.
 **/
static void
newsreader_apply_options (newsreader_t * newsreader)
{
	int iUpdateMin;
	PangoFontDescription* pFontDesc = NULL;
	
	gboolean bFeedUpdate = FALSE;
	
    newsreader->labelWidth =
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (newsreader->oGUI.wSpnWidth));

	iUpdateMin = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(newsreader->oGUI.wSpnMin));
	if(newsreader->feedUpdateMin != iUpdateMin){
		bFeedUpdate = TRUE;
		newsreader->feedUpdateMin = iUpdateMin;
	}

	/* $B%i%Y%kI}$N@_Dj(B */
	//    gtk_widget_set_size_request(GTK_WIDGET(newsreader->wLblItem), newsreader->labelWidth, -1);
	//    gtk_widget_set_size_request(newsreader->widget, newsreader->labelWidth, -1);
	gtk_widget_set_size_request(newsreader->control->base, newsreader->labelWidth, -1);

	if(newsreader->strItemFontName)
		g_string_free(newsreader->strItemFontName,TRUE);

	newsreader->strItemFontName = g_string_new(gtk_font_button_get_font_name(GTK_FONT_BUTTON(newsreader->oGUI.wFbtItemFont)));

	g_free(newsreader->pchBrowserCmd);
	newsreader->pchBrowserCmd = g_strdup(gtk_entry_get_text(GTK_ENTRY(newsreader->oGUI.wEntBrowserCmd)));

	/* $B%3%s%U%#%0$N%U%)%s%H@_Dj$r<hF@(B */
	pFontDesc = pango_font_description_from_string(newsreader->strItemFontName->str);

	/* $B%i%Y%k$K%U%)%s%H$r@_Dj(B */
	gtk_widget_modify_font(GTK_WIDGET(newsreader->wLblItem), pFontDesc);
	
	pango_font_description_free(pFontDesc);

	/* $B%D!<%k%A%C%W(B */
	newsreader->bUseTooltips = gtk_toggle_button_get_active((GtkToggleButton*)newsreader->oGUI.wChkTooltips);

	gboolean valid;
	GtkTreeIter iter;

	NEWSREADER_LOCK();

	free_outlineList(newsreader->poOutlineList);
	newsreader->poOutlineList = NULL;
	
	valid = gtk_tree_model_get_iter_first(newsreader->oGUI.wModel, &iter);

	while(valid){
		outline_t* pfeed;
		gchar* iter_title;
		gchar* iter_url;
		gboolean iter_available;
		gint iter_until;

		gtk_tree_model_get(newsreader->oGUI.wModel, &iter,
						   COL_TOGGLE, &iter_available,
						   COL_TITLE, &iter_title,
						   COL_URL, &iter_url,
						   COL_UNTIL, &iter_until,
						   -1);

		pfeed = g_new0 (outline_t, 1);
		pfeed->strTitle = g_string_new(iter_title);
		pfeed->strURL = g_string_new(iter_url);
		pfeed->bAvailable = iter_available;
		pfeed->iValidHours = iter_until;

		newsreader->poOutlineList = g_list_append(newsreader->poOutlineList, pfeed);

		if(iter_title)
			g_free(iter_title);
		if(iter_url)
			g_free(iter_url);

		valid = gtk_tree_model_iter_next(newsreader->oGUI.wModel, &iter);
	}
	
	NEWSREADER_UNLOCK();

	if(bFeedUpdate)
		SetUpdateFeedTimer( newsreader );
	
	SetTimer(newsreader);

    /* apply new settings */
}

/**
 * entry_lost_focus
 *
 * This shows a way to update plugin settings when the user leaves a text
 * entry, by connecting to the "focus-out" event on the entry.
 **/
#if 0
static gboolean
entry_lost_focus (NewsreaderDialog * sd)
{
    newsreader_apply_options (sd);

    /* NB: needed to let entry handle focus-out as well */
    return FALSE;
}
#endif

static void
width_changed (GtkSpinButton * spin, newsreader_t *newsreader )
{
    newsreader->labelWidth = gtk_spin_button_get_value_as_int (spin);
    //gtk_label_set_width_char((GtkLabel*)rd->newsreader->widget, rd->newsreader->labelWidth);
    
}

static void
updateSec_changed (GtkSpinButton * spin, newsreader_t * newsreader)
{
    newsreader->itemUpdateSec = gtk_spin_button_get_value_as_int (spin);
}

static void
set_model_data(newsreader_t* newsreader)
{
	gtk_list_store_clear(GTK_LIST_STORE(newsreader->oGUI.wModel));

	outline_t* poOutline;
	GList* poList;
	GtkTreeIter iter;

	NEWSREADER_LOCK();

	for(poList = newsreader->poOutlineList; poList; poList = poList->next){
		poOutline = (outline_t*)poList->data;
		gtk_list_store_append (GTK_LIST_STORE(newsreader->oGUI.wModel), &iter);
		gtk_list_store_set (GTK_LIST_STORE(newsreader->oGUI.wModel), &iter,
							COL_TOGGLE, poOutline->bAvailable,
							COL_TITLE, poOutline->strTitle->str,
							COL_URL, poOutline->strURL->str,
							COL_UNTIL, poOutline->iValidHours,
							-1);
	}

	NEWSREADER_UNLOCK();
}

static void
create_list_column(newsreader_t* newsreader)
{
	GtkWidget* view;
	GtkCellRenderer* renderer;

	view = newsreader->oGUI.wTrFeedList;
	renderer = gtk_cell_renderer_toggle_new ();
	g_signal_connect(renderer, "toggled", G_CALLBACK( row_toggled ), newsreader->oGUI.wModel);
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
												 -1,
												 "",  
												 renderer,
												 "active", COL_TOGGLE,
												 NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
												 -1,
												 "Title",  
												 renderer,
												 "text", COL_TITLE,
												 NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
												 -1,
												 "URL",  
												 renderer,
												 "text", COL_URL,
												 NULL);

	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL(newsreader->oGUI.wModel));
	//	g_object_unref (model); /* destroy model automatically with view */

	return;
}

/**
 * newsreader_create_options
 *
 * @control   : #Control to create dialog for
 * @container : #GtkContainer to pack our option widgets into.
 * @done      : the close button on the dialog as #GtkWidget
 *
 * The plugin should create the widgets to set options and connect the
 * appropriate signals. The plugin is expected to use instant-apply of the
 * settings as much as possible. At least for settings that affect the
 * appearance.
 **/
static void
newsreader_create_options (Control * control, GtkContainer * container,
		       GtkWidget * done)
{

	GtkTreeSelection* selection;
    newsreader_t *newsreader;

	newsreader = (newsreader_t*) control->data;

    xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

	(void)newsreader_CreateConfigGUI(GTK_WIDGET(container), &newsreader->oGUI);

    newsreader->oGUI.wTopLevel = gtk_widget_get_toplevel (done);

    gtk_spin_button_set_value (GTK_SPIN_BUTTON (newsreader->oGUI.wSpnWidth),
			       newsreader->labelWidth);

	g_signal_connect (newsreader->oGUI.wSpnWidth, "value-changed",
					  G_CALLBACK (width_changed), newsreader);

	/* $B99?7;~4V(B(min) */
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (newsreader->oGUI.wSpnMin),
			       newsreader->feedUpdateMin);

	/* item$B99?7;~4V(B(sec) */
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (newsreader->oGUI.wSpnSec),
			       newsreader->itemUpdateSec);

	g_signal_connect (newsreader->oGUI.wSpnSec, "value-changed",
					  G_CALLBACK (updateSec_changed), newsreader);

	if(newsreader->strItemFontName)
		gtk_font_button_set_font_name(GTK_FONT_BUTTON(newsreader->oGUI.wFbtItemFont), newsreader->strItemFontName->str);

	if(newsreader->pchBrowserCmd)
		gtk_entry_set_text(GTK_ENTRY(newsreader->oGUI.wEntBrowserCmd), newsreader->pchBrowserCmd);

	/* $B%D!<%k%A%C%W(B */
	gtk_toggle_button_set_active((GtkToggleButton*)(newsreader->oGUI.wChkTooltips), newsreader->bUseTooltips);

	/* set feed list model data */
	set_model_data(newsreader);

	create_list_column(newsreader);

	g_signal_connect(newsreader->oGUI.wBtnListApply, "clicked", G_CALLBACK(apply_clicked), newsreader);

	g_signal_connect(newsreader->oGUI.wBtnListRemove, "clicked", G_CALLBACK(remove_clicked), newsreader);

	g_signal_connect(newsreader->oGUI.wBtnListAdd, "clicked", G_CALLBACK(add_clicked), newsreader);

	g_signal_connect(newsreader->oGUI.wBtnListUp, "clicked", G_CALLBACK(listUp_clicked), newsreader);

	g_signal_connect(newsreader->oGUI.wBtnListDown, "clicked", G_CALLBACK(listDown_clicked), newsreader);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(newsreader->oGUI.wTrFeedList));
	gtk_tree_selection_set_select_function(selection, view_selection_func, newsreader, NULL);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(newsreader->oGUI.wTrFeedList), FALSE);

    /* update settings when dialog is closed */
    g_signal_connect_swapped (done, "clicked",
			      G_CALLBACK (newsreader_apply_options), newsreader);

	/*    g_signal_connect_swapped (sd->dialog, "destroy-event",
			      G_CALLBACK (free_newsreader_dialog), sd);
	*/

	g_signal_connect(newsreader->oGUI.wEntFeedTitle, "changed", G_CALLBACK(item_changed), newsreader);
	g_signal_connect(newsreader->oGUI.wEntFeedURL, "changed", G_CALLBACK(item_changed), newsreader);
	g_signal_connect(newsreader->oGUI.wSpnUntil, "changed", G_CALLBACK(item_changed), newsreader);

	gtk_widget_set_sensitive(newsreader->oGUI.wBtnListApply, FALSE);
	gtk_widget_set_sensitive(newsreader->oGUI.wBtnListRemove, FALSE);

	return;
}

static void About (GtkWidget *w, void *unused)
	/* Called back when the About button in clicked */
{
    xfce_info ("%s %s - Newsreader \n"
	       "RSS/Atom news feed reader \n\n"
	       "(c) 2005 mueki <mueki@users.sourceforge.jp>",
	       PACKAGE, VERSION);
}				/* About() */

/* $B%]%C%W%"%C%W%a%K%e!<$N=i4|2=(B */
static GtkMenu*
create_initial_menu(newsreader_t* newsreader)
{
	GtkWidget* menu;
	GtkWidget* submenu;
	GtkWidget* itemUpdate;
	GtkWidget* itemSeparator;
	GtkWidget* itemFeedList;
	GtkWidget* itemSubMenu;
	GtkWidget* itemAbout;
	GtkWidget* itemImage;

	menu = gtk_menu_new();
	itemImage = gtk_image_new_from_stock("gtk-refresh", GTK_ICON_SIZE_MENU);
	itemUpdate = gtk_image_menu_item_new_with_mnemonic (_("_Update"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(itemUpdate), itemImage);
	gtk_widget_show(itemUpdate);
	gtk_menu_append(menu, itemUpdate);
	newsreader->wItmUpdate = GTK_MENU_ITEM(itemUpdate);
	g_signal_connect (G_OBJECT(newsreader->wItmUpdate), "activate",
					  G_CALLBACK(on_itemupdate_activate_cb), newsreader);
	
	itemSeparator = gtk_separator_menu_item_new();
	gtk_widget_show(itemSeparator);
	gtk_menu_append(menu, itemSeparator);

	itemFeedList =  gtk_menu_item_new_with_label (_("Feed List"));
	gtk_widget_show(itemFeedList);
	gtk_menu_append(menu, itemFeedList);
	newsreader->wItmFeedList = GTK_MENU_ITEM(itemFeedList);

	submenu = gtk_menu_new();
	itemSubMenu = gtk_menu_item_new_with_label(_("(no Feed)"));
	gtk_widget_show(itemSubMenu);
	gtk_menu_append(submenu, itemSubMenu);
	gtk_widget_set_sensitive(itemSubMenu, FALSE);

	itemSeparator = gtk_separator_menu_item_new();
	gtk_widget_show(itemSeparator);
	gtk_menu_append(menu, itemSeparator);


	itemImage = gtk_image_new_from_stock("gtk-dialog-info", GTK_ICON_SIZE_MENU);
	itemAbout = gtk_image_menu_item_new_with_mnemonic (_("_About..."));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(itemAbout), GTK_WIDGET(itemImage));

	gtk_widget_show(itemAbout);
	gtk_menu_append(menu, itemAbout);
	newsreader->wItmAbout = GTK_MENU_ITEM(itemAbout);
	g_signal_connect (G_OBJECT(newsreader->wItmAbout), "activate",
					  G_CALLBACK(About), newsreader);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(itemFeedList), submenu);

	return GTK_MENU(menu);
}

/* DnD$B$5$l$?(BURL$B$+$i%U%#!<%I%G!<%?$r<hF@$7$F%j%9%H$KDI2C(B */
static void 
dragedfeed_append_to_list(newsreader_t* newsreader)
{
	char *buff;
	gint bufflen;
	time_t tmLastModified;
	feed_t* pFeed = NULL;
	feeddata_t* pFeedData = NULL;
	outline_t* pOutline = NULL;

	pthread_mutex_lock(&getfeed_mutex);

	if(!newsreader->pchURI){
		DEBUG_PRINT("newsreader->pchURI is NULL");
		goto error;
	}

	if((bufflen = httpfetch(newsreader->pchURI, &buff, -1, &tmLastModified)) < 0){
		g_warning("!!! newsreader - httpfetch error");
		goto error;
	}
  
	if(!(pFeed = GetFeed(buff, bufflen))){
		g_warning("!!! newsreader - GetFeed error");
		goto error;
	}

	if(buff)
		g_free(buff);

	pFeedData = create_new_feeddata();
	pFeedData->pchURI = g_strdup(newsreader->pchURI);
	pFeedData->tmLastModified = tmLastModified;
	pFeedData->pFeed = pFeed;

	AppendIconList(newsreader, pFeedData);

	NEWSREADER_LOCK();

	pOutline = g_new0(outline_t, 1);
	pOutline->strTitle = g_string_new(pFeed->title->str);
	pOutline->strURL = g_string_new(newsreader->pchURI);
	pOutline->bAvailable = TRUE;

	newsreader->poOutlineList = g_list_prepend(newsreader->poOutlineList, pOutline);

	newsreader->g_pFeedList = g_list_insert(newsreader->g_pFeedList, pFeedData, 1);

	newsreader->g_pCurFeed = NULL;
	newsreader->g_pCurItem = NULL;

	//update_feed_list_menu(newsreader);
	update_popupmenu_register(newsreader);

	g_free(newsreader->pchURI);

	StoreFeedList2File(newsreader);

	NEWSREADER_UNLOCK();

	pthread_mutex_unlock(&getfeed_mutex);

	return;

 error:
	pthread_mutex_unlock(&getfeed_mutex);
	message_register(newsreader, "Could not Acquire RSS/Atom feed");
	return;
}

/* event handler for drag and drop */
void on_wLblItem_drag_drop( GtkWidget        *widget,
						GdkDragContext   *context,
						gint              x,
						gint              y,
						GtkSelectionData *data,
						guint             info,
						guint             time_,
						gpointer          user_data )
{

	newsreader_t* newsreader = (newsreader_t*)user_data;
	GString* strURI = g_string_new("");
	gint len;
	guchar *p;
	gchar* pchURI = NULL;
	gchar* pchData = NULL;
	gchar** ppchSplit = NULL;
	pthread_t thread;

	if ((data->length <= 0) || (data->format != 8)) {
		g_warning("drag have no data");
		goto error;
	}

	len = data->length;
	p = data->data;

	switch(info){
	case DROP_TEXT_X_MOZ_URL:
		pchData = g_utf16_to_utf8((const gunichar2*)p, (gint)(len /2), NULL, NULL, NULL);
		if(pchData)
			ppchSplit = g_strsplit(pchData, "\n", 0);
		if(ppchSplit)
			pchURI = g_strdup(ppchSplit[0]);

		if(ppchSplit)g_strfreev(ppchSplit);
		if(pchData)g_free(pchData);
		break;
	default:
		/* FIX ME */
		while (len--) {
			if (0x0a == *p)break;

			if (*p != 0) {
				g_string_append_c(strURI, *p);
			}
			p++;
		}
		pchURI = g_strdup(strURI->str);
		g_string_free(strURI, TRUE);
		break;
	}
	
	DEBUG_PRINT("newsreader - drag and get");
	DEBUG_PRINT("newsreader - fetch URI: \"%s\"", pchURI);

	newsreader->pchURI = g_strdup(pchURI);
	g_free(pchURI);

	gtk_drag_finish(context, TRUE, TRUE, time_);

	pthread_create(&thread, NULL, (void*)dragedfeed_append_to_list, (void*)newsreader);
	pthread_detach(thread);

	return;

 error:
	gtk_drag_finish(context, FALSE, TRUE, time_);
	message_register(newsreader, "Could not Acquire RSS/Atom feed");
	return;
}

/**
 * create_newsreader_control
 *
 * Create a new instance of the plugin.
 * 
 * @control : #Control parent container
 *
 * Returns %TRUE on success, %FALSE on failure.
 **/
static gboolean
create_newsreader_control (Control * control)
{
    newsreader_t *newsreader = newsreader_new ();

    gtk_container_add (GTK_CONTAINER (control->base), newsreader->widget);

	newsreader->wMenu = create_initial_menu(newsreader);

    control->data = newsreader;

    newsreader->control = control;

	/* $B%i%Y%k%5%$%:$ND4@0(B */
	//	gtk_widget_set_size_request(GTK_WIDGET(newsreader->wLblItem), newsreader->labelWidth, -1);
	//    gtk_widget_set_size_request(newsreader->widget, newsreader->labelWidth, -1);
    gtk_widget_set_size_request(newsreader->control->base, newsreader->labelWidth, -1);

	g_signal_connect (G_OBJECT(newsreader->wEvItem), "button_release_event",
					  G_CALLBACK (on_button_release_cb), newsreader);

	g_signal_connect (G_OBJECT(newsreader->wEvItem), "enter_notify_event",
					  G_CALLBACK (on_enter_notify_cb), newsreader);

	g_signal_connect (G_OBJECT(newsreader->wEvItem), "leave_notify_event",
					  G_CALLBACK (on_leave_notify_cb), newsreader);
	
	g_signal_connect_swapped (G_OBJECT(newsreader->wBtnMenu), "button_press_event",
							  G_CALLBACK (on_menubutton_press_cb), newsreader);

	g_signal_connect (G_OBJECT(newsreader->wEvItem), "event",
					   G_CALLBACK (on_button_press_cb), newsreader);
	
    gtk_widget_set_size_request (control->base, -1, -1);
	
	if(newsreader->strItemFontName){
		g_string_free(newsreader->strItemFontName, TRUE);
	}
	
	newsreader->strItemFontName = g_string_new(pango_font_description_to_string(pango_context_get_font_description(gtk_widget_create_pango_context(GTK_WIDGET(newsreader->wLblItem)))));

	newsreader->g_pFeedList = NULL;
	newsreader->g_pFeedList = g_list_append(newsreader->g_pFeedList, create_static_feed());
	newsreader->g_pCurFeed = NULL;
	newsreader->g_pCurItem = NULL;

	/* drag and drop setting */
	gtk_drag_dest_set(GTK_WIDGET(newsreader->wLblItem), (GtkDestDefaults) (GTK_DEST_DEFAULT_ALL), drop_types, n_drop_types, GDK_ACTION_COPY);
	gtk_signal_connect(GTK_OBJECT(newsreader->wLblItem), "drag_data_received", GTK_SIGNAL_FUNC(on_wLblItem_drag_drop), newsreader);

	SetUpdateFeedTimer( newsreader );
	SetTimer( newsreader );
	
    return TRUE;

}

void
newsreader_set_size (Control *control, int size)
{
	newsreader_t* newsreader = (newsreader_t*)control->data;

	gtk_widget_set_size_request(control->base, newsreader->labelWidth, -1);	
}

/**
 * xfce_control_class_init
 *
 * Ideally, this should be the only exported symbol in the plugin, since this
 * is the only function that is directly accessed from the panel.
 *
 * Here you set up the virtual function table for the plugin and specify its
 * behaviour (e.g. uniqueness).
 * 
 * @cc : #ControlClass to initialize
 **/
G_MODULE_EXPORT void
xfce_control_class_init (ControlClass * cc)
{
    xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

    /* Must be present: 
       - name            : unique id 
       - caption         : display name 
       - create_control  : create a new instance
       - attach_callback : connect a signal (e.g. right-click menu)
       - free            : free allocated resources of Control instance
	*/
    cc->name = "Newsreader";
    cc->caption = "Newsreader";

    cc->create_control = (CreateControlFunc) create_newsreader_control;

    cc->attach_callback = newsreader_attach_callback;
    cc->free = newsreader_free;

    /* Optional, leave as NULL to get default behaviour
       - read_config     : read configuration from xml
       - write_config    : write configuration to xml
       - create_options  : create widgets for the options dialog
       - set_size        : set size (SMALL, NORMAL, LARGE or HUGE)
       - set_orientation : set orientation (HORIZONTAL or VERTICAL)
       - set_theme       : set icon theme
       - about           : show an about dialog
	*/
    cc->read_config = newsreader_read_config;
    cc->write_config = newsreader_write_config;

    cc->create_options = newsreader_create_options;

    cc->set_theme = newsreader_set_theme;

    cc->set_size = newsreader_set_size;
    /* cc->set_orientation = NULL;
     * cc->about = NULL;
     */

    /* Additional API calls */

    /* use if there should be only one instance per screen */
    control_class_set_unique (cc, TRUE);

    /* use if the gmodule should not be unloaded *
     * (usually because of library issues)       */
    control_class_set_unloadable (cc, FALSE);

    /* use to set an icon to represent the module        *
     * (you could even update it when the theme changes) */
	/*    if (1)
		  {
		  GdkPixbuf *pixbuf;

		  pixbuf = xfce_icon_theme_load (xfce_icon_theme_get_for_screen (NULL),
		  "sampleicon.png", 48);

		  if (pixbuf)
		  {
		  control_class_set_icon (cc, pixbuf);
		  g_object_unref (pixbuf);
		  }
		  }
	*/

	/* thread initialized */
	/*	if (!g_thread_supported ()){
		g_thread_init (NULL);
		gdk_threads_init();
		}*/
	
}

/* Macro that checks panel API version */
XFCE_PLUGIN_CHECK_INIT
