/*
 * Created on 2004/04/01
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.messageformat;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import netjfwatcher.snmp.preference.SnmpBERCodec;
import netjfwatcher.snmp.snmpobject.integer.SnmpInteger;
import netjfwatcher.snmp.snmpobject.integer.SnmpTimeTicks;
import netjfwatcher.snmp.snmpobject.message.AbstractSnmpObject;
import netjfwatcher.snmp.snmpobject.message.SnmpObjectIdentifier;
import netjfwatcher.snmp.snmpobject.message.SnmpSequence;
import netjfwatcher.snmp.snmpobject.octetstring.SnmpIPAddress;



/**
 * The SNMPTrapPDU class represents an SNMP Trap PDU from RFC 1157, as indicated below. This
 * forms the payload of an SNMP Trap message.
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpTrapPDU extends SnmpSequence {
    /* Contentsłenterprise OIDʒu */
    private static final int ENTERPRISE_OID_POS = 0;

    /* ContentsłAgent Addressʒu */
    private static final int AGNET_ADDRESS_POS = 1;

    /* ContentsłGeneric TrapR[hʒu */
    private static final int GENERIC_TRAP_CODE_POS = 2;

    /* ContentsłSpecific TrapR[hʒu */
    private static final int SPECIFIC_TRAP_CODE_POS = 3;

    /* ContentsłTimestampʒu */
    private static final int TIMESTAMP_POS = 4;

    /* Contentsłvariable binding listʒu */
    private static final int VA_BIND_LIST_POS = 5;

    /**
     * Create a new Trap PDU of the specified type, with given request ID,
     * error status, and error index,
     * and containing the supplied SNMP sequence as data.
     *
     * @param enterpriseOID Enterprise OID
     * @param agentAddress AgnetAhX
     * @param genericTrap Generic Trap
     * @param specificTrap Specific Trap
     * @param timestamp Timestamp
     * @param varList varList
     * @throws SnmpBadValueException
     */
    public SnmpTrapPDU(
        SnmpObjectIdentifier enterpriseOID, SnmpIPAddress agentAddress,
        int genericTrap, int specificTrap, SnmpTimeTicks timestamp,
        SnmpSequence varList) throws SnmpBadValueException {
        super();

        tag = SnmpBERCodec.SNMPTRAP;

        List contents = Collections.synchronizedList(new ArrayList());

        contents.add(enterpriseOID);
        contents.add(agentAddress);
        contents.add(new SnmpInteger(genericTrap));
        contents.add(new SnmpInteger(specificTrap));
        contents.add(timestamp);
        contents.add(varList);

        this.setValue(contents);
    }

    /**
     * Create a new Trap PDU of the specified type, with given request ID,
     * error status, and error index,
     * and containing an empty SNMP sequence (VarBindList) as additional data.
     *
     * @param enterpriseOID Enterprise OID
     * @param agentAddress AgentAhX
     * @param genericTrap Generic Trap
     * @param specificTrap Specific Trap
     * @param timestamp Timestamp
     * @throws SnmpBadValueException
     */
    public SnmpTrapPDU(
        SnmpObjectIdentifier enterpriseOID, SnmpIPAddress agentAddress,
        int genericTrap, int specificTrap, SnmpTimeTicks timestamp)
        throws SnmpBadValueException {
        super();

        tag = SnmpBERCodec.SNMPTRAP;

        List contents = Collections.synchronizedList(new ArrayList());

        contents.add(enterpriseOID);
        contents.add(agentAddress);
        contents.add(new SnmpInteger(genericTrap));
        contents.add(new SnmpInteger(specificTrap));
        contents.add(timestamp);
        contents.add(new SnmpVarBindList());

        this.setValue(contents);
    }

    /**
     * Create a new PDU of the specified type from the supplied BER encoding.
     * @param enc
     * @throws SnmpBadValueException Indicates invalid SNMP PDU encoding supplied in enc.
     */
    public SnmpTrapPDU(byte[] enc) throws SnmpBadValueException {
        tag = SnmpBERCodec.SNMPTRAP;
        extractFromBEREncoding(enc);
    }

    /**
     * PDU TypeԂ܂B
     *
     * @return tag PDU Type
     */
    public byte getPDUType() {
        return tag;
    }

    /**
     * A utility method that extracts the variable binding list from the pdu. Useful for retrieving
     * the set of (object identifier, value) pairs returned in response to a request to an SNMP
     * device. The variable binding list is just an SNMP sequence containing the identifier,
     * value pairs.
     * @see snmp.SnmpVarBindList
     * @return SnmpVarBindList
     */
    public SnmpSequence getVarBindList() {
        List contents = (List) (this.getValue());

        if (contents.get(VA_BIND_LIST_POS) instanceof SnmpSequence) {
            return (SnmpSequence) (contents.get(VA_BIND_LIST_POS));
        }

        return null;
    }

    /**
     * A utility method that extracts the enterprise OID from this PDU.
     *
     * @return Enterprise OID
     */
    public SnmpObjectIdentifier getEnterpriseOID() {
        List contents = (List) (this.getValue());

        if (contents.get(ENTERPRISE_OID_POS) instanceof SnmpObjectIdentifier) {
            return (SnmpObjectIdentifier) contents.get(ENTERPRISE_OID_POS);
        }

        return null;
    }

    /**
     * G[WFgAddressԂ܂B
     *
     * @return G[WFgAddress
     */
    public SnmpIPAddress getAgentAddress() {
        List contents = (List) (this.getValue());

        if (contents.get(AGNET_ADDRESS_POS) instanceof SnmpIPAddress) {
            return (SnmpIPAddress) contents.get(AGNET_ADDRESS_POS);
        }

        return null;
    }

    /**
     * A utility method that returns the generic trap code for this PDU.
     *
     * @return Generic TrapR[h
     */
    public int getGenericTrap() {
        List contents = (List) (this.getValue());

        return ((BigInteger) ((SnmpInteger) (contents.get(
            GENERIC_TRAP_CODE_POS))).getValue()).intValue();
    }

    /**
     * A utility method that returns the specific trap code for this PDU.
     *
     * @return Specific TrapR[h
     */
    public int getSpecificTrap() {
        List contents = (List) (this.getValue());

        return ((BigInteger) ((SnmpInteger) (contents.get(
            SPECIFIC_TRAP_CODE_POS))).getValue()).intValue();
    }

    /**
     * A utility method that returns the timestamp for this PDU.
     *
     * @return Time stamp
     */
    public long getTimestampData() {
        List contents = (List) (this.getValue());

        if (contents.get(4) instanceof SnmpTimeTicks) {
            return ((BigInteger) ((SnmpTimeTicks) (contents.get(TIMESTAMP_POS)))
            .getValue()).longValue();
        }

        return 0;
    }

    /**
     * TimestampԂ܂B
     *
     * @return Time stamp
     */
    public AbstractSnmpObject getTimestamp() {
        List contents = (List) (this.getValue());

        if (contents.get(TIMESTAMP_POS) instanceof AbstractSnmpObject) {
            return (AbstractSnmpObject) contents.get(TIMESTAMP_POS);
        }

        return null;
    }
}
