using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Reflection;
using NDac.Datas;
using NDac.Keys;
using NDac.Modules.DataTransferObjects;
using NDac.Modules.Entities;
using NDac.Modules.Extensions;

namespace NDac.Modules
{
	/// <summary>
	/// テーブルモジュールの抽象クラスを表します。
	/// </summary>
	public abstract partial class TableModule : ITableModule< IKey, IEntity, IDataTransferObject >
	{
		private DataTable						_tableReal;
		private LazyInitializer					_initializer;
		protected string						_tableName;
		protected Dictionary< IKey, DataRow >	_cache;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="table">データテーブル</param>
		public TableModule( DataTable table )
		{
			this._table	= table;

			this._tableName = this._table.TableName;

			this._cache	= new Dictionary< IKey, DataRow >();
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="ds">データセット</param>
		/// <param name="tableName">テーブル名</param>
		public TableModule( DataSet ds, string tableName )
		{
			this._table	= ds.Tables[ tableName ];

			this._tableName = tableName;

			this._cache	= new Dictionary< IKey, DataRow >();

			if( this._table == null )
			{
				this._initializer = new LazyInitializer( ds, tableName );
			}
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <param name="tableName">テーブル名</param>
		public TableModule( DataSetHolder holder, string tableName )
		{
			this._table	= holder[ tableName ];

			this._tableName = tableName;

			this._cache	= new Dictionary< IKey, DataRow >();

			if( this._table == null )
			{
				this._initializer = new LazyInitializer( holder, tableName );
			}
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		protected abstract void Set( IDataTransferObject dto );

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <typeparam name="T">エンティティの型</typeparam>
		/// <param name="row">データ行</param>
		/// <returns>エンティティ</returns>
		protected internal abstract T ToEntity< T >( DataRow row ) where T : class, IEntity;

		/// <summary>
		/// SELECT実行時のフィルタ処理式に変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>フィルタ処理式</returns>
		protected abstract string ToFilterExpression( IKey key );

		/// <summary>
		/// データテーブルを表します。(DataSetにテーブルが充填されていない場合は、レイジーロードされます。)
		/// </summary>
		protected DataTable _table
		{
			get
			{
				if( ( ( this._tableReal == null ) || ( this._tableReal.Rows.Count == 0 ) ) && ( this._initializer != null ) )
				{
					this._tableReal = this._initializer.Initialize();

					if( ( this._tableReal != null ) && ( this._tableReal.Site == null ) )
					{
						this._tableReal.Site = new DataTableSite( this._tableReal );
					}
				}

				return( this._tableReal );
			}
			set
			{
				this._tableReal = value;

				if( ( this._tableReal != null ) && ( this._tableReal.Site == null ) )
				{
					this._tableReal.Site = new DataTableSite( this._tableReal );
				}
			}
		}

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="key">キー</param>
		protected void Delete( IKey key )
		{
			DataRow row = this.GetRow( key );

			if( this._cache.ContainsKey( key ) )
			{
				this._cache.Remove( key );
			}

			if( row != null )
			{
				row.Delete();
			}
		}

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>データ行</returns>
		protected DataRow GetRow( IKey key )
		{
			if( this._cache.ContainsKey( key ) )
			{
				return( this._cache[ key ] );
			}
			else
			{
				DataRow[] matchRows = this._table.Select( this.ToFilterExpression( key ) );

				if( 0 < matchRows.Length )
				{
					this._cache.Add( key, matchRows[ 0 ] );

					return( matchRows[ 0 ] );
				}
				else
				{
					return( null );
				}
			}
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		protected void Insert( IDataTransferObject dto )
		{
			if( !dto.HasEntity() && dto.IsNew() )
			{
				DataRow row = this.NewRow();

				( ( DataTransferObject )dto ).BindNewEntity( this.ToEntity< IEntity >( row ) );

				this._table.Rows.Add( row );
			}
		}

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		protected bool IsDeleted( IKey key )
		{
			return( 0 < this._table.GetDeleted().Select( this.ToFilterExpression( key ) ).Length );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>新しい行のエンティティ</returns>
		protected IEntity NewEntity( IKey key )
		{
			try
			{
				BindingFlags flag = BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance;

				MethodInfo method = this.GetType().GetMethod( "NewEntity", flag, null, new Type[]{ key.GetType() }, null );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );

					return( ( IEntity )returnValue );
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするNewEntityメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		/// <summary>
		/// 新しいデータ行を作成します。作成されたデータ行はデフォルト値が登録されます。
		/// </summary>
		/// <returns>デフォルト値のデータ行</returns>
		protected DataRow NewRow()
		{
			DataRow row = this._table.NewRow();

			foreach( DataColumn column in this._table.Columns )
			{
				row[ column ] = column.DefaultValue;

				if( ( row[ column ] == Convert.DBNull ) && ( !column.AllowDBNull ) )
				{
					row[ column ] = this.ToDefaultValue( column.DataType );
				}
			}

			return( row );
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		protected void Remove( IEntity entity )
		{
			if( this._cache.ContainsKey( entity.GetPrimaryKey() ) )
			{
				this._cache.Remove( entity.GetPrimaryKey() );
			}

			this._table.Rows.Remove( entity.Row );
		}

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="key">キー</param>
		protected void RemoveBy( IKey key )
		{
			foreach( IEntity entity in ( ( ITableModule )this ).SelectBy( key ) )
			{
				this.Remove( entity );
			}
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <returns>エンティティ配列</returns>
		public IEntity[] Select( string filterExpression, string sort )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression, sort ) ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		public IEntity[] Select( string filterExpression, string sort, FilteringCondition filteringCondition )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression, sort ), filteringCondition ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのデータ行配列を取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>データ行配列</returns>
		protected DataRow[] SelectDataRows( string filterExpression )
		{
			string[] filter = this.SplitOrderBy( filterExpression );

			if( filter != null )
			{
				return( this._table.Select( filter[ 0 ], filter[ 1 ] ) );
			}
			else
			{
				return( this._table.Select( filterExpression ) );
			}
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのデータ行配列を取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <returns>データ行配列</returns>
		protected DataRow[] SelectDataRows( string filterExpression, string sort )
		{
			return( this._table.Select( filterExpression, sort ) );
		}

		/// <summary>
		/// フィルタ処理式をWHERE句とORDER BY句に分割します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>ORDER BY句が含まれている場合は分割された文字列の配列が返ります。</returns>
		private string[] SplitOrderBy( string filterExpression )
		{
			const string ORDER		= "ORDER";
			const string BY			= "BY";
			string		upperFilter	= filterExpression.ToUpper();
			string[]	result		= new string[ 2 ];
			int			orderStart	= 0;
			int			byEnd		= 0;

			if( 0 <= ( orderStart = upperFilter.IndexOf( ORDER ) ) )
			{
				int byFindStart		= orderStart + ORDER.Length;

				int byFindLength	= upperFilter.Length - ( orderStart + ORDER.Length );

				string byString		= upperFilter.Substring( byFindStart, byFindLength ).Trim();

				if( byString.StartsWith( BY ) )
				{
					byEnd = upperFilter.IndexOf( BY, orderStart ) + 1;

					result[ 0 ] = filterExpression.Substring( 0, orderStart ).Trim();

					result[ 1 ] = filterExpression.Substring( byEnd + 1, filterExpression.Length - ( byEnd + 1 ) ).Trim();

					return( result );
				}
			}

			return( null );
		}

		/// <summary>
		/// デフォルト値に変換します。
		/// </summary>
		/// <param name="type">データタイプ</param>
		/// <returns>デフォルト値</returns>
		protected object ToDefaultValue( Type type )
		{
			if( type == typeof( string ) )
			{
				return( string.Empty );
			}
			else if( type == typeof( DateTime ) )
			{
				return( new DateTime( 1900, 1, 1 ) );
			}
			else if( type == typeof( bool ) )
			{
				return( false );
			}
			else if( type == typeof( char ) )
			{
				return( ' ' );
			}
			else
			{
				return( 0 );
			}
		}

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティ</returns>
		protected IEntity ToEntity( IKey key )
		{
			return( this.ToEntity< IEntity >( this.GetRow( key ) ) );
		}

		/// <summary>
		/// データ行配列をエンティティ配列に変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <returns>エンティティ配列</returns>
		private IEntity[] ToEntityArray( DataRow[] rows )
		{
			return( this.ToEntityArray( rows, FilteringCondition.Default ) );
		}

		/// <summary>
		/// データ行配列をエンティティ配列に変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		private IEntity[] ToEntityArray( DataRow[] rows, FilteringCondition filteringCondition )
		{
			return( ( IEntity[] )this.ToEntityList( rows, filteringCondition ).ToArray() );
		}

		/// <summary>
		/// データ行配列をエンティティリストに変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		private List< IEntity > ToEntityList( DataRow[] rows, FilteringCondition filteringCondition )
		{
			List< IEntity > entities = new List< IEntity >();

			foreach( DataRow row in rows )
			{
				if( filteringCondition.CanConvert( row ) )
				{
					entities.Add( this.ToEntity< IEntity >( row ) ); 
				}
			}

			return( entities );
		}

		/// <summary>
		/// データセットにテーブルスキーマを充填します。
		/// </summary>
		/// <typeparam name="TEntity">エンティティの型</typeparam>
		/// <param name="tableName">テーブル名</param>
		/// <param name="ds">データセット</param>
		protected static void FillSchema< TEntity >( string tableName, DataSet ds )
		{
			if( !ds.Tables.Contains( tableName ) )
			{
				ds.Tables.Add( tableName );
			}

			FillSchema< TEntity >( tableName, ds.Tables[ tableName ] );
		}

		/// <summary>
		/// データテーブルにテーブルスキーマを充填します。
		/// </summary>
		/// <typeparam name="TEntity">エンティティの型</typeparam>
		/// <param name="tableName">テーブル名</param>
		/// <param name="table">データテーブル</param>
		protected static void FillSchema< TEntity >( string tableName, DataTable table )
		{
			table.TableName = tableName;

			foreach( PropertyInfo info in typeof( TEntity ).GetProperties() )
			{
				if( ( info.GetIndexParameters().Length == 0 ) && ( !table.Columns.Contains( info.Name ) ) )
				{
					if( IsNullableGeneric( info.PropertyType ) )
					{
						table.Columns.Add( info.Name, Nullable.GetUnderlyingType( info.PropertyType ) );
					}
					else
					{
						table.Columns.Add( info.Name, info.PropertyType );
					}
				}
			}
		}

		/// <summary>
		/// データ型がNullableジェネリック型であるか判定します。
		/// </summary>
		/// <param name="type">データ型</param>
		/// <returns>データ型がNullableジェネリック型である場合trueが返ります。</returns>
		private static bool IsNullableGeneric( Type type )
		{
			return( ( type.IsGenericType ) && ( type.GetGenericTypeDefinition() == typeof( Nullable<> ) ) );
		}

		#region ITableModule<IKey,IEntity,IDataTransferObject> メンバ

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		public IEntity this[ IKey primaryKey ]
		{
			get
			{
				return( this.ToEntity( primaryKey ) );
			}
		}

		/// <summary>
		/// テーブルモジュールにエンティティを追加します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public void Add( IEntity entity )
		{
			this._table.ImportRow( entity.Row );
		}

		/// <summary>
		/// テーブルモジュールにエンティティが存在するかを判断します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <returns>テーブルモジュールにエンティティが存在する場合はtrueが返ります。</returns>
		public bool Contains( IEntity entity )
		{
			return( this._table.Rows.Contains( entity.Row ) );
		}

		/// <summary>
		/// テーブルモジュール内のエンティティをエンティティの配列にコピーします。
		/// </summary>
		/// <param name="entities">エンティティの配列</param>
		/// <param name="index">配列へコピー開始するインデックス</param>
		public void CopyTo( Array entities, int index )
		{
			foreach( IEntity entity in this )
			{
				entities.SetValue( entity, index );

				index++;
			}
		}

		/// <summary>
		/// テーブルモジュール内のエンティティをエンティティの配列にコピーします。
		/// </summary>
		/// <param name="entities">エンティティの配列</param>
		/// <param name="arrayIndex">配列へコピー開始するインデックス</param>
		public void CopyTo( IEntity[] entities, int arrayIndex )
		{
			foreach( IEntity entity in this )
			{
				entities[ arrayIndex ] = entity;

				arrayIndex++;
			}
		}

		/// <summary>
		/// データ行を削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.Delete( IKey primaryKey )
		{
			this.Delete( primaryKey );
		}

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>データ行</returns>
		DataRow ITableModule< IKey, IEntity, IDataTransferObject >.GetRow( IKey primaryKey )
		{
			return( this.GetRow( primaryKey ) );
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.Insert( IDataTransferObject dto )
		{
			this.Insert( dto );
		}

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		bool ITableModule< IKey, IEntity, IDataTransferObject >.IsDeleted( IKey primaryKey )
		{
			return( this.IsDeleted( primaryKey ) );
		}

		/// <summary>
		/// 指定したテーブルモジュールの内容を結合します。
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.Merge( ITableModule<IKey, IEntity, IDataTransferObject> module )
		{
			this._table.Merge( module.Table );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		public IEntity NewEntity()
		{
			DataRow row = this.NewRow();

			this._table.Rows.Add( row );

			return( this.ToEntity< IEntity >( row ) );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>新しい行のエンティティ</returns>
		IEntity ITableModule< IKey, IEntity, IDataTransferObject >.NewEntity( IKey primaryKey )
		{
			return( this.NewEntity( primaryKey ) );
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.Remove( IEntity entity )
		{
			this.Remove( entity );
		}

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.RemoveBy( IKey primaryKey )
		{
			this.RemoveBy( primaryKey );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式　ORDER BY句を続けて記述可能です。</param>
		/// <returns>エンティティ配列</returns>
		public IEntity[] Select( string filterExpression )
		{
			return( this.Select( filterExpression, FilteringCondition.Default ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式　ORDER BY句を続けて記述可能です。</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		public IEntity[] Select( string filterExpression, FilteringCondition filteringCondition )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression ), filteringCondition ) );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		List< IEntity > ITableModule< IKey, IEntity, IDataTransferObject >.SelectBy( IKey key )
		{
			try
			{
				BindingFlags flag = BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance;

				MethodInfo method = this.GetType().GetMethod( "SelectBy", flag, null, new Type[]{ key.GetType() }, null );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );

					if( returnValue.GetType().GetInterface( typeof( IList ).ToString() ) != null )
					{
						IList returnList = ( IList )returnValue;

						List< IEntity > entities = new List< IEntity >();

						foreach( object entity in returnList )
						{
							entities.Add( entity as IEntity );
						}

						return( entities );
					}
					else
					{
						throw( new MissingMethodException( string.Format( "{0}をパラメータとし、{1}を戻り値としするSelectByメソッドは存在しません。", key.GetType().ToString(), typeof( List< IEntity > ).ToString() ) ) );
					}
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするSelectByメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void ITableModule< IKey, IEntity, IDataTransferObject >.Set( IDataTransferObject dto )
		{
			this.Set( dto );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public List< IEntity > ToEntities()
		{
			return( this.ToEntities( FilteringCondition.Default ) );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public List< IEntity > ToEntities( FilteringCondition filteringCondition )
		{
			List< IEntity > entities = new List< IEntity >();

			foreach( IEntity entity in new TableModuleEnumerable( this, filteringCondition ) )
			{
				entities.Add( entity );
			}

			return( entities );
		}

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		IEntity ITableModule< IKey, IEntity, IDataTransferObject >.ToEntity( IKey primaryKey )
		{
			return( this.ToEntity( primaryKey ) );
		}

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public IEntity ToEntity( IndexNumber index )
		{
			if( index == IndexNumber.Last )
			{
				if( 0 < this.Count )
				{
					return( this.ToEntity< IEntity >( this.GetRow( this.Count - 1 ) ) );
				}
				else
				{
					return( null );
				}
			}
			else
			{
				return( this.ToEntity< IEntity >( this.GetRow( index.Number ) ) );
			}
		}

		#endregion

		#region ITableModule メンバ

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public IEntity this[ IndexNumber index ]
		{
			get
			{
				return( this.ToEntity( index ) );
			}
		}

		/// <summary>
		/// データが変更されているか否かを表します。
		/// </summary>
		public bool IsModified
		{
			get
			{
				return( this._table.GetChanges() != null );
			}
		}

		/// <summary>
		/// データテーブルを表します。
		/// </summary>
		public DataTable Table
		{
			get
			{
				return( this._table );
			}
		}

		/// <summary>
		/// テーブル名を表します。
		/// </summary>
		public string TableName
		{
			get
			{
				return( this._tableName );
			}
		}

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableModule.Delete( IKey key )
		{
			this.Delete( key );
		}

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>データ行</returns>
		public DataRow GetRow( int index )
		{
			if( index < this._table.Rows.Count )
			{
				return( this._table.Rows[ index ] );
			}
			else
			{
				return( null );
			}
		}

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>データ行</returns>
		DataRow ITableModule.GetRow( IKey key )
		{
			return( this.GetRow( key ) );
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void ITableModule.Insert( IDataTransferObject dto )
		{
			this.Insert( dto );
		}

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		bool ITableModule.IsDeleted( IKey key )
		{
			return( this.IsDeleted( key ) );
		}

		/// <summary>
		/// 指定したテーブルモジュールの内容を結合します。
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		void ITableModule.Merge( ITableModule module )
		{
			this._table.Merge( module.Table );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>新しい行のエンティティ</returns>
		IEntity ITableModule.NewEntity( IKey key )
		{
			return( this.NewEntity( key ) );
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		void ITableModule.Remove( IEntity entity )
		{
			this.Remove( entity );
		}

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableModule.RemoveBy( IKey key )
		{
			this.RemoveBy( key );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		List<IEntity> ITableModule.SelectBy( IKey key )
		{
			return( ( ( ITableModule< IKey, IEntity, IDataTransferObject > )this ).SelectBy( key ) );
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void ITableModule.Set( IDataTransferObject dto )
		{
			this.Set( dto );
		}

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティ</returns>
		IEntity ITableModule.ToEntity( IKey key )
		{
			return( this.ToEntity( key ) );
		}

		/// <summary>
		/// エンティティリストに変換します。(主にDataGridViewのDataSourceにバインドする時に使用して下さい。)
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public ArrayList ToEntityArrayList()
		{
			return( this.ToEntityArrayList( FilteringCondition.Default ) );
		}

		/// <summary>
		/// エンティティリストに変換します。(主にDataGridViewのDataSourceにバインドする時に使用して下さい。)
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		public ArrayList ToEntityArrayList( FilteringCondition filteringCondition )
		{
			ArrayList entities = new ArrayList();

			foreach( IEntity entity in new TableModuleEnumerable( this, filteringCondition ) )
			{
				entities.Add( entity );
			}

			return( entities );
		}

		#endregion

		#region ICollection<IEntity> メンバ

		/// <summary>
		/// テーブルモジュール内の全てのエンティティを消去します。
		/// </summary>
		public void Clear()
		{
			this._table.Rows.Clear();
		}

		/// <summary>
		/// テーブルモジュールが読み取り専用かを表します。
		/// </summary>
		bool ICollection< IEntity >.IsReadOnly
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="item">削除するエンティティ</param>
		/// <returns>削除が成功した場合はtrueが返ります。</returns>
		bool ICollection< IEntity >.Remove( IEntity item )
		{
			if( this.Contains( item ) )
			{
				this.Remove( item );

				return( true );
			}
			else
			{
				return( true );
			}
		}

		#endregion

		#region ICollection メンバ

		/// <summary>
		/// 保持しているエンティティ数（レコード数）を表します。
		/// </summary>
		public int Count
		{
			get
			{
				if( this._tableReal != null )
				{
					return( this._table.Rows.Count );
				}
				else
				{
					try
					{
						if( this._table == null )
						{
							return( 0 );
						}
						else
						{
							return( this._table.Rows.Count );
						}
					}
					catch
					{
						return( 0 );
					}
				}
			}
		}

		/// <summary>
		/// コレクションがスレッドセーフであるかを表します。
		/// </summary>
		bool ICollection.IsSynchronized
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// 同期処理用のインスタンスを表します。
		/// </summary>
		object ICollection.SyncRoot
		{
			get
			{
				return( this );
			}
		}

		#endregion

		#region IEnumerable<IEntity> メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		public IEnumerator< IEntity > GetEnumerator()
		{
			return( new TableModuleEnumerator( this ) );
		}

		#endregion

		#region IEnumerable メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		IEnumerator IEnumerable.GetEnumerator()
		{
			return( this.GetEnumerator() );
		}

		#endregion
	}

	/// <summary>
	/// テーブルモジュールの抽象クラスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	/// <typeparam name="TDataTransferObject">データ変換オブジェクトの型</typeparam>
	public abstract partial class TableModule< TKey, TEntity, TDataTransferObject > : TableModule, ITableModule< TKey, TEntity, TDataTransferObject >, IEnumerable< TEntity >
		where TKey : class, IKey
		where TEntity : class, IEntity
		where TDataTransferObject : class, IDataTransferObject
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="table">データテーブル</param>
		public TableModule( DataTable table ) : base( table )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="ds">データセット</param>
		/// <param name="tableName">テーブル名</param>
		public TableModule( DataSet ds, string tableName ) : base( ds, tableName )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <param name="tableName">テーブル名</param>
		public TableModule( DataSetHolder holder, string tableName ) : base( holder, tableName )
		{
		}

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <param name="row">データ行</param>
		/// <returns>エンティティ</returns>
		protected internal abstract TEntity ToEntity( DataRow row );

		/// <summary>
		/// SELECT実行時のフィルタ処理式に変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>フィルタ処理式</returns>
		protected abstract string ToFilterExpression( TKey key );

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>新しい行のエンティティ</returns>
		protected abstract TEntity NewEntity( TKey primaryKey );

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public abstract void Set( TDataTransferObject dto );

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <returns>エンティティ配列</returns>
		public new TEntity[] Select( string filterExpression, string sort )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression ) ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		public new TEntity[] Select( string filterExpression, string sort, FilteringCondition filteringCondition )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression, sort ), filteringCondition ) );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティリスト</returns>
		public List< TEntity > SelectBy( TKey primaryKey )
		{
			return( this.SelectByCommand( this.ToFilterExpression( primaryKey ) ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式　ORDER BY句を続けて記述可能です。</param>
		/// <returns>エンティティリスト</returns>
		protected List< TEntity > SelectByCommand( string filterExpression )
		{
			return( this.ToEntityList( this.SelectDataRows( filterExpression ), FilteringCondition.Default ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="sort">列と並べ替え方向の指定</param>
		/// <returns>エンティティリスト</returns>
		protected List< TEntity > SelectByCommand( string filterExpression, string sort )
		{
			return( this.ToEntityList( this.SelectDataRows( filterExpression, sort ), FilteringCondition.Default ) );
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		protected override void Set( IDataTransferObject dto )
		{
			this.Set( ( TDataTransferObject )dto );
		}

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <typeparam name="T">エンティティの型</typeparam>
		/// <param name="row">データ行</param>
		/// <returns>エンティティ</returns>
		protected internal override T ToEntity< T >( DataRow row )
		{
			IEntity entity = this.ToEntity( row );

			return( ( T )entity );
		}

		/// <summary>
		/// データ行配列をエンティティ配列に変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		private TEntity[] ToEntityArray( DataRow[] rows )
		{
			return( this.ToEntityArray( rows, FilteringCondition.Default ) );
		}

		/// <summary>
		/// データ行配列をエンティティ配列に変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		private TEntity[] ToEntityArray( DataRow[] rows, FilteringCondition filteringCondition )
		{
			return( this.ToEntityList( rows, filteringCondition ).ToArray() );
		}

		/// <summary>
		/// データ行配列をエンティティリストに変換します。
		/// </summary>
		/// <param name="rows">データ行配列</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		private List< TEntity > ToEntityList( DataRow[] rows, FilteringCondition filteringCondition )
		{
			List< TEntity > entities = new List< TEntity >();

			foreach( DataRow row in rows )
			{
				if( filteringCondition.CanConvert( row ) )
				{
					entities.Add( this.ToEntity( row ) ); 
				}
			}

			return( entities );
		}

		/// <summary>
		/// SELECT実行時のフィルタ処理式に変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>フィルタ処理式</returns>
		protected override string ToFilterExpression( IKey key )
		{
			try
			{
				BindingFlags flag = BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance;

				MethodInfo method = this.GetType().GetMethod( "ToFilterExpression", flag, null, new Type[]{ key.GetType() }, null );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );

					if( returnValue is string )
					{
						return( ( string )returnValue );
					}
					else
					{
						throw( new MissingMethodException( string.Format( "{0}をパラメータとし、{1}を戻り値としするToFilterExpressionメソッドは存在しません。", key.GetType().ToString(), typeof( List< IEntity > ).ToString() ) ) );
					}
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするToFilterExpressionメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
			//return( this.ToFilterExpression( ( TKey )key ) );
		}

		/// <summary>
		/// データセットにテーブルスキーマを充填します。
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="ds">データセット</param>
		protected static void FillSchema( string tableName, DataSet ds )
		{
			TableModule.FillSchema< TEntity >( tableName, ds );
		}

		/// <summary>
		/// データテーブルにテーブルスキーマを充填します。
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="table">データテーブル</param>
		protected static void FillSchema( string tableName, DataTable table )
		{
			TableModule.FillSchema< TEntity >( tableName, table );
		}

		#region ITableModule<TKey,TEntity,TDataTransferObject> メンバ

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		public TEntity this[ TKey primaryKey ]
		{
			get
			{
				return( ( TEntity )base[ primaryKey ] );
			}
		}

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public new TEntity this[ IndexNumber index ]
		{
			get
			{
				return( ( TEntity )base[ index ] );
			}
		}

		/// <summary>
		/// テーブルモジュールにエンティティを追加します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public void Add( TEntity entity )
		{
			base.Add( entity );
		}

		/// <summary>
		/// テーブルモジュールにエンティティが存在するかを判断します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <returns>テーブルモジュールにエンティティが存在する場合はtrueが返ります。</returns>
		public bool Contains( TEntity entity )
		{
			return( base.Contains( entity ) );
		}

		/// <summary>
		/// テーブルモジュール内のエンティティをエンティティの配列にコピーします。
		/// </summary>
		/// <param name="entities">エンティティの配列</param>
		/// <param name="arrayIndex">配列へコピー開始するインデックス</param>
		public void CopyTo( TEntity[] entities, int arrayIndex )
		{
			foreach( TEntity entity in this )
			{
				entities[ arrayIndex ] = entity;

				arrayIndex++;
			}
		}

		/// <summary>
		/// データ行を削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void Delete( TKey primaryKey )
		{
			base.Delete( primaryKey );
		}

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>データ行</returns>
		public DataRow GetRow( TKey primaryKey )
		{
			return( base.GetRow( primaryKey ) );
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public void Insert( TDataTransferObject dto )
		{
			base.Insert( dto );
		}

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		public bool IsDeleted( TKey primaryKey )
		{
			return( base.IsDeleted( primaryKey ) );
		}

		/// <summary>
		/// 指定したテーブルモジュールの内容を結合します。
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		public void Merge( ITableModule<TKey, TEntity, TDataTransferObject> module )
		{
			this._table.Merge( module.Table );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		public new TEntity NewEntity()
		{
			return( ( TEntity )base.NewEntity() );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>新しい行のエンティティ</returns>
		TEntity ITableModule< TKey, TEntity, TDataTransferObject >.NewEntity( TKey primaryKey )
		{
			return( this.NewEntity( primaryKey ) );
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		public void Remove( TEntity entity )
		{
			base.Remove( entity );
		}

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void RemoveBy( TKey primaryKey )
		{
			base.RemoveBy( primaryKey );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>エンティティ配列</returns>
		public new TEntity[] Select( string filterExpression )
		{
			return( this.Select( filterExpression, FilteringCondition.Default ) );
		}

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>エンティティ配列</returns>
		public new TEntity[] Select( string filterExpression, FilteringCondition filteringCondition )
		{
			return( this.ToEntityArray( this.SelectDataRows( filterExpression ), filteringCondition ) );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		List< TEntity > ITableModule< TKey, TEntity, TDataTransferObject >.SelectBy( IKey key )
		{
			try
			{
				BindingFlags flag = BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance;

				MethodInfo method = this.GetType().GetMethod( "SelectBy", flag, null, new Type[]{ key.GetType() }, null );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );

					if( returnValue is List< TEntity > )
					{
						return( ( List< TEntity > )returnValue );
					}
					else
					{
						throw( new MissingMethodException( string.Format( "{0}をパラメータとし、{1}を戻り値としするSelectByメソッドは存在しません。", key.GetType().ToString(), typeof( List< TEntity > ).ToString() ) ) );
					}
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするSelectByメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void ITableModule< TKey, TEntity, TDataTransferObject >.Set( TDataTransferObject dto )
		{
			this.Set( dto );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public new List< TEntity > ToEntities()
		{
			return( this.ToEntities( FilteringCondition.Default ) );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public new List< TEntity > ToEntities( FilteringCondition filteringCondition )
		{
			List< TEntity > entities = new List< TEntity >();

			foreach( TEntity entity in new TableModuleEnumerable< TKey, TEntity, TDataTransferObject >( this, filteringCondition ) )
			{
				entities.Add( entity );
			}

			return( entities );
		}

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		public TEntity ToEntity( TKey primaryKey )
		{
			return( ( TEntity )base.ToEntity( primaryKey ) );
		}

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public new TEntity ToEntity( IndexNumber index )
		{
			return( ( TEntity )base.ToEntity( index ) );
		}

		#endregion

		#region ICollection<TEntity> メンバ

		/// <summary>
		/// テーブルモジュールが読み取り専用かを表します。
		/// </summary>
		bool ICollection< TEntity >.IsReadOnly
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="item">削除するエンティティ</param>
		/// <returns>削除が成功した場合はtrueが返ります。</returns>
		bool ICollection< TEntity >.Remove( TEntity item )
		{
			if( this.Contains( item ) )
			{
				this.Remove( item );

				return( true );
			}
			else
			{
				return( true );
			}
		}

		#endregion

		#region ICollection メンバ

		/// <summary>
		/// コレクションがスレッドセーフであるかを表します。
		/// </summary>
		bool ICollection.IsSynchronized
		{
			get
			{
				return( false );
			}
		}

		/// <summary>
		/// 同期処理用のインスタンスを表します。
		/// </summary>
		object ICollection.SyncRoot
		{
			get
			{
				return( this );
			}
		}

		#endregion

		#region IEnumerable<TEntity> メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		public new IEnumerator< TEntity > GetEnumerator()
		{
			return( new TableModuleEnumerator< TKey, TEntity, TDataTransferObject >( this ) );
		}

		#endregion

		#region IEnumerable メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		IEnumerator IEnumerable.GetEnumerator()
		{
			return( this.GetEnumerator() );
		}

		#endregion
	}
}
