using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using NDac.Keys;
using NDac.Modules.DataTransferObjects;
using NDac.Modules.Entities;

namespace NDac.Modules
{
	/// <summary>
	/// テーブルモジュールのインターフェイスを表します。
	/// </summary>
	public interface ITableModule : ICollection, IEnumerable
	{
		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="key">エンティティを識別するキー</param>
		/// <returns>エンティティ</returns>
		IEntity this[ IKey key ]{ get; }

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		IEntity this[ IndexNumber index ]{ get; }

		/// <summary>
		/// データが変更されているか否かを表します。
		/// </summary>
		bool IsModified{ get; }

		/// <summary>
		/// データテーブルを表します。
		/// </summary>
		DataTable Table{ get; }

		/// <summary>
		/// テーブル名を表します。
		/// </summary>
		string TableName{ get; }

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="key">キー</param>
		void Delete( IKey key );

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>データ行</returns>
		DataRow GetRow( IKey key );

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>データ行</returns>
		DataRow GetRow( int index );

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void Insert( IDataTransferObject dto );

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		bool IsDeleted( IKey key );

		/// <summary>
		/// テーブルモジュールの内容を結合します。
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		void Merge( ITableModule module );

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		IEntity NewEntity();

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>新しい行のエンティティ</returns>
		IEntity NewEntity( IKey key );

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		void Remove( IEntity entity );

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="key">キー</param>
		void RemoveBy( IKey key );

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>エンティティ配列</returns>
		IEntity[] Select( string filterExpression );

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		IEntity[] Select( string filterExpression, FilteringCondition filteringCondition );

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		List< IEntity > SelectBy( IKey key );

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void Set( IDataTransferObject dto );

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		List< IEntity > ToEntities();

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		List< IEntity > ToEntities( FilteringCondition filteringCondition );

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティ</returns>
		IEntity ToEntity( IKey key );

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		IEntity ToEntity( IndexNumber index );

		/// <summary>
		/// エンティティリストに変換します。(主にDataGridViewのDataSourceにバインドする時に使用して下さい。)
		/// </summary>
		/// <returns>エンティティリスト</returns>
		ArrayList ToEntityArrayList();

		/// <summary>
		/// エンティティリストに変換します。(主にDataGridViewのDataSourceにバインドする時に使用して下さい。)
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		ArrayList ToEntityArrayList( FilteringCondition filteringCondition );
	}

	/// <summary>
	/// テーブルモジュールのインターフェイスを表します。
	/// </summary>
	public interface ITableModule< TKey, TEntity, TDataTransferObject > : ITableModule, ICollection< TEntity >, IEnumerable< TEntity >
		where TKey : IKey
		where TEntity : IEntity
		where TDataTransferObject : IDataTransferObject
	{
		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		TEntity this[ TKey primaryKey ]{ get; }

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		new TEntity this[ IndexNumber index ]{ get; }

		/// <summary>
		/// テーブルモジュールにエンティティを追加します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		new void Add( TEntity entity );

		/// <summary>
		/// テーブルモジュールにエンティティが存在するかを判断します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <returns>テーブルモジュールにエンティティが存在する場合はtrueが返ります。</returns>
		new bool Contains( TEntity entity );

		/// <summary>
		/// テーブルモジュール内のエンティティをエンティティの配列にコピーします。
		/// </summary>
		/// <param name="entities">エンティティの配列</param>
		/// <param name="index">配列へコピー開始するインデックス</param>
		new void CopyTo( Array entities, int index );

		/// <summary>
		/// テーブルモジュール内のエンティティをエンティティの配列にコピーします。
		/// </summary>
		/// <param name="entities">エンティティの配列</param>
		/// <param name="index">配列へコピー開始するインデックス</param>
		new void CopyTo( TEntity[] entities, int arrayIndex );

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void Delete( TKey primaryKey );

		/// <summary>
		/// データ行を取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>データ行</returns>
		DataRow GetRow( TKey primaryKey );

		/// <summary>
		/// データ変換オブジェクトを挿入します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void Insert( TDataTransferObject dto );

		/// <summary>
		/// 指定したキーのエンティティが削除されているかを取得します。このメソッドはデータの読込みまたは前回の変更のコミットから次のデータの変更をコミットされるまでの間有効です。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>指定したキーのエンティティが削除されている場合trueが返ります。</returns>
		bool IsDeleted( TKey primaryKey );

		/// <summary>
		/// テーブルモジュールの内容を結合します。
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		void Merge( ITableModule< TKey, TEntity, TDataTransferObject > module );

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		new TEntity NewEntity();

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>新しい行のエンティティ</returns>
		TEntity NewEntity( TKey primaryKey );

		/// <summary>
		/// テーブルモジュール内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		new void Remove( TEntity entity );

		/// <summary>
		/// テーブルモジュール内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void RemoveBy( TKey primaryKey );

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <returns>エンティティ配列</returns>
		new TEntity[] Select( string filterExpression );

		/// <summary>
		/// フィルタ処理式に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="filterExpression">フィルタ処理式</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティ配列</returns>
		new TEntity[] Select( string filterExpression, FilteringCondition filteringCondition );

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		new List< TEntity > SelectBy( IKey key );

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		void Set( TDataTransferObject dto );

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		new List< TEntity > ToEntities();

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		new List< TEntity > ToEntities( FilteringCondition filteringCondition );

		/// <summary>
		/// 行データをエンティティに変換します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		TEntity ToEntity( TKey primaryKey );

		/// <summary>
		/// データ行をエンティティに変換します。
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		new TEntity ToEntity( IndexNumber index );
	}
}
