using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NDac.Keys;
using NDac.Modules.Entities;

namespace NDac.Modules.DataTransferObjects
{
	/// <summary>
	/// データ変換オブジェクトのインターフェイスを表します。
	/// </summary>
	public interface IDataTransferObject : IEquatable< IDataTransferObject >
	{
		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		void Copy( IDataTransferObject source );

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="entity">コピー元のエンティティ</param>
		void Copy( IEntity entity );

		/// <summary>
		/// データベースよりデータを削除します。
		/// </summary>
		void Delete();

		/// <summary>
		/// 内部に保有しているエンティティを取得します。
		/// </summary>
		/// <returns>エンティティ</returns>
		IEntity GetEntity();
		
		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		IKey GetPrimaryKey();

		/// <summary>
		/// 内部にエンティティを保有しているか判定します。
		/// </summary>
		/// <returns>内部にエンティティを保有している場合trueが返ります。</returns>
		bool HasEntity();

		/// <summary>
		/// エンティティの内容が削除されているか否かを判定します。
		/// </summary>
		/// <returns>内容が削除されている場合trueが返ります。</returns>
		bool IsDeleted();

		/// <summary>
		/// エンティティのデータが未更新かを判定します。データが更新されている場合はfalseが返ります。それ以外の全ての状態はtrueが返ります。
		/// </summary>
		/// <returns>未更新(新規，変更，削除)の場合trueが返ります。</returns>
		bool IsDirty();

		/// <summary>
		/// エンティティの内容が変更されているか否かを判定します。
		/// </summary>
		/// <returns>内容が変更されている場合trueが返ります。</returns>
		bool IsModified();

		/// <summary>
		/// エンティティが新規作成されたデータかを判定します。
		/// </summary>
		/// <returns>新規作成されたデータの場合trueが返ります。</returns>
		bool IsNew();

		/// <summary>
		/// データベースよりプライマリキーでデータを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void Load( IKey primaryKey );

		/// <summary>
		/// データベースへデータをセーブします。
		/// </summary>
		void Save();
	}

	/// <summary>
	/// データ変換オブジェクトのインターフェイスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	public interface IDataTransferObject< TKey, TEntity > : IDataTransferObject
		where TKey : class, IKey
		where TEntity : class, IEntity< TKey >
	{
		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		void Copy( IDataTransferObject< TKey, TEntity > source );

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="entity">コピー元のエンティティ</param>
		void Copy( TEntity entity );

		/// <summary>
		/// 内部に保有しているエンティティを取得します。
		/// </summary>
		/// <returns>エンティティ</returns>
		new TEntity GetEntity();

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		new TKey GetPrimaryKey();

		/// <summary>
		/// データベースよりプライマリキーでデータを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		void Load( TKey primaryKey );
	}
}
