using System;
using System.Text;

namespace NDac.Modules
{
	/// <summary>
	/// 文字列の切り詰めクラスを表します。
	/// </summary>
	static class StringCutDowner
	{
		static string _encodingName = "shift_jis"; 

		/// <summary>
		/// 可変長の文字列の切り詰めを行います。
		/// </summary>
		/// <param name="value">文字列</param>
		/// <param name="maxByteLength">最大バイトサイズ</param>
		/// <returns>切り詰められた文字列</returns>
		public static string Variable( string value, int maxByteLength )
		{
			int byteCount = value == null ? 0 : Encoding.GetEncoding( _encodingName ).GetByteCount( value );

			if( byteCount <= maxByteLength )
			{
				return( value );
			}
			else
			{
				byte[] bytes = Encoding.GetEncoding( _encodingName ).GetBytes( value );

				Array.Resize< byte >( ref bytes, maxByteLength );

				return( Encoding.GetEncoding( _encodingName ).GetString( bytes ) );
			}
		}

		/// <summary>
		/// 固定長の文字列の切り詰めを行います。固定長バイトサイズに満たない場合は0x20（スペース）を埋めます。
		/// </summary>
		/// <param name="value">文字列</param>
		/// <param name="fixedByteLength">固定長バイトサイズ</param>
		/// <returns>切り詰められた文字列</returns>
		public static string Fixed( string value, int fixedByteLength )
		{
			int byteCount = value == null ? 0 : Encoding.GetEncoding( _encodingName ).GetByteCount( value );

			if( byteCount == fixedByteLength )
			{
				return( value );
			}
			else if( fixedByteLength < byteCount )
			{
				byte[] bytes = Encoding.GetEncoding( _encodingName ).GetBytes( value );

				Array.Resize< byte >( ref bytes, fixedByteLength );

				return( Encoding.GetEncoding( _encodingName ).GetString( bytes ) );
			}
			else if( byteCount == 0 )
			{
				byte[] bytes = new byte[ fixedByteLength ];

				for( int i = byteCount; i < bytes.Length; i++ )
				{
					bytes[ i ] = ( byte )0x20;
				}

				return( Encoding.GetEncoding( _encodingName ).GetString( bytes ) );
			}
			else
			{
				byte[] bytes = Encoding.GetEncoding( _encodingName ).GetBytes( value );

				Array.Resize< byte >( ref bytes, fixedByteLength );

				for( int i = byteCount; i < bytes.Length; i++ )
				{
					bytes[ i ] = ( byte )0x20;
				}

				return( Encoding.GetEncoding( _encodingName ).GetString( bytes ) );
			}
		}

		/// <summary>
		/// エンコーディングを変更します。
		/// </summary>
		/// <param name="encodingName">エンコーディング名</param>
		public static void ChangeEncoding( string encodingName )
		{
			_encodingName = encodingName;
		}
	}
}
