using System.Collections;
using System.Collections.Generic;
using System.Data;
using NDac.Datas;
using NDac.Gateways;
using NDac.Keys;
using NDac.Modules;
using NDac.Modules.DataTransferObjects;
using NDac.Modules.Entities;

namespace NDac.RemoteFacades
{
	/// <summary>
	/// リモートファサードの抽象クラスを表します。
	/// </summary>
	public abstract class RemoteFacade : IEnumerable< IEntity >
	{
		protected ITableDataGateway	_gateway;
		protected ITableModule		_module;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		public RemoteFacade( DataSetHolder holder )
		{
			this._gateway	= this.NewGateway< ITableDataGateway >( holder );

			this._module	= this.NewModule< ITableModule >( holder );
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <param name="loadKey">データを充填するキー</param>
		public RemoteFacade( DataSetHolder holder, IKey loadKey )
		{
			this._gateway	= this.NewGateway< ITableDataGateway >( holder );

			this._module	= this.NewModule< ITableModule >( holder );

			this._gateway.LoadBy( loadKey );
		}

		/// <summary>
		/// 新しいテーブルデータゲートウェイを作成します。
		/// </summary>
		/// <typeparam name="T">テーブルデータゲートウェイの型</typeparam>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルデータゲートウェイ</returns>
		protected abstract T NewGateway< T >( DataSetHolder holder ) where T : class, ITableDataGateway;

		/// <summary>
		/// 新しいテーブルモジュールを作成します。
		/// </summary>
		/// <typeparam name="T">テーブルモジュールの型</typeparam>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルモジュール</returns>
		protected abstract T NewModule< T >( DataSetHolder holder ) where T : class, ITableModule;

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティ</returns>
		public IEntity this[ IKey key ]
		{
			get
			{
				return( this._module[ key ] );
			}
		}

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public IEntity this[ IndexNumber index ]
		{
			get
			{
				return( this._module[ index ] );
			}
		}

		/// <summary>
		/// 保持しているエンティティ数（レコード数）を表します。
		/// </summary>
		public int Count
		{
			get
			{
				return( this._module.Count );
			}
		}

		/// <summary>
		/// テーブルデータゲートウェイを表します。
		/// </summary>
		public ITableDataGateway Gateway
		{
			get
			{
				return( this._gateway );
			}
		}

		/// <summary>
		/// テーブルからデータを充填済みであるかを判定します。
		/// </summary>
		public bool IsLoaded
		{
			get
			{
				return( this.Holder.HasTable( this._gateway.TableName ) );
			}
		}

		/// <summary>
		/// テーブルモジュールを表します。
		/// </summary>
		public ITableModule Module
		{
			get
			{
				return( this._module );
			}
		}

		/// <summary>
		/// データセットホルダーを表します。
		/// </summary>
		public DataSetHolder Holder
		{
			get
			{
				return( this._gateway.Holder );
			}
		}

		/// <summary>
		/// データテーブルを表します。
		/// </summary>
		public DataTable Table
		{
			get
			{
				return( this._gateway.Table );
			}
		}

		/// <summary>
		/// テーブル名を表します。
		/// </summary>
		public string TableName
		{
			get
			{
				return( this._gateway.TableName );
			}
		}

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="key">キー</param>
		public void Delete( IKey key )
		{
			this._module.Delete( key );
		}

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		public bool HasValue( IKey key )
		{
			return( this._gateway.HasValue( key ) );
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。
		/// <para>データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。</para>
		/// <para>また内部に保持しているDataSetHolderに対象テーブルが存在しない場合は、このメソッド内でテーブルスキーマを充填します。</para>
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public void Insert( IDataTransferObject dto )
		{
			if( !this.Holder.Data.Tables.Contains( this._gateway.TableName ) )
			{
				this._gateway.LoadBy( new TableSchemaKey() );
			}

			this._module.Insert( dto );
		}

		/// <summary>
		/// テーブルより全データを充填します。
		/// </summary>
		public void LoadAll()
		{
			this._gateway.LoadAll();
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="key">キー</param>
		public void LoadBy( IKey key )
		{
			this._gateway.LoadBy( key );
		}

		/// <summary>
		/// 指定したリモートファサードの内容を結合します。
		/// </summary>
		/// <param name="facade">リモートファサード</param>
		public void Merge( RemoteFacade facade )
		{
			this.Table.Merge( facade.Table );
		}

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="key">キー</param>
		public void MoreLoadBy( IKey key )
		{
			this._gateway.MoreLoadBy( key );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		public IEntity NewEntity()
		{
			return( this._module.NewEntity() );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>新しい行のエンティティ</returns>
		public IEntity NewEntity( IKey key )
		{
			return( this._module.NewEntity( key ) );
		}

		/// <summary>
		/// リモートファサード内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		public void Remove( IEntity entity )
		{
			this._module.Remove( entity );
		}

		/// <summary>
		/// リモートファサード内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="key">キー</param>
		public void Remove( IKey key )
		{
			this._module.RemoveBy( key );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		public List< IEntity > SelectBy( IKey key )
		{
			return( this._module.SelectBy( key ) );
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public void Set( IDataTransferObject dto )
		{
			if( !this.Holder.Data.Tables.Contains( this._gateway.TableName ) )
			{
				this._gateway.LoadBy( new TableSchemaKey() );
			}

			this._module.Set( dto );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public List< IEntity > ToEntities()
		{
			return( this._module.ToEntities() );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		public List< IEntity > ToEntities( FilteringCondition filteringCondition )
		{
			return( this._module.ToEntities( filteringCondition ) );
		}

		/// <summary>
		/// データを更新します。
		/// </summary>
		public void Update()
		{
			this._gateway.Holder.Update();
		}

		#region IEnumerable<IEntity> メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		public IEnumerator< IEntity > GetEnumerator()
		{
			return( ( ( TableModule )this._module ).GetEnumerator() );
		}

		#endregion

		#region IEnumerable メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		IEnumerator IEnumerable.GetEnumerator()
		{
			return( this._module.GetEnumerator() );
		}

		#endregion
	}

	/// <summary>
	/// リモートファサードの抽象クラスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TGateway">テーブルデータゲートウェイの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	/// <typeparam name="TDataTransferObject">データ変換オブジェクトの型</typeparam>
	/// <typeparam name="TModule">テーブルモジュールの型</typeparam>
	public abstract class RemoteFacade< TKey, TGateway, TEntity, TDataTransferObject, TModule > : RemoteFacade, IEnumerable< TEntity >
		where TKey : class, IKey
		where TGateway : TableDataGateway< TKey >
		where TEntity : class, IEntity< TKey >
		where TDataTransferObject : class, IDataTransferObject< TKey, TEntity >
		where TModule : class, ITableModule< TKey, TEntity, TDataTransferObject >
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		public RemoteFacade( DataSetHolder holder ) : base( holder )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <param name="loadKey">データを充填するキー</param>
		public RemoteFacade( DataSetHolder holder, IKey key ) : base( holder, key )
		{
		}

		/// <summary>
		/// 新しいテーブルデータゲートウェイを作成します。
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルデータゲートウェイ</returns>
		protected abstract TGateway NewGateway( DataSetHolder holder );

		/// <summary>
		/// 新しいテーブルモジュールを作成します。
		/// </summary>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルモジュール</returns>
		protected abstract TModule NewModule( DataSetHolder holder );

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>エンティティ</returns>
		public TEntity this[ TKey primaryKey ]
		{
			get
			{
				return( ( TEntity )base[ ( IKey )primaryKey ] );
			}
		}

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="index">インデックス</param>
		/// <returns>エンティティ</returns>
		public new TEntity this[ IndexNumber index ]
		{
			get
			{
				return( ( TEntity )this.Module[ index ] );
			}
		}

		/// <summary>
		/// テーブルデータゲートウェイを表します。
		/// </summary>
		public new TGateway Gateway
		{
			get
			{
				return( ( TGateway )base.Gateway );
			}
		}

		/// <summary>
		/// テーブルモジュールを表します。
		/// </summary>
		public new TModule Module
		{
			get
			{
				return( ( TModule )base.Module );
			}
		}

		/// <summary>
		/// 行データを削除します。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void Delete( TKey primaryKey )
		{
			this.Module.Delete( primaryKey );
		}

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="key">プライマリキー</param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		public bool HasValue( TKey primaryKey )
		{
			return( this.Gateway.HasValue( primaryKey ) );
		}

		/// <summary>
		/// データ変換オブジェクトを挿入します。
		/// <para>データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。</para>
		/// <para>また内部に保持しているDataSetHolderに対象テーブルが存在しない場合は、このメソッド内でテーブルスキーマを充填します。</para>
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public void Insert( TDataTransferObject dto )
		{
			if( !this.Holder.Data.Tables.Contains( this.Gateway.TableName ) )
			{
				this.Gateway.LoadBy( new TableSchemaKey() );
			}

			this.Module.Insert( dto );
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="key">プライマリキー</param>
		public void LoadBy( TKey primaryKey )
		{
			this.Gateway.LoadBy( primaryKey );
		}

		/// <summary>
		/// 指定したリモートファサードの内容を結合します。
		/// </summary>
		/// <param name="facade">リモートファサード</param>
		public void Merge( RemoteFacade< TKey, TGateway, TEntity, TDataTransferObject, TModule > facade )
		{
			this.Table.Merge( facade.Table );
		}

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="key">プライマリキー</param>
		public void MoreLoadBy( TKey primaryKey )
		{
			this.Gateway.MoreLoadBy( primaryKey );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <returns>新しい行のエンティティ</returns>
		public new TEntity NewEntity()
		{
			return( this.Module.NewEntity() );
		}

		/// <summary>
		/// 新しい行のエンティティを作成します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>新しい行のエンティティ</returns>
		public TEntity NewEntity( TKey primaryKey )
		{
			return( this.Module.NewEntity( primaryKey ) );
		}

		/// <summary>
		/// 新しいテーブルデータゲートウェイを作成します。
		/// </summary>
		/// <typeparam name="T">テーブルデータゲートウェイの型</typeparam>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルデータゲートウェイ</returns>
		protected override T NewGateway< T >( DataSetHolder holder )
		{
			ITableDataGateway gateway = this.NewGateway( holder );

			return( ( T )gateway );
		}

		/// <summary>
		/// 新しいテーブルモジュールを作成します。
		/// </summary>
		/// <typeparam name="T">テーブルモジュールの型</typeparam>
		/// <param name="holder">データセットホルダー</param>
		/// <returns>テーブルモジュール</returns>
		protected override T NewModule< T >( DataSetHolder holder )
		{
			ITableModule module = this.NewModule( holder );

			return( ( T )module );
		}

		/// <summary>
		/// リモートファサード内からエンティティを削除します。
		/// </summary>
		/// <param name="entity">削除するエンティティ</param>
		public void Remove( TEntity entity )
		{
			this._module.Remove( entity );
		}

		/// <summary>
		/// リモートファサード内からキー情報に一致する全てのエンティティを削除します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public void Remove( TKey primaryKey )
		{
			this._module.RemoveBy( primaryKey );
		}

		/// <summary>
		/// キー情報に一致する全てのエンティティを取得します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>エンティティリスト</returns>
		public new List< TEntity > SelectBy( IKey key )
		{
			return( ( ( TModule )this._module ).SelectBy( key ) );
		}

		/// <summary>
		/// データ変換オブジェクトをセットします。データ変換オブジェクトのキーに一致するデータが存在する場合はUpdate，存在しない場合はInsertします。データベースに反映する時はDataSetHolderのUpdate()を呼び出して下さい。
		/// </summary>
		/// <param name="dto">データ変換オブジェクト</param>
		public void Set( TDataTransferObject dto )
		{
			if( !this.Holder.Data.Tables.Contains( this.Gateway.TableName ) )
			{
				this.Gateway.LoadBy( new TableSchemaKey() );
			}

			this.Module.Set( dto );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <returns>エンティティリスト</returns>
		public new List< TEntity > ToEntities()
		{
			return( ( ( TModule )this._module ).ToEntities() );
		}

		/// <summary>
		/// エンティティリストに変換します。
		/// </summary>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		/// <returns>エンティティリスト</returns>
		public new List< TEntity > ToEntities( FilteringCondition filteringCondition )
		{
			return( ( ( TModule )this._module ).ToEntities( filteringCondition ) );
		}

		#region IEnumerable<TEntity> メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		public new IEnumerator< TEntity > GetEnumerator()
		{
			return( ( ( TModule )this.Module ).GetEnumerator() );
		}

		#endregion

		#region IEnumerable メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		IEnumerator IEnumerable.GetEnumerator()
		{
			return( this.GetEnumerator() );
		}

		#endregion
	}
}
