#!/usr/bin/env bash

#*****************************************************************************
# xtmvavg.sh
# 移動平均算出スクリプト
# Y.Hamuro
# 1.0b 2004/05/07
# 1.0 -k追加,-fの指定方法変更,トラップ対応 2004/11/09
#*****************************************************************************

function help {
cat >/dev/stderr <<EOF
--------------------------
xtmvavg.sh version 1.0
--------------------------
概要) 移動平均を計算する。
書式) xtmvavg.sh -k キー項目名 -s 並べ換える項目名リスト -f 集計項目名:新項目名
                 -n 移動する日数 [-i 入力ファイル名] [-o出力ファイル名]
                  [-T ワークディレクトリ名] [-V] [-h]

      -V : コマンドの完了メッセージを表示させる。
      -h : ヘルプの表示

例1) xtmvavg.sh -s 期 -f 金額:移動平均 -n 3 -i input.xt -o output.xt
例2) xtmvavg.sh -s 期%r -f 数量:数量平均,金額:金額平均 -n 5 -i input2.xt -o output2.xt
例3) xtmvavg.sh -k 店 -s 期 -f 金額:金額平均 -n 4 <input3.xt >output3.xt

例1の入出力データ例
--------------
input.xtの内容
--------------
期 金額
1 10
2 20
3 30
4 40
5 50
--------------
output.xtの内容
--------------
期 金額 移動平均
1 10 *
2 20 *
3 30 20
4 40 30
5 50 40
--------------
期3 : (10+20+30)/3 = 20
期4 : (20+30+40)/3 = 30
期5 : (30+40+50)/3 = 40
EOF
exit 1
}

#デフォルトでは、コマンドのメッセージはOFF
export mssQuiet=1

#完了メッセージ用にコマンドラインを保存
cmdLine="$0 $*"

#シグナルによる終了
function endByInt {
  rm -f $TD-xx*
  echo "#ERROR# $$" \"$cmdLine\" \""end by signal(ctr^C)"\" >/dev/stderr
  exit 1
}

#パラメータのチェック＆セット
while getopts ":k:s:f:n:a:i:o:T:Vh" opt; do
  case $opt in
    k  ) KEY=$OPTARG ;;
    s  ) SORT=$OPTARG ;;
    f  ) FLD=$OPTARG ;;
    n  ) MVCNT=$OPTARG ;;
    i  ) INPUT=$OPTARG ;;
    o  ) OUTPUT=$OPTARG ;;
    T  ) TMPD=$OPTARG ;;
    V  ) mssQuiet=0 ;;
    h  ) help ;;
    \? ) help ;;
  esac
done
shift $(($OPTIND -1 ))

if [ "$SORT" = "" ]; then
  echo "#ERROR# $$ \"$0\" \"option -s is mandatory\"" >/dev/stderr
  exit 1
fi
if [ "$FLD" = "" ]; then
  echo "#ERROR# $$ \"$0\" \"option -f is mandatory\"" >/dev/stderr
  exit 1
fi
if [ "$MVCNT" = "" ]; then
  echo "#ERROR# $$ \"$0\" \"option -n is mandatory\"" >/dev/stderr
  exit 1
fi
if [ "$INPUT" = "" ]; then
   INPUT=/dev/stdin
fi
if [ "$OUTPUT" = "" ]; then
 OUTPUT=/dev/stdout
fi
if [ "$TMPD" = "" ]; then
   TMPD=/tmp
fi
if [ "$KEY" = "" ]; then
  KEY="#same#"
  KEYPAR=""
else
  KEYPAR=${KEY},
fi

#-sのパラメータを%で分割 -s 日付%r -> SORTFLD=日付 SORTPAR=r
SORTFLD=${SORT%\%*}
SORTPAR=${SORT#*%}
if [ "$SORTFLD" = "$SORTPAR" ]; then
  SORTPAR=""
else
  SORTPAR="%$SORTPAR"
fi

#-fのパラメータを入力ファイル項目名と出力ファイル項目名に分割する
#-f a:b,c:d -> iFldList=a,c oFldList=b,d
ARRY=(${FLD//[,]/ })
iFldList=""
oFldList=""
for FN in "${ARRY[@]}"; do
  FN1=${FN%:*}
  FN2=${FN#*:}
  if [ "$FN1" = "$FN2" ] ; then
    echo "#MSG# $$ \"$0\" \"option -f must take fieldName:newFieldName\"" >/dev/stderr
    exit 1
  fi
  iFldList=$iFldList,$FN1
  oFldList=$oFldList,$FN2
done
iFldList=${iFldList#,}
oFldList=${oFldList#,}

#ワークファイル名
TD="$TMPD/xtmvavg-$$"

#トラップ発動
trap endByInt INT QUIT TERM HUP

#=====================================================
# 【ステップ１】
# 入力ファイルの各行を移動数(-nの値)分duplicateし、
# 期内での連番項目(##mvSeq##)を作成する。
# この連番は期が新しくなる毎に開始番号をずらしておく。
#-----------------------------------------------------
# 期 金額  ($INPUT: 以下の込めんとはこの入力データを例にして説明している)
# 1 10
# 2 20
# 3 30
# 4 40
# 5 50
#  ↓
# 期 金額 ##mvSeq##  ($TD-xxbase)
# 1 10 1001
# 1 10 1002
# 1 10 1003
# 2 20 1002
# 2 20 1003
# 2 20 1004
# 3 30 1003
# 3 30 1004
# 3 30 1005
# 4 40 1004
# 4 40 1005
# 4 40 1006
# 5 50 1005
# 5 50 1006
# 5 50 1007
#=====================================================
xtnumber -k $KEY -s $SORTFLD$SORTPAR -S 10000000 -a '##seqNo##' -i $INPUT |
xtsetchr -v $MVCNT -a days |
xtduprec -f days |
xtcal -k $KEYPAR'##seqNo##' -c '$##seqNo##+keyLine()' -a '##mvSeq##' |
xtcut -r -f 'days,##seqNo##' -o "$TD-xxbase"

#=====================================================
# 【ステップ２】
# 次のステップ３で##mvSeq##をキーにして移動平均を求めるが
# その際、件数が移動数(-nの値)に満たないものは意味
# がない。
# そこで件数が移動数(-nの値)ある##mvSeq##を選んでおく。
#-----------------------------------------------------
# ##mvSeq## ##cnt##  ($TD-xx2)
# 1003 3
# 1004 3
# 1005 3
#=====================================================
xtcut   -f $KEYPAR'##mvSeq##' -i $TD-xxbase | 
xtcount -k $KEYPAR'##mvSeq##' -a '##cnt##' |
xtsel   -c '$##cnt##=='$MVCNT -o $TD-xx2

#=====================================================
# 【ステップ３】
# ##mvSeq##をキーにして移動平均を求める。
# その際、上記で求めた##mvSeq##の件数が移動数(-nの値)
# ある行のみを選ぶ。
#-----------------------------------------------------
# ##mvSeq## ##cnt##  ($TD-xx3)
# 1003 20
# 1004 30
# 1005 40
#=====================================================
xtcut    -f $KEYPAR'##mvSeq##',$iFldList -i $TD-xxbase |
xtagg    -k $KEYPAR'##mvSeq##' -f $FLD -c avg |
xtcommon -k $KEYPAR'##mvSeq##' -m $TD-xx2 -o $TD-xx3

#=====================================================
# 【ステップ４】
# ステップ１で求めたデータから、##mvSeqに対応する期を
# 選択し、ステップ３で求めた移動平均値を結合する。
#-----------------------------------------------------
# 期,金額,移動平均 ($OUTPUT)
# 1 10 *
# 2 20 *
# 3 30 20
# 4 40 30
# 5 50 40
#=====================================================
xtcut  -f $KEYPAR'##mvSeq##',$SORTFLD,$iFldList -i $TD-xxbase |
xtbest -k $KEYPAR$SORTFLD -s '##mvSeq##' -R 1 |
xtjoin -n -k $KEYPAR'##mvSeq##' -m $TD-xx3 -f $oFldList |
xtcut  -r -f '##mvSeq##' -o $OUTPUT

#ワークファイル削除
rm -f $TD-xx*

#完了メッセージ表示
echo "#END# $$" \"$cmdLine\" >/dev/stderr
exit 0
#===============================================================
