/*
 * arp.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ARP protocol.
 */


#include<user/include/share/user_config.h>
#include<types.h>
#include<lib.h>
#include<lock.h>
#include<device.h>
#include<proc.h>
#include<errno.h>
#include<net/net.h>
#include<net/ether.h>
#include<net/ip.h>
#include<net/arp.h>


enum{
	/* Hard type. */
	HARD_TYPE_ETHER= 1,		/* Ethernet. */

	/* Protocol type. */
	PROT_TYPE_IP=0x0800,	/* IP. */

	/* Operation code. */
	OPE_CODE_ARP_REQ= 1,	/* ARP request. */
	OPE_CODE_ARP_REP= 2,	/* ARP repry. */
	OPE_CODE_RARP_REQ=3,	/* RARP request. */
	OPE_CODE_RARP_REP=4,	/* RARP repry. */
};


/**************************************************************************
 *
 * ARPå
 *
 **************************************************************************/


enum{
	ARP_CACHE_NUM=64,				/* ARPå */
	ARP_CACHE_HASH=ARP_CACHE_NUM/2,	/* ARPåϥå */
};


typedef struct{
	uint ip;
	char mac[6];
	char next;
}ARP_CACHE;


static ARP_CACHE arpCache[ARP_CACHE_NUM];
static char cacheEmpty=0;
static char hashTbl[ARP_CACHE_HASH];
static int cacheLock=0;
static char broadcastMac[]={0xff,0xff,0xff,0xff,0xff,0xff};


/*
 * PRIVATE
 * Search previous index address.
 * parameters : IP address
 * return : prev pointer or NULL
 */
static inline char *searchPrevIndex(uint ip)
{
	uchar tbl;
	int i;


	tbl=(ip>>24)%ARP_CACHE_HASH;
	if(hashTbl[tbl]==-1)return &hashTbl[tbl];
	else
		for(i=hashTbl[tbl];;i=arpCache[i].next)
		{
			if(arpCache[i].ip==ip)return NULL;
			if(arpCache[i].next==-1)return &arpCache[i].next;
		}
}


/*
 * PRIVATE
 * åβꡣ
 */
static inline int snatchCache()
{
	int num;
	int i;


	for(i=0;hashTbl[i]==-1;++i);
	num=hashTbl[i];
	hashTbl[i]=arpCache[num].next;

	return num;
}


/*
 * PUBLIC
 * Search cache.
 * parameters : IP address
 * return : Mac address or failed=NULL
 */
static inline char *searchCache(uint ip)
{
	uchar tbl;
	int i;


	tbl=(ip>>24)%ARP_CACHE_HASH;
	for(i=hashTbl[tbl];i!=-1;i=arpCache[i].next)
		if(arpCache[i].ip==ip)return arpCache[i].mac;

	return broadcastMac;
}


/*
 * PUBLIC
 * Add cache.
 */
static void addCache(uint ip,char *mac)
{
	char *prev;
	char num;


	enter_spinlock(&cacheLock);
	{
		if((prev=searchPrevIndex(ip))==NULL)
			memcpy(searchCache(ip),mac,6);
		else
		{
			if(cacheEmpty==-1)num=snatchCache();
			else
			{
				num=cacheEmpty;
				cacheEmpty=arpCache[(int)num].next;
			}

			arpCache[(int)num].ip=ip;
			memcpy(arpCache[(int)num].mac,mac,6);
			arpCache[(int)num].next=-1;
			*prev=num;
		}
	}
	exit_spinlock(&cacheLock);
}


/*
 * PUBLIC
 * Init arp cache.
 */
static void initCache()
{
	int i;


	/* å󥯤Ϣ롣 */
	for(i=0;i<ARP_CACHE_NUM;++i)arpCache[i].next=i+1;
	arpCache[ARP_CACHE_NUM-1].next=-1;

	/* ϥåơ֥ν */
	memset(hashTbl,0xff,ARP_CACHE_HASH);
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


/*
 * PUBLIC
 * Transfer ARP frame.
 * parameters : device number,destination IP,destination mac,operation code
 */
static void transArp(int num,uint dstip,char *dstmac,ushort opecode)
{
	ARP_HEADER head;
	TRANS_BUF_INFO tbi;


	head.hardType=swapWord(HARD_TYPE_ETHER);
	head.protType=swapWord(PROT_TYPE_IP);
	head.hardAddrLen=6;
	head.protAddrLen=4;
	head.opeCode=swapWord(opecode);
	memcpy(head.srcMac,ethDev[num].mac,6);
	head.srcIp=ethDev[num].ip;
	memcpy(head.dstMac,dstmac,6);
	head.dstIp=dstip;

	tbi.data[0]=&head;
	tbi.size[0]=sizeof(ARP_HEADER);
	tbi.size[1]=0;
	tbi.size[2]=0;
	tbi.type=DIX_TYPE_ARP;

	ethDev[num].dev->write(ethDev[num].dev->linf,(size_t)&tbi,(size_t)dstmac);
}


typedef struct{
	uint ip;
	WAIT_INTR wait;
}REPLY_WAIT;


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


/*
 * GLOBAL
 * Receive ARP frame.
 * parameters : device number,ARP_HEADER address
 */
void receiveArp(int num,ARP_HEADER *head)
{
/***************************************************************************************************************/
uchar *srcip,*dstip;

srcip=(uchar*)&head->srcIp;
dstip=(uchar*)&head->dstIp;
printk("ARP_HEADER htype=%x,ptype=%x,hsize=%x,psize=%x,opecode=%x\n"
		"source mac=%d-%d-%d-%d-%d-%d,IP=%d-%d-%d-%d\n"
		"dest   mac=%d-%d-%d-%d-%d-%d,IP=%d-%d-%d-%d\n",
       swapWord(head->hardType),swapWord(head->protType),head->hardAddrLen,head->protAddrLen,
       swapWord(head->opeCode),
       head->srcMac[0],head->srcMac[1],head->srcMac[2],head->srcMac[3],head->srcMac[4],head->srcMac[5],
       srcip[0],srcip[1],srcip[2],srcip[3],
       head->dstMac[0],head->dstMac[1],head->dstMac[2],head->dstMac[3],head->dstMac[4],head->dstMac[5],
       dstip[0],dstip[1],dstip[2],dstip[3]);
/***************************************************************************************************************/
	/* IPγǧ */
	if(head->dstIp!=IP_ADDRESS0)return;

	/* ڥɤγǧ */
	switch(swapWord(head->opeCode))
	{
		case OPE_CODE_ARP_REQ:
			transArp(num,head->srcIp,head->srcMac,OPE_CODE_ARP_REP);
			addCache(head->dstIp,head->dstMac);
			break;
		case OPE_CODE_ARP_REP:
			break;
		case OPE_CODE_RARP_REQ:
		case OPE_CODE_RARP_REP:
			break;
	}
}


/**************************************************************************
 *
 * ꥯ
 *
 **************************************************************************/


/*
 * GLOBAL
 * Get destination MAC.
 * parameters : device number,ip address
 * return : MAC address or NULL
 */
char *getMac(int num,uint ip)
{
	enum{
		RETRY_COUNT=3,		/* ȥ饤 */
	};

	char *mac;


	/* å򸡺 */
	if((mac=searchCache(ip))!=broadcastMac)return mac;

	/* ꥯARP */
	transArp(num,ip,broadcastMac,OPE_CODE_ARP_REQ);

	return NULL;
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


/*
 * GLOBAL
 * Init arp protocol.
 */
void initArp()
{
	initCache();
}
